/*
 * This file is part of the Meeds project (https://meeds.io/).
 *
 * Copyright (C) 2020 - 2025 Meeds Association contact@meeds.io
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
package io.meeds.billig.service;

import com.stripe.model.Subscription;
import io.meeds.billing.model.HubBilling;
import io.meeds.billing.model.HubBillingSettings;
import io.meeds.billing.service.HubBillingPlanService;
import io.meeds.billing.service.HubSettingService;
import io.meeds.billing.service.BillingService;
import io.meeds.billing.storage.HubBillingSettingsStorage;
import org.exoplatform.commons.ObjectAlreadyExistsException;
import org.exoplatform.commons.exception.ObjectNotFoundException;
import org.exoplatform.social.core.identity.model.Identity;
import org.exoplatform.social.core.space.model.Space;
import org.exoplatform.social.core.space.spi.SpaceService;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.mockito.Mockito;
import org.mockito.junit.jupiter.MockitoExtension;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.test.context.bean.override.mockito.MockitoBean;

import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertThrows;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.ArgumentMatchers.anyString;
import static org.mockito.ArgumentMatchers.argThat;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

@SpringBootTest(classes = { HubSettingService.class, })
@ExtendWith(MockitoExtension.class)
public class HubSettingServiceTest {

  @MockitoBean
  private HubBilling                hubBilling;

  @MockitoBean
  private HubBillingSettingsStorage hubBillingSettingsStorage;

  @MockitoBean
  private BillingService            billingService;

  @MockitoBean
  private HubBillingPlanService     hubBillingPlanService;

  @MockitoBean
  private SpaceService              spaceService;

  @Autowired
  private HubSettingService         hubSettingService;

  @Test
  public void testCreateSettingsHubBillingSettings() throws ObjectAlreadyExistsException {
    Space space = Mockito.mock(Space.class);
    Identity identity = Mockito.mock(Identity.class);
    String customerId = "customer_1";
    String subscriptionId = "subscription_1";
    String subscriptionStatus = "subscription_status_1";
    when(identity.getId()).thenReturn("1");
    when(space.getSpaceId()).thenReturn(1L);
    Subscription subscription = Mockito.mock(Subscription.class);
    when(subscription.getId()).thenReturn(subscriptionId);
    when(subscription.getStatus()).thenReturn(subscriptionStatus);
    when(subscription.getCustomer()).thenReturn(customerId);
    hubSettingService.createSettings(space,subscription, identity);
    verify(hubBillingSettingsStorage,
           times(1)).save(argThat(hubBillingSettings -> customerId.equals(hubBillingSettings.getCustomerId()) && subscriptionId.equals(hubBillingSettings.getSubscriptionId()) && subscriptionStatus.equals(hubBillingSettings.getSubscriptionStatus())));
  }

  @Test
  public void shouldThrowObjectNotFoundExceptionIfSpaceNotFound() {
    long spaceId = 1L;
    when(spaceService.getSpaceById(spaceId)).thenReturn(null);
    assertThrows(ObjectNotFoundException.class, () ->
            hubSettingService.getSettingsBySpaceId(spaceId, "user"));
  }

  @Test
  public void shouldThrowIllegalAccessExceptionIfUserNotAuthorized() {
    long spaceId = 1L;
    Space space = mock(Space.class);
    when(spaceService.getSpaceById(spaceId)).thenReturn(space);
    when(spaceService.isManager(any(Space.class), anyString())).thenReturn(false);
    assertThrows(IllegalAccessException.class, () ->
            hubSettingService.getSettingsBySpaceId(spaceId, "user"));
  }

  @Test
  public void shouldReturnHubBillingSettings() throws ObjectNotFoundException, IllegalAccessException {
    long spaceId = 1L;
    String currentUser = "user";
    Space space = mock(Space.class);
    when(spaceService.getSpaceById(spaceId)).thenReturn(space);
    when(spaceService.canManageSpace(space, currentUser)).thenReturn(true);
    when(hubBillingSettingsStorage.getHubBillingSettingsBySpaceId(spaceId)).thenReturn(new HubBillingSettings());
    Assertions.assertNotNull(hubSettingService.getSettingsBySpaceId(spaceId, currentUser));
  }

}
