/*
 * This file is part of the Meeds project (https://meeds.io/).
 * Copyright (C) 2020 - 2025 Meeds Association contact@meeds.io
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
package io.meeds.billig.service;

import io.meeds.billing.service.SubscriptionEmailReminderService;
import io.meeds.portal.permlink.model.PermanentLinkObject;
import io.meeds.portal.permlink.service.PermanentLinkService;
import org.exoplatform.commons.utils.CommonsUtils;
import org.exoplatform.portal.Constants;
import org.exoplatform.portal.branding.BrandingService;
import org.exoplatform.services.mail.MailService;
import org.exoplatform.services.organization.OrganizationService;
import org.exoplatform.services.organization.User;
import org.exoplatform.services.organization.UserHandler;
import org.exoplatform.services.organization.UserProfile;
import org.exoplatform.services.organization.UserProfileHandler;
import org.exoplatform.services.resources.ResourceBundleService;
import org.exoplatform.social.notification.LinkProviderUtils;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.mockito.MockedStatic;
import org.mockito.junit.jupiter.MockitoExtension;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.test.context.bean.override.mockito.MockitoBean;
import javax.mail.Session;

import javax.mail.internet.MimeMessage;

import java.util.Collections;
import java.util.List;
import java.util.Properties;

import static org.mockito.ArgumentMatchers.any;
import static org.mockito.ArgumentMatchers.anyString;
import static org.mockito.Mockito.doNothing;
import static org.mockito.Mockito.mockStatic;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

@SpringBootTest(classes = { SubscriptionEmailReminderService.class, })
@ExtendWith(MockitoExtension.class)
class SubscriptionEmailReminderServiceTest {

  @Autowired
  private SubscriptionEmailReminderService plugin;

  @MockitoBean
  private MailService                     mailService;

  @MockitoBean
  private OrganizationService             organizationService;

  @MockitoBean
  private ResourceBundleService           resourceBundleService;

  @MockitoBean
  private BrandingService                 brandingService;

  @MockitoBean
  private PermanentLinkService            permanentLinkService;

  @MockitoBean
  private UserProfileHandler              userProfileHandler;

  @MockitoBean
  private User                            user;

  @MockitoBean
  private UserProfile                     userProfile;

  @MockitoBean
  private UserHandler                     userHandler;

  @BeforeEach
  @SuppressWarnings({ "rawtypes", "unchecked" })
  public void init() throws Exception {
    plugin.setEmailBodyPath("fake-path.html");
    plugin.setEmailBodyTemplate("$SENDER_FULL_NAME wants to remind you that your trial period is ending soon");

    when(organizationService.getUserHandler()).thenReturn(userHandler);
    when(organizationService.getUserProfileHandler()).thenReturn(userProfileHandler);

    when(userHandler.findUserByName("sender")).thenReturn(user);
    when(user.getEmail()).thenReturn("sender@example.com");
    when(user.getDisplayName()).thenReturn("sender sender");

    when(userProfileHandler.findUserProfileByName("sender")).thenReturn(userProfile);

    when(userHandler.findUserByName("recipient")).thenReturn(user);
    when(user.getEmail()).thenReturn("recipient@example.com");
    when(user.getDisplayName()).thenReturn("recipient recipient");

    when(userProfileHandler.findUserProfileByName("recipient")).thenReturn(userProfile);
    when(userProfile.getAttribute(Constants.USER_LANGUAGE)).thenReturn("en");

    when(resourceBundleService.getSharedString(anyString(), any())).thenReturn("TranslatedText");
    when(brandingService.getCompanyName()).thenReturn("MyCompany");
    when(brandingService.getThemeStyle()).thenReturn(Collections.singletonMap("primaryColor", "#123456"));

    when(permanentLinkService.getPermanentLink(any(PermanentLinkObject.class))).thenReturn("fak-url");

    Session mockSession = Session.getInstance(new Properties());
    when(mailService.getMailSession()).thenReturn(mockSession);

    doNothing().when(mailService).sendMessage(any(MimeMessage.class));
  }

  @Test
  void shouldSendEmailSuccessfully() throws Exception {
    try (MockedStatic<CommonsUtils> commonsUtilsMock = mockStatic(CommonsUtils.class);
        MockedStatic<LinkProviderUtils> linkProviderUtilsMock = mockStatic(LinkProviderUtils.class)) {

      commonsUtilsMock.when(CommonsUtils::getCurrentDomain).thenReturn("fake-domain");
      linkProviderUtilsMock.when(() -> LinkProviderUtils.getRedirectUrl(anyString(), anyString())).thenReturn("fake-url");

      linkProviderUtilsMock.when(LinkProviderUtils::getBaseUrl).thenReturn("fake-url");
      doNothing().when(mailService).sendMessage(any(MimeMessage.class));
      plugin.sendEmailNotification("sender", List.of("recipient"), 123L, "TRIAL_ENDING");

      verify(mailService, times(1)).sendMessage(any(MimeMessage.class));
    }
  }
}
