/*
 * This file is part of the Meeds project (https://meeds.io/).
 *
 * Copyright (C) 2020 - 2025 Meeds Association contact@meeds.io
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
package io.meeds.billig.service;

import com.google.api.client.util.Maps;
import com.stripe.exception.SignatureVerificationException;
import com.stripe.model.Event;
import com.stripe.model.EventDataObjectDeserializer;
import com.stripe.model.Invoice;
import com.stripe.model.Plan;
import com.stripe.model.Subscription;
import com.stripe.net.Webhook;
import io.meeds.billing.model.HubBilling;
import io.meeds.billing.model.HubBillingPlan;
import io.meeds.billing.model.HubBillingSettings;
import io.meeds.billing.service.BillingService;
import io.meeds.billing.service.HubBillingPlanService;
import io.meeds.billing.service.HubSettingService;
import io.meeds.billing.service.WebhookHandlingService;
import io.meeds.billing.utils.Utils;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.mockito.Mockito;
import org.mockito.junit.jupiter.MockitoExtension;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.test.context.bean.override.mockito.MockitoBean;

import java.util.HashMap;
import java.util.Map;
import java.util.Optional;

import static org.junit.Assert.assertThrows;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.ArgumentMatchers.anyMap;
import static org.mockito.ArgumentMatchers.anyString;
import static org.mockito.ArgumentMatchers.eq;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

@SpringBootTest(classes = { WebhookHandlingService.class, })
@ExtendWith(MockitoExtension.class)
public class WebhookHandlingServiceTest {

  @MockitoBean
  private HubBilling             hubBilling;

  @MockitoBean
  HubSettingService              hubSettingService;

  @MockitoBean
  private HubBillingPlanService  hubBillingPlanService;

  @MockitoBean
  private BillingService         billingService;

  @Autowired
  private WebhookHandlingService webhookHandlingService;

  private static final String    SIGNATURE       = "test-signature";

  private static final String    SECRET_KEY      = "whsec_test";

  private static final String    SUBSCRIPTION_ID = "sub_12345";

  @Test
  public void testHandleSubscriptionUpdatedEvent_success() throws Exception {
    Subscription subscription = new Subscription();
    subscription.setId(SUBSCRIPTION_ID);
    subscription.setStatus("active");
    Event event = Mockito.mock(Event.class);
    EventDataObjectDeserializer deserializer = Mockito.mock(EventDataObjectDeserializer.class);
    Event.Data eventData = mock(Event.Data.class);
    when(eventData.getPreviousAttributes()).thenReturn(new HashMap<>());
    when(event.getData()).thenReturn(eventData);
    when(event.getType()).thenReturn("customer.subscription.updated");
    when(event.getDataObjectDeserializer()).thenReturn(deserializer);
    when(deserializer.getObject()).thenReturn(Optional.of(subscription));
    when(hubBilling.getWebHookSecretKey()).thenReturn(SECRET_KEY);
    HubBillingSettings hubBillingSettings = mock(HubBillingSettings.class);
    HubBillingPlan hubBillingPlan = mock(HubBillingPlan.class);
    when(hubBillingSettings.getHubBillingPlan()).thenReturn(hubBillingPlan);
    when(hubBillingPlan.getPlanId()).thenReturn("planId");
    Plan plan = mock(Plan.class);
    when(plan.getId()).thenReturn("planId");
    when(hubSettingService.getSettingsBySubscriptionId(SUBSCRIPTION_ID)).thenReturn(hubBillingSettings);

    // Mock Stripe Webhook parser
    try (var mockedWebHook = Mockito.mockStatic(Webhook.class);
         var mockedUtils = Mockito.mockStatic(Utils.class)) {
      mockedWebHook.when(() -> Webhook.constructEvent(any(), eq(SIGNATURE), eq(SECRET_KEY))).thenReturn(event);
      mockedUtils.when(() -> Utils.getPlanFromSubscription(any(Subscription.class))).thenReturn(plan);

      webhookHandlingService.handleWebHookEvent("{}", SIGNATURE);

      // Assert
      verify(hubSettingService, times(1)).updateHuBillingSetting(any(HubBillingSettings.class), any(Subscription.class), anyMap());
    }
  }

  @Test
  void testHandleSubscriptionUpdatedEvent_fail() throws Exception {
    when(hubBilling.getWebHookSecretKey()).thenReturn(SECRET_KEY);

    try (var mocked = Mockito.mockStatic(Webhook.class)) {
      mocked.when(() -> Webhook.constructEvent(any(), eq(SIGNATURE), eq(SECRET_KEY)))
            .thenThrow(new SignatureVerificationException("Invalid signature", SIGNATURE));

      // Expect exception
      assertThrows(SignatureVerificationException.class, () -> webhookHandlingService.handleWebHookEvent("{}", SIGNATURE));

      // Ensure updateSubscriptionInfo was never called
      verify(hubSettingService, times(0)).updateHuBillingSetting(any(HubBillingSettings.class), any(Subscription.class), anyMap());
    }
  }

  @Test
  public void testHandleInvoiceUpdatedEvent_success() throws Exception {
    Invoice invoice = mock(Invoice.class);
    Invoice.Parent invoiceParent = mock(Invoice.Parent.class);
    Invoice.Parent.SubscriptionDetails subscriptionDetails = mock(Invoice.Parent.SubscriptionDetails.class);
    when(invoice.getParent()).thenReturn(invoiceParent);
    when(invoiceParent.getSubscriptionDetails()).thenReturn(subscriptionDetails);
    when(invoice.getAmountPaid()).thenReturn(2L);
    when(subscriptionDetails.getSubscription()).thenReturn(SUBSCRIPTION_ID);
    Event event = Mockito.mock(Event.class);
    EventDataObjectDeserializer deserializer = Mockito.mock(EventDataObjectDeserializer.class);
    when(event.getType()).thenReturn("invoice.paid");
    when(event.getDataObjectDeserializer()).thenReturn(deserializer);
    when(deserializer.getObject()).thenReturn(Optional.of(invoice));
    when(hubBilling.getWebHookSecretKey()).thenReturn(SECRET_KEY);
    HubBillingSettings hubBillingSettings = mock(HubBillingSettings.class);
    when(hubBillingSettings.getSubscriptionStatus()).thenReturn(Utils.SubscriptionStatus.TRIALING.name());
    when(hubSettingService.getSettingsBySubscriptionId(anyString())).thenReturn(hubBillingSettings);
    Subscription subscription = mock(Subscription.class);
    when(billingService.getSubscription(anyString())).thenReturn(subscription);

    // Mock Stripe Webhook parser
    try (var mocked = Mockito.mockStatic(Webhook.class)) {
      mocked.when(() -> Webhook.constructEvent(any(), eq(SIGNATURE), eq(SECRET_KEY))).thenReturn(event);

      webhookHandlingService.handleWebHookEvent("{}", SIGNATURE);

      // Assert
      verify(hubSettingService, times(1)).updateSettings(any(HubBillingSettings.class));
    }
  }


  @Test
  public void testHandlePlanUpdatedEvent_success() throws Exception {
    Plan plan = mock(Plan.class);
    when(plan.getId()).thenReturn("plan_id");
    when(plan.getMeter()).thenReturn(null);
    when(plan.getProduct()).thenReturn("product_id");
    Map<String, String> metadata = Maps.newHashMap();
    metadata.put("maxOfUsers", "10");
    when(plan.getMetadata()).thenReturn(metadata);
    Event event = Mockito.mock(Event.class);
    EventDataObjectDeserializer deserializer = Mockito.mock(EventDataObjectDeserializer.class);
    when(event.getType()).thenReturn("plan.updated");
    when(event.getDataObjectDeserializer()).thenReturn(deserializer);
    when(deserializer.getObject()).thenReturn(Optional.of(plan));
    when(hubBilling.getWebHookSecretKey()).thenReturn(SECRET_KEY);
    HubBillingPlan HubBillingPlan = mock(HubBillingPlan.class);
    when(hubBillingPlanService.getHubBillingPlanByPlanId("plan_id")).thenReturn(HubBillingPlan);

    // Mock Stripe Webhook parser
    try (var mocked = Mockito.mockStatic(Webhook.class)) {
      mocked.when(() -> Webhook.constructEvent(any(), eq(SIGNATURE), eq(SECRET_KEY))).thenReturn(event);

      webhookHandlingService.handleWebHookEvent("{}", SIGNATURE);

      // Assert
      verify(hubBillingPlanService, times(1)).saveHubBillingPlan(any(HubBillingPlan.class));
    }
  }
}
