/*
 * This file is part of the Meeds project (https://meeds.io/).
 * Copyright (C) 2020 - 2025 Meeds Association contact@meeds.io
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
package io.meeds.billing.service;

import com.stripe.model.Plan;
import com.stripe.model.Subscription;
import io.meeds.billing.model.HubBillingPlan;
import io.meeds.billing.storage.HubBillingPlanStorage;
import io.meeds.billing.utils.Utils;
import org.apache.commons.collections.MapUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.Optional;

@Service
public class HubBillingPlanService {

  @Autowired
  private HubBillingPlanStorage hubBillingPlanStorage;

  public HubBillingPlan getOrCreateHubBillingPlan(Subscription subscription) {
    Plan plan = Utils.getPlanFromSubscription(subscription);
    HubBillingPlan hubBillingPlan = hubBillingPlanStorage.findByPlanId(plan.getId());
    if (hubBillingPlan != null) {
      return hubBillingPlan;
    }
    hubBillingPlan = new HubBillingPlan();
    hubBillingPlan.setPlanId(plan.getId());
    hubBillingPlan.setMeterId(plan.getMeter());
    hubBillingPlan.setProductId(plan.getProduct());
    long maxOfUsers = Optional.ofNullable(plan.getMetadata())
                              .filter(MapUtils::isNotEmpty)
                              .map(metadata -> metadata.get("maxOfUsers"))
                              .map(Long::parseLong)
                              .orElse(0L);
    hubBillingPlan.setMaxUsers(maxOfUsers);
    hubBillingPlan = hubBillingPlanStorage.saveHubBillingPlan(hubBillingPlan);
    return hubBillingPlan;
  }

  public HubBillingPlan getHubBillingPlanByPlanId(String planId) {
    return hubBillingPlanStorage.findByPlanId(planId);
  }

  public HubBillingPlan saveHubBillingPlan(HubBillingPlan hubBillingPlan) {
    return hubBillingPlanStorage.saveHubBillingPlan(hubBillingPlan);
  }
}
