/*
 * Decompiled with CFR 0.152.
 */
package io.meeds.deeds.common.service;

import io.meeds.deeds.common.constant.BlockchainLeaseStatus;
import io.meeds.deeds.common.constant.TransactionStatus;
import io.meeds.deeds.common.constant.UnauthorizedOperationException;
import io.meeds.deeds.common.elasticsearch.model.DeedTenant;
import io.meeds.deeds.common.elasticsearch.model.DeedTenantLease;
import io.meeds.deeds.common.elasticsearch.storage.LeaseRepository;
import io.meeds.deeds.common.model.DeedLeaseBlockchainState;
import io.meeds.deeds.common.model.DeedTenantLeaseDTO;
import io.meeds.deeds.common.model.DeedTenantOfferDTO;
import io.meeds.deeds.common.model.LeaseFilter;
import io.meeds.deeds.common.service.BlockchainService;
import io.meeds.deeds.common.service.ListenerService;
import io.meeds.deeds.common.service.OfferService;
import io.meeds.deeds.common.service.TenantService;
import io.meeds.deeds.common.utils.DeedTenantLeaseMapper;
import io.meeds.wom.api.constant.ObjectNotFoundException;
import java.math.BigInteger;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.stream.Stream;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.elasticsearch.core.ElasticsearchOperations;
import org.springframework.data.elasticsearch.core.SearchHit;
import org.springframework.data.elasticsearch.core.SearchHitSupport;
import org.springframework.data.elasticsearch.core.SearchHits;
import org.springframework.data.elasticsearch.core.SearchPage;
import org.springframework.data.elasticsearch.core.query.Criteria;
import org.springframework.data.elasticsearch.core.query.CriteriaQuery;
import org.springframework.data.elasticsearch.core.query.Query;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

@Component
public class LeaseService {
    private static final Logger LOG = LoggerFactory.getLogger(LeaseService.class);
    public static final String LEASE_ACQUIRED_EVENT = "deed.event.leaseAcquired";
    public static final String LEASE_ACQUISITION_CONFIRMED_EVENT = "deed.event.leaseAcquisionConfirmed";
    public static final String LEASE_RENT_PAYED_EVENT = "deed.event.leaseRentPayed";
    public static final String LEASE_RENT_PAYMENT_CONFIRMED_EVENT = "deed.event.leaseRentPaymentConfirmed";
    public static final String LEASE_END_EVENT = "deed.event.leaseEndSent";
    public static final String LEASE_ENDED_CONFIRMED_EVENT = "deed.event.leaseEndedConfirmed";
    public static final String LEASE_TENANT_EVICT_EVENT = "deed.event.leaseTenantEvict";
    public static final String LEASE_TENANT_EVICTED_CONFIRMED_EVENT = "deed.event.leaseTenantEvictedConfirmed";
    private static final String TRANSACTION_HASH_IS_MANDATORY_MESSAGE = "Transaction Hash is Mandatory";
    @Autowired
    private LeaseRepository leaseRepository;
    @Autowired
    private ElasticsearchOperations elasticsearchOperations;
    @Autowired
    private OfferService offerService;
    @Autowired
    private TenantService tenantService;
    @Autowired
    private BlockchainService blockchainService;
    @Autowired
    private ListenerService listenerService;

    public Page<DeedTenantLeaseDTO> getLeases(LeaseFilter leaseFilter, Pageable pageable) {
        Criteria criteria = new Criteria("enabled").is((Object)true);
        if (!leaseFilter.isIncludeOutdated()) {
            Criteria endDateCriteria = new Criteria("endDate").greaterThan((Object)Instant.now());
            criteria.and(endDateCriteria);
        }
        if (leaseFilter.isExcludeNotConfirmed()) {
            Criteria confirmedCriteria = new Criteria("confirmed").is((Object)true);
            criteria.and(confirmedCriteria);
        }
        if (leaseFilter.getNftId() >= 0L) {
            Criteria nftIdCriteria = new Criteria("nftId").is((Object)leaseFilter.getNftId());
            criteria.and(nftIdCriteria);
        }
        if (!CollectionUtils.isEmpty(leaseFilter.getCardTypes())) {
            Criteria cardTypeCriteria = new Criteria("cardType").in(leaseFilter.getCardTypes());
            criteria.and(cardTypeCriteria);
        }
        if (!CollectionUtils.isEmpty(leaseFilter.getTransactionStatus())) {
            Criteria transactionStatusCriteria = new Criteria("transactionStatus").in(leaseFilter.getTransactionStatus());
            criteria.and(transactionStatusCriteria);
        }
        if (StringUtils.isNotBlank((CharSequence)leaseFilter.getCurrentAddress())) {
            visibilityCriteria = new Criteria("viewAddresses").in(new Object[]{StringUtils.lowerCase((String)leaseFilter.getCurrentAddress()), "ALL"});
            criteria.and(visibilityCriteria);
        } else {
            visibilityCriteria = new Criteria("viewAddresses").in(new Object[]{"ALL"});
            criteria.and(visibilityCriteria);
        }
        if (leaseFilter.getOwner() != null) {
            if (leaseFilter.getOwner().booleanValue()) {
                Criteria ownerCriteria = new Criteria("owner").in(new Object[]{StringUtils.lowerCase((String)leaseFilter.getCurrentAddress())});
                criteria.and(ownerCriteria);
            } else {
                Criteria managerCriteria = new Criteria("manager").in(new Object[]{StringUtils.lowerCase((String)leaseFilter.getCurrentAddress())});
                criteria.and(managerCriteria);
            }
        }
        CriteriaQuery query = new CriteriaQuery(criteria, pageable);
        SearchHits result = this.elasticsearchOperations.search((Query)query, DeedTenantLease.class);
        SearchPage searchPage = SearchHitSupport.searchPageFor((SearchHits)result, (Pageable)pageable);
        return searchPage.map(SearchHit::getContent).map(this::buildLeaseDTO);
    }

    public DeedTenantLeaseDTO getLease(long leaseId, String walletAddress, boolean refreshFromBlockchain) throws Exception {
        DeedTenantLeaseDTO lease;
        if (refreshFromBlockchain) {
            LOG.debug("Refreshing Changed lease with id {} on blockchain on request of user {}", (Object)leaseId, (Object)walletAddress);
            long lastBlockNumber = this.blockchainService.getLastBlock();
            DeedLeaseBlockchainState blockchainLease = this.blockchainService.getLeaseById(BigInteger.valueOf(leaseId), BigInteger.valueOf(lastBlockNumber), null);
            this.updateLeaseStatusFromBlockchain(blockchainLease, null);
        }
        if ((lease = this.getLease(leaseId)) == null) {
            throw new ObjectNotFoundException();
        }
        return lease;
    }

    public DeedTenantLeaseDTO getLease(long leaseId) {
        DeedTenantLease lease = this.leaseRepository.findById(leaseId).orElse(null);
        return this.buildLeaseDTO(lease);
    }

    public DeedTenantLeaseDTO createLease(String managerAddress, String managerEmail, String offerId, String transactionHash) throws ObjectNotFoundException, UnauthorizedOperationException {
        DeedTenantOfferDTO deedTenantOffer = this.offerService.getOffer(offerId);
        if (deedTenantOffer == null) {
            throw new ObjectNotFoundException("Offer with id " + offerId + "wasn't found");
        }
        if (StringUtils.isBlank((CharSequence)managerAddress)) {
            throw new IllegalArgumentException("Hub Manager Address is mandatory");
        }
        if (StringUtils.isBlank((CharSequence)transactionHash)) {
            throw new IllegalArgumentException(TRANSACTION_HASH_IS_MANDATORY_MESSAGE);
        }
        String ownerAddress = deedTenantOffer.getOwner();
        if (!this.tenantService.isDeedOwner(ownerAddress, deedTenantOffer.getNftId())) {
            throw new UnauthorizedOperationException("Address " + ownerAddress + " isn't owner of the Deed anymore");
        }
        DeedTenantLease deedTenantLease = this.createLeaseFromOffer(deedTenantOffer, ownerAddress, managerAddress, managerEmail, transactionHash);
        this.listenerService.publishEvent(LEASE_ACQUIRED_EVENT, deedTenantLease);
        return this.buildLeaseDTO(deedTenantLease);
    }

    public DeedTenantLeaseDTO payRents(String managerAddress, String ownerAddress, long leaseId, int paidMonths, String transactionHash) throws ObjectNotFoundException, UnauthorizedOperationException {
        DeedTenantLease deedTenantLease = this.leaseRepository.findById(leaseId).orElse(null);
        if (deedTenantLease == null) {
            throw new ObjectNotFoundException("Lease with identifier " + leaseId + "doesn't exist found");
        }
        if (!StringUtils.equalsIgnoreCase((CharSequence)managerAddress, (CharSequence)deedTenantLease.getManager())) {
            throw new UnauthorizedOperationException("Lease with id " + leaseId + " does't belong to " + managerAddress);
        }
        if (StringUtils.isBlank((CharSequence)transactionHash)) {
            throw new IllegalArgumentException(TRANSACTION_HASH_IS_MANDATORY_MESSAGE);
        }
        this.addPendingTransactionHash(deedTenantLease, transactionHash);
        deedTenantLease.setTransactionStatus(TransactionStatus.IN_PROGRESS);
        deedTenantLease.setMonthPaymentInProgress(paidMonths);
        if (!StringUtils.equalsIgnoreCase((CharSequence)deedTenantLease.getOwner(), (CharSequence)ownerAddress) && this.tenantService.isDeedOwner(ownerAddress, deedTenantLease.getNftId())) {
            deedTenantLease.setOwner(ownerAddress);
        }
        deedTenantLease = this.saveLease(deedTenantLease);
        this.listenerService.publishEvent(LEASE_RENT_PAYED_EVENT, deedTenantLease);
        return this.buildLeaseDTO(deedTenantLease);
    }

    public DeedTenantLeaseDTO endLease(String managerOrOwnerAddress, long leaseId, String transactionHash) throws ObjectNotFoundException, UnauthorizedOperationException {
        DeedTenantLease deedTenantLease = this.leaseRepository.findById(leaseId).orElse(null);
        if (deedTenantLease == null) {
            throw new ObjectNotFoundException("Provided lease with id " + leaseId + "doesn't exists");
        }
        boolean isManager = this.tenantService.isDeedManager(managerOrOwnerAddress, deedTenantLease.getNftId());
        boolean isOwner = this.tenantService.isDeedOwner(managerOrOwnerAddress, deedTenantLease.getNftId());
        if (!isManager && !isOwner) {
            throw new UnauthorizedOperationException("Lease with id " + leaseId + " does't belong to " + managerOrOwnerAddress);
        }
        if (StringUtils.isBlank((CharSequence)transactionHash)) {
            throw new IllegalArgumentException(TRANSACTION_HASH_IS_MANDATORY_MESSAGE);
        }
        this.addPendingTransactionHash(deedTenantLease, transactionHash);
        deedTenantLease.setTransactionStatus(TransactionStatus.IN_PROGRESS);
        deedTenantLease.setEndingLeaseAddress(StringUtils.lowerCase((String)managerOrOwnerAddress));
        deedTenantLease.setEndingLease(true);
        if (isOwner) {
            deedTenantLease.setOwner(StringUtils.lowerCase((String)managerOrOwnerAddress));
            deedTenantLease = this.saveLease(deedTenantLease);
            this.listenerService.publishEvent(LEASE_TENANT_EVICT_EVENT, deedTenantLease);
            return this.buildLeaseDTO(deedTenantLease);
        }
        deedTenantLease = this.saveLease(deedTenantLease);
        this.listenerService.publishEvent(LEASE_END_EVENT, deedTenantLease);
        return this.buildLeaseDTO(deedTenantLease);
    }

    public void updateLeaseStatusFromBlockchain(DeedLeaseBlockchainState blockchainLease, BlockchainLeaseStatus status) throws Exception {
        long leaseId = blockchainLease.getId().longValue();
        DeedTenantLease deedTenantLease = this.leaseRepository.findById(leaseId).orElse(null);
        if (deedTenantLease == null) {
            DeedTenantOfferDTO offer = this.offerService.getOfferByBlockchainId(leaseId);
            deedTenantLease = this.createLeaseFromOffer(offer, offer.getOwner(), blockchainLease.getTenant(), null, null);
        }
        this.updateLeaseStatusFromBlockchain(deedTenantLease, blockchainLease, status, null);
    }

    public void updateLeaseStatusFromBlockchain(long leaseId, String transactionHash, Map<BlockchainLeaseStatus, DeedLeaseBlockchainState> minedEvents) throws Exception {
        DeedTenantLease lease = this.leaseRepository.findById(leaseId).orElse(null);
        if (minedEvents.isEmpty()) {
            if (lease == null) {
                throw new ObjectNotFoundException("Wrong Lease id: " + leaseId);
            }
            this.saveLeaseTransactionAsError(lease, transactionHash);
        } else {
            if (minedEvents.size() > 1) {
                LOG.warn("Mined events for a single transaction seems to hold more than one Offer event. This is not supported yet. Will use the first retrieved event");
            }
            Map.Entry<BlockchainLeaseStatus, DeedLeaseBlockchainState> entry = minedEvents.entrySet().iterator().next();
            BlockchainLeaseStatus status = entry.getKey();
            DeedLeaseBlockchainState blockchainLease = entry.getValue();
            if (lease == null) {
                this.updateLeaseStatusFromBlockchain(blockchainLease, status);
            } else {
                this.checkLeaseBlockchainStatus(lease, blockchainLease);
                this.updateLeaseStatusFromBlockchain(lease, blockchainLease, status, transactionHash);
            }
        }
    }

    public List<DeedTenantLease> getPendingTransactions() {
        return this.leaseRepository.findByTransactionStatusInOrderByCreatedDateAsc(Arrays.asList(TransactionStatus.IN_PROGRESS)).stream().filter(lease -> !CollectionUtils.isEmpty(lease.getPendingTransactions())).toList();
    }

    public void transferDeedOwnership(String newOnwer, long nftId) throws UnauthorizedOperationException {
        if (!this.tenantService.isDeedOwner(newOnwer, nftId)) {
            throw new UnauthorizedOperationException("Address " + newOnwer + " isn't owner of the Deed anymore");
        }
        List<DeedTenantLease> leases = this.leaseRepository.findByEnabledTrueAndNftIdAndEndDateGreaterThan(nftId, Instant.now());
        if (!CollectionUtils.isEmpty(leases)) {
            leases.forEach(lease -> {
                lease.setOwner(newOnwer);
                this.saveLease((DeedTenantLease)lease);
            });
        }
    }

    public void saveLeaseTransactionAsError(long leaseId, String transactionHash) {
        DeedTenantLease lease = this.leaseRepository.findById(leaseId).orElse(null);
        if (lease == null) {
            throw new IllegalArgumentException("Wrong Lease identifier");
        }
        this.saveLeaseTransactionAsError(lease, transactionHash);
    }

    private DeedTenantLease createLeaseFromOffer(DeedTenantOfferDTO offer, String ownerAddress, String managerAddress, String managerEmail, String transactionHash) throws ObjectNotFoundException {
        DeedTenantLease deedTenantLease;
        DeedTenant deedTenant;
        if (StringUtils.isBlank((CharSequence)ownerAddress)) {
            ownerAddress = offer.getOwner();
        }
        if ((deedTenant = this.getDeedTenant(offer.getNftId(), managerAddress, ownerAddress)) == null) {
            throw new ObjectNotFoundException("Deed Tenant with id " + offer.getNftId() + " doesn't exists");
        }
        if (StringUtils.isBlank((CharSequence)deedTenant.getOwnerAddress())) {
            deedTenant.setOwnerAddress(ownerAddress);
            deedTenant = this.tenantService.saveDeedTenant(deedTenant);
        }
        if ((deedTenantLease = this.leaseRepository.findById(offer.getOfferId()).orElse(DeedTenantLeaseMapper.fromOffer(offer, deedTenant, managerAddress, managerEmail, StringUtils.lowerCase((String)transactionHash)))) == null) {
            throw new ObjectNotFoundException();
        }
        if (StringUtils.isBlank((CharSequence)deedTenantLease.getManager()) && StringUtils.isNotBlank((CharSequence)managerAddress)) {
            deedTenantLease.setManager(managerAddress);
        }
        if (StringUtils.isBlank((CharSequence)deedTenantLease.getManagerEmail()) && StringUtils.isNotBlank((CharSequence)managerEmail)) {
            deedTenantLease.setManagerEmail(managerEmail);
        }
        if (StringUtils.isBlank((CharSequence)deedTenantLease.getOwner()) && StringUtils.isNotBlank((CharSequence)ownerAddress)) {
            deedTenantLease.setOwner(ownerAddress);
        }
        return this.saveLease(deedTenantLease);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void updateLeaseStatusFromBlockchain(DeedTenantLease lease, DeedLeaseBlockchainState blockchainLease, BlockchainLeaseStatus status, String transactionHash) throws ObjectNotFoundException {
        DeedTenantLease freshLeaseEntity;
        long leaseId = blockchainLease.getId().longValue();
        if (leaseId == 0L) {
            throw new ObjectNotFoundException("Retrieved Lease from blockchain has a 0 identifier");
        }
        transactionHash = StringUtils.isEmpty((CharSequence)transactionHash) ? StringUtils.lowerCase((String)blockchainLease.getTransactionHash()) : StringUtils.lowerCase((String)transactionHash);
        this.removePendingTransactionHash(lease, transactionHash);
        long leaseEndDateInSeconds = blockchainLease.getLeaseEndDate().longValue();
        Instant leaseEndDate = Instant.ofEpochSecond(leaseEndDateInSeconds);
        if (lease.isEndingLease() && lease.getEndDate() != null && leaseEndDateInSeconds > 0L && leaseEndDate.isBefore(lease.getEndDate())) {
            lease.setEndingLease(false);
        }
        if (lease.getMonthPaymentInProgress() > 0) {
            int newlyPaidMonths = blockchainLease.getPaidMonths().intValue() - lease.getPaidMonths();
            int monthPaymentInProgress = lease.getMonthPaymentInProgress() - newlyPaidMonths;
            lease.setMonthPaymentInProgress(monthPaymentInProgress > 0 ? monthPaymentInProgress : 0);
        }
        if (CollectionUtils.isEmpty(lease.getPendingTransactions())) {
            lease.setEndingLease(false);
            lease.setMonthPaymentInProgress(0);
        }
        lease.setConfirmed(true);
        lease.setId(leaseId);
        lease.setNftId(blockchainLease.getDeedId().longValue());
        lease.setManager(StringUtils.lowerCase((String)blockchainLease.getTenant()));
        lease.setPaidMonths(blockchainLease.getPaidMonths().intValue());
        lease.setPaidRentsDate(Instant.ofEpochSecond(blockchainLease.getPaidRentsDate().longValue()));
        lease.setStartDate(Instant.ofEpochSecond(blockchainLease.getLeaseStartDate().longValue()));
        lease.setEndDate(leaseEndDate);
        lease.setNoticeDate(Instant.ofEpochSecond(blockchainLease.getNoticePeriodDate().longValue()));
        long blockNumber = blockchainLease.getBlockNumber().longValue();
        if (lease.getId() > 0L && blockNumber > 0L && (freshLeaseEntity = (DeedTenantLease)this.leaseRepository.findById(lease.getId()).orElse(null)) != null && freshLeaseEntity.getLastCheckedBlock() >= blockNumber) {
            LOG.debug("Lease Status is already updated in block {} (last checked {}) with status {}. Avoid publish an Event about change made.", new Object[]{blockNumber, freshLeaseEntity.getLastCheckedBlock(), status});
            this.saveLease(lease);
            return;
        }
        try {
            lease.setLastCheckedBlock(blockNumber);
            lease = this.saveLease(lease);
        }
        finally {
            this.broadcastLeaseEvent(lease, status);
        }
    }

    private void broadcastLeaseEvent(DeedTenantLease lease, BlockchainLeaseStatus status) {
        if (status == BlockchainLeaseStatus.LEASE_ACQUIRED) {
            LOG.debug("Lease {} Acquisition confirmed on blockchain for manager {} and owner {}", new Object[]{lease.getId(), lease.getManager(), lease.getOwner()});
            this.listenerService.publishEvent(LEASE_ACQUISITION_CONFIRMED_EVENT, this.buildLeaseDTO(lease));
        } else if (status == BlockchainLeaseStatus.LEASE_PAYED) {
            LOG.debug("Lease {} payment confirmed on blockchain for manager {} for owner {}", new Object[]{lease.getId(), lease.getManager(), lease.getOwner()});
            this.listenerService.publishEvent(LEASE_RENT_PAYMENT_CONFIRMED_EVENT, this.buildLeaseDTO(lease));
        } else if (status == BlockchainLeaseStatus.LEASE_ENDED) {
            LOG.debug("Lease {} end confirmed on blockchain for manager {}", (Object)lease.getId(), (Object)lease.getManager());
            this.listenerService.publishEvent(LEASE_ENDED_CONFIRMED_EVENT, this.buildLeaseDTO(lease));
        } else if (status == BlockchainLeaseStatus.LEASE_MANAGER_EVICTED) {
            LOG.debug("Lease {} evit tenant {} confirmed on blockchain with owner {}", new Object[]{lease.getId(), lease.getManager(), lease.getOwner()});
            this.listenerService.publishEvent(LEASE_TENANT_EVICTED_CONFIRMED_EVENT, this.buildLeaseDTO(lease));
        }
    }

    private void saveLeaseTransactionAsError(DeedTenantLease lease, String transactionHash) {
        if (StringUtils.isNotBlank((CharSequence)transactionHash)) {
            if (!lease.isConfirmed() && !CollectionUtils.isEmpty(lease.getPendingTransactions()) && lease.getPendingTransactions().contains(transactionHash.toLowerCase())) {
                lease.setEnabled(false);
            }
            this.removePendingTransactionHash(lease, transactionHash);
            if (lease.getPendingTransactions().isEmpty()) {
                lease.setEndingLease(false);
                lease.setMonthPaymentInProgress(0);
            }
            this.saveLease(lease);
        }
    }

    private DeedTenantLease saveLease(DeedTenantLease deedTenantLease) {
        if (!deedTenantLease.isEnabled()) {
            deedTenantLease.setViewAddresses(Collections.emptyList());
        } else if (!deedTenantLease.isConfirmed()) {
            List<String> viewAddresses = Arrays.asList(StringUtils.lowerCase((String)deedTenantLease.getOwner()), StringUtils.lowerCase((String)deedTenantLease.getManager()));
            deedTenantLease.setViewAddresses(viewAddresses);
        } else {
            deedTenantLease.setViewAddresses(Collections.singletonList("ALL"));
        }
        if (CollectionUtils.isEmpty(deedTenantLease.getPendingTransactions())) {
            if (deedTenantLease.isConfirmed()) {
                deedTenantLease.setTransactionStatus(TransactionStatus.VALIDATED);
            } else {
                deedTenantLease.setTransactionStatus(TransactionStatus.ERROR);
            }
        } else {
            deedTenantLease.setTransactionStatus(TransactionStatus.IN_PROGRESS);
        }
        return (DeedTenantLease)this.leaseRepository.save(deedTenantLease);
    }

    private void checkLeaseBlockchainStatus(DeedTenantLease lease, DeedLeaseBlockchainState blockchainLease) {
        if (lease.getId() != blockchainLease.getId().longValue()) {
            LOG.warn("HACK Tentative or Bug? The transaction hash has been replaced by a transaction of another LEASE Identifier: {} VS {}. The information will be retrieved from Blockchain anyway to replace it.", (Object)lease.getId(), (Object)blockchainLease.getId().longValue());
        }
        if (lease.getNftId() != blockchainLease.getDeedId().longValue()) {
            LOG.warn("HACK Tentative or Bug? The transaction hash has been replaced by a transaction of another DEED: {} VS {}. The information will be retrieved from Blockchain anyway to replace it.", (Object)lease.getNftId(), (Object)blockchainLease.getDeedId().longValue());
        }
        if (!StringUtils.equalsIgnoreCase((CharSequence)lease.getManager(), (CharSequence)blockchainLease.getTenant())) {
            LOG.warn("HACK Tentative or Bug? The transaction owner isn't the same as Lease owner: {} VS {}. The information will be retrieved from Blockchain anyway to replace it.", (Object)blockchainLease.getTenant(), (Object)lease.getManager());
        }
    }

    private DeedTenantLeaseDTO buildLeaseDTO(DeedTenantLease lease) {
        if (lease == null) {
            return null;
        }
        DeedTenant deedTenant = this.getDeedTenant(lease.getNftId(), lease.getManager(), lease.getOwner());
        if (deedTenant == null) {
            return DeedTenantLeaseMapper.toDTO(lease, null, null);
        }
        return DeedTenantLeaseMapper.toDTO(lease, deedTenant.getTenantStatus(), deedTenant.getTenantProvisioningStatus());
    }

    private void addPendingTransactionHash(DeedTenantLease lease, String transactionHash) {
        ArrayList<String> pendingTransactions = !CollectionUtils.isEmpty(lease.getPendingTransactions()) ? new ArrayList<String>(lease.getPendingTransactions()) : new ArrayList<String>();
        pendingTransactions.add(StringUtils.lowerCase((String)transactionHash));
        lease.setPendingTransactions(pendingTransactions);
    }

    private void removePendingTransactionHash(DeedTenantLease lease, String transactionHash) {
        if (StringUtils.isNotBlank((CharSequence)transactionHash) && !CollectionUtils.isEmpty(lease.getPendingTransactions())) {
            ArrayList<String> pendingTransactions = new ArrayList<String>(lease.getPendingTransactions());
            pendingTransactions.remove(StringUtils.lowerCase((String)transactionHash));
            lease.setPendingTransactions(pendingTransactions);
        }
    }

    private DeedTenant getDeedTenant(long nftId, String managerAddress, String ownerAddress) {
        try {
            this.tenantService.getDeedTenantOrImport(ownerAddress, nftId);
            return this.tenantService.getDeedTenantOrImport(managerAddress, nftId);
        }
        catch (ObjectNotFoundException e) {
            throw new IllegalStateException("Unable to locate nft with id " + nftId, e);
        }
    }

    public DeedTenantLeaseDTO getCurrentLease(long nftId) {
        LeaseFilter leaseFilter = new LeaseFilter();
        leaseFilter.setNftId(nftId);
        leaseFilter.setExcludeNotConfirmed(true);
        leaseFilter.setTransactionStatus(Collections.singletonList(TransactionStatus.VALIDATED));
        return this.getLeases(leaseFilter, Pageable.unpaged()).stream().filter(lease -> lease.getStartDate().isBefore(Instant.now()) && lease.getEndDate().isAfter(Instant.now())).findFirst().orElse(null);
    }

    public Stream<DeedTenantLeaseDTO> getLeasesEndDateBetween(Instant from, Instant to) {
        List<DeedTenantLease> leases = this.leaseRepository.getByEnabledTrueAndConfirmedTrueAndEndDateBetween(from, to);
        return leases.stream().map(this::buildLeaseDTO);
    }
}

