/*
 * Decompiled with CFR 0.152.
 */
package io.meeds.deeds.service;

import io.meeds.deeds.constant.BlockchainOfferStatus;
import io.meeds.deeds.constant.DeedCard;
import io.meeds.deeds.constant.DeedCity;
import io.meeds.deeds.constant.ObjectAlreadyExistsException;
import io.meeds.deeds.constant.ObjectNotFoundException;
import io.meeds.deeds.constant.OfferType;
import io.meeds.deeds.constant.RentalPaymentPeriodicity;
import io.meeds.deeds.constant.TransactionStatus;
import io.meeds.deeds.constant.UnauthorizedOperationException;
import io.meeds.deeds.elasticsearch.model.DeedTenant;
import io.meeds.deeds.model.DeedOfferBlockchainState;
import io.meeds.deeds.model.DeedTenantOffer;
import io.meeds.deeds.model.DeedTenantOfferDTO;
import io.meeds.deeds.model.OfferFilter;
import io.meeds.deeds.service.BlockchainService;
import io.meeds.deeds.service.ListenerService;
import io.meeds.deeds.service.TenantService;
import io.meeds.deeds.storage.OfferRepository;
import io.meeds.deeds.utils.DeedTenantOfferMapper;
import java.math.BigInteger;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.StampedLock;
import java.util.stream.Collectors;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.elasticsearch.core.ElasticsearchOperations;
import org.springframework.data.elasticsearch.core.SearchHit;
import org.springframework.data.elasticsearch.core.SearchHitSupport;
import org.springframework.data.elasticsearch.core.SearchHits;
import org.springframework.data.elasticsearch.core.SearchPage;
import org.springframework.data.elasticsearch.core.query.Criteria;
import org.springframework.data.elasticsearch.core.query.CriteriaQuery;
import org.springframework.data.elasticsearch.core.query.Query;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;
import org.web3j.abi.datatypes.Address;

@Component
public class OfferService {
    private static final Logger LOG = LoggerFactory.getLogger(OfferService.class);
    public static final String OFFER_CREATED_EVENT = "deed.event.offerCreated";
    public static final String OFFER_CREATED_CONFIRMED_EVENT = "deed.event.offerCreatedConfirmed";
    public static final String OFFER_UPDATED_EVENT = "deed.event.offerUpdated";
    public static final String OFFER_UPDATED_CONFIRMED_EVENT = "deed.event.offerUpdatedConfirmed";
    public static final String OFFER_DELETED_EVENT = "deed.event.offerDeleted";
    public static final String OFFER_DELETED_CONFIRMED_EVENT = "deed.event.offerDeletedConfirmed";
    public static final String OFFER_CANCELED_EVENT = "deed.event.offerCanceled";
    public static final String OFFER_ACQUISITION_PROGRESS_EVENT = "deed.event.offerAcquisitionInProgress";
    public static final String OFFER_ACQUISITION_CONFIRMED_EVENT = "deed.event.offerAcquisitionConfirmed";
    private static final String TRANSACTION_HASH_IS_MANDATORY_MESSAGE = "Transaction Hash is Mandatory";
    @Autowired
    private OfferRepository offerRepository;
    @Autowired
    private ElasticsearchOperations elasticsearchOperations;
    @Autowired
    private TenantService tenantService;
    @Autowired
    private BlockchainService blockchainService;
    @Autowired
    private ListenerService listenerService;
    private Map<String, StampedLock> blockchainRefreshLocks = new ConcurrentHashMap<String, StampedLock>();
    private Map<String, Long> blockchainRefreshStamp = new ConcurrentHashMap<String, Long>();

    public Page<DeedTenantOfferDTO> getOffers(OfferFilter offerFilter, Pageable pageable) {
        if (offerFilter.getNetworkId() > 0L && !this.tenantService.isBlockchainNetworkValid(offerFilter.getNetworkId())) {
            return Page.empty((Pageable)pageable);
        }
        Criteria criteria = new Criteria("parentId").not().exists();
        criteria.and(new Criteria[]{criteria, new Criteria("acquired").is((Object)false)});
        if (StringUtils.isNotBlank((CharSequence)offerFilter.getOwnerAddress())) {
            criteria.and(new Criteria[]{criteria, new Criteria("owner").is((Object)StringUtils.lowerCase((String)offerFilter.getOwnerAddress()))});
        }
        if (offerFilter.isExcludeDisabled()) {
            Criteria enabledCriteria = new Criteria("enabled").is((Object)true);
            criteria.and(new Criteria[]{criteria, enabledCriteria});
        }
        if (offerFilter.isExcludeNotStarted()) {
            Criteria startDateCriteria = new Criteria("startDate").lessThan((Object)Instant.now());
            criteria.and(new Criteria[]{criteria, startDateCriteria});
        }
        if (offerFilter.getNftId() >= 0L) {
            Criteria nftIdCriteria = new Criteria("nftId").is((Object)offerFilter.getNftId());
            criteria.and(nftIdCriteria);
        }
        if (!CollectionUtils.isEmpty(offerFilter.getCardTypes())) {
            Criteria cardTypeCriteria = new Criteria("cardType").in(offerFilter.getCardTypes());
            criteria.and(new Criteria[]{criteria, cardTypeCriteria});
        }
        if (!CollectionUtils.isEmpty(offerFilter.getOfferTypes())) {
            Criteria offerCriteria = new Criteria("offerType").in(offerFilter.getOfferTypes());
            criteria.and(new Criteria[]{criteria, offerCriteria});
        }
        if (offerFilter.isExcludeExpired()) {
            Criteria expirationDateCriteria = new Criteria("expirationDate").greaterThan((Object)Instant.now());
            criteria.and(new Criteria[]{criteria, expirationDateCriteria});
        }
        if (!CollectionUtils.isEmpty(offerFilter.getTransactionStatus())) {
            Criteria transactionStatusCriteria = new Criteria("offerTransactionStatus").in(offerFilter.getTransactionStatus());
            criteria.and(new Criteria[]{criteria, transactionStatusCriteria});
        }
        if (StringUtils.isNotBlank((CharSequence)offerFilter.getCurrentAddress())) {
            visibilityCriteria = new Criteria("viewAddresses").in(new Object[]{StringUtils.lowerCase((String)offerFilter.getCurrentAddress()), "ALL"});
            criteria.and(new Criteria[]{criteria, visibilityCriteria});
        } else {
            visibilityCriteria = new Criteria("viewAddresses").in(new Object[]{"ALL"});
            criteria.and(new Criteria[]{criteria, visibilityCriteria});
        }
        CriteriaQuery query = new CriteriaQuery(criteria, pageable);
        SearchHits result = this.elasticsearchOperations.search((Query)query, DeedTenantOffer.class);
        SearchPage searchPage = SearchHitSupport.searchPageFor((SearchHits)result, (Pageable)pageable);
        return searchPage.map(SearchHit::getContent).map(DeedTenantOfferMapper::toDTO);
    }

    public DeedTenantOfferDTO getOffer(String id, String walletAddress, boolean refreshFromBlockchain) throws Exception {
        Object offer;
        if (refreshFromBlockchain) {
            offer = this.offerRepository.findById(id).orElse(null);
            id = this.refreshOfferConcurrently((DeedTenantOffer)offer, walletAddress);
        }
        if ((offer = this.getOffer(id)) == null) {
            throw new ObjectNotFoundException(this.getOfferNotExistsMessage(id));
        }
        if (!((DeedTenantOfferDTO)offer).isEnabled()) {
            throw new UnauthorizedOperationException("Offer with id " + id + " is disabled");
        }
        return offer;
    }

    public DeedTenantOfferDTO getOffer(String id) {
        DeedTenantOffer offer = this.offerRepository.findById(id).orElse(null);
        return DeedTenantOfferMapper.toDTO(offer);
    }

    public DeedTenantOfferDTO getOfferByBlockchainId(long blockchainOfferId) throws Exception {
        DeedTenantOffer offer = this.getOfferByBlockchainOfferId(blockchainOfferId, true);
        return DeedTenantOfferMapper.toDTO(offer);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public DeedTenantOfferDTO createRentingOffer(String ownerAddress, String ownerEmail, DeedTenantOfferDTO deedTenantOfferDTO) throws ObjectNotFoundException, UnauthorizedOperationException, ObjectAlreadyExistsException {
        long nftId = deedTenantOfferDTO.getNftId();
        if (!this.tenantService.isDeedOwner(ownerAddress, nftId)) {
            throw new UnauthorizedOperationException(this.getNotOwnerMessage(ownerAddress, nftId));
        }
        String transactionHash = StringUtils.lowerCase((String)deedTenantOfferDTO.getOfferTransactionHash());
        if (StringUtils.isBlank((CharSequence)transactionHash)) {
            throw new IllegalStateException(TRANSACTION_HASH_IS_MANDATORY_MESSAGE);
        }
        DeedTenant deedTenant = this.getDeedTenant(nftId, ownerAddress);
        if (deedTenant == null) {
            throw new ObjectNotFoundException(this.getNftNotExistsMessage(nftId));
        }
        this.checkTransactionHashIsUnknown(transactionHash);
        DeedTenantOffer deedTenantOffer = DeedTenantOfferMapper.fromDTO(deedTenantOfferDTO);
        try {
            this.setOfferNftInformation(deedTenantOffer, deedTenant);
            deedTenantOffer.setOwner(StringUtils.lowerCase((String)ownerAddress));
            deedTenantOffer.setOwnerEmail(ownerEmail);
            deedTenantOffer = this.saveOffer(deedTenantOffer);
            DeedTenantOfferDTO deedTenantOfferDTO2 = DeedTenantOfferMapper.toDTO(deedTenantOffer);
            return deedTenantOfferDTO2;
        }
        finally {
            this.listenerService.publishEvent(OFFER_CREATED_EVENT, (Object)deedTenantOffer);
        }
    }

    public DeedTenantOfferDTO updateRentingOffer(String walletAddress, DeedTenantOfferDTO deedTenantOfferDTO) throws ObjectNotFoundException, UnauthorizedOperationException, ObjectAlreadyExistsException {
        if (!this.tenantService.isDeedOwner(walletAddress, deedTenantOfferDTO.getNftId())) {
            throw new UnauthorizedOperationException(this.getNotOwnerMessage(walletAddress, deedTenantOfferDTO.getNftId()));
        }
        if (!deedTenantOfferDTO.isEnabled()) {
            throw new UnauthorizedOperationException("Offer has already been canceled");
        }
        DeedTenantOffer existingOffer = this.offerRepository.findById(deedTenantOfferDTO.getId()).orElse(null);
        if (existingOffer == null) {
            throw new ObjectNotFoundException("Offer with id  " + deedTenantOfferDTO.getId() + " doesn't exist");
        }
        existingOffer.setDescription(deedTenantOfferDTO.getDescription());
        existingOffer.setPaymentPeriodicity(deedTenantOfferDTO.getPaymentPeriodicity());
        String transactionHash = deedTenantOfferDTO.getOfferTransactionHash();
        if (StringUtils.isNotBlank((CharSequence)transactionHash)) {
            this.checkTransactionHashIsUnknown(transactionHash);
            DeedTenantOffer deedTenantOfferUpdate = DeedTenantOfferMapper.toOfferUpdateChangeLog(deedTenantOfferDTO, existingOffer);
            deedTenantOfferUpdate = this.saveOffer(deedTenantOfferUpdate);
            existingOffer.setUpdateId(deedTenantOfferUpdate.getId());
        }
        existingOffer = this.saveOffer(existingOffer);
        this.listenerService.publishEvent(OFFER_UPDATED_EVENT, (Object)existingOffer);
        return DeedTenantOfferMapper.toDTO(existingOffer);
    }

    public void markOfferAcquisitionInProgress(long nftId, String transactionHash, Instant validStartDate) throws ObjectAlreadyExistsException {
        List<DeedTenantOffer> offers;
        LOG.debug("Mark Offers of nft {} as acquisition in progress", (Object)nftId);
        if (StringUtils.isNotBlank((CharSequence)transactionHash)) {
            this.checkTransactionHashIsUnknown(transactionHash);
        }
        if (!CollectionUtils.isEmpty(offers = this.offerRepository.findByNftId(nftId))) {
            offers.forEach(parentOffer -> {
                if (this.isOfferOngoing((DeedTenantOffer)parentOffer, null, validStartDate)) {
                    this.createOfferAcquisitionChangeLog((DeedTenantOffer)parentOffer, transactionHash);
                }
            });
        }
    }

    public void deleteRentingOffer(String walletAddress, String offerId, String transactionHash) throws ObjectNotFoundException, UnauthorizedOperationException, ObjectAlreadyExistsException {
        DeedTenantOffer existingOffer = this.offerRepository.findById(offerId).orElse(null);
        if (existingOffer == null) {
            throw new ObjectNotFoundException("Offer with id  " + offerId + " doesn't exist");
        }
        if (!this.tenantService.isDeedOwner(walletAddress, existingOffer.getNftId())) {
            throw new UnauthorizedOperationException(this.getNotOwnerMessage(walletAddress, existingOffer.getNftId()));
        }
        if (!existingOffer.isEnabled()) {
            throw new UnauthorizedOperationException("Offer has already been canceled");
        }
        if (StringUtils.isBlank((CharSequence)transactionHash)) {
            throw new IllegalStateException(TRANSACTION_HASH_IS_MANDATORY_MESSAGE);
        }
        this.checkTransactionHashIsUnknown(transactionHash);
        DeedTenantOffer offerDelete = DeedTenantOfferMapper.toOfferChangeLog(existingOffer, transactionHash);
        offerDelete = this.saveOffer(offerDelete);
        existingOffer.setDeleteId(offerDelete.getId());
        existingOffer = this.saveOffer(existingOffer);
        this.listenerService.publishEvent(OFFER_DELETED_EVENT, (Object)existingOffer);
    }

    public void cancelOffers(String newOwner, long nftId) {
        LOG.debug("Cancel offers of nftId {} due to a changed nft owner {}", (Object)nftId, (Object)newOwner);
        List<DeedTenantOffer> offers = this.offerRepository.findByOwnerNotAndNftIdAndEnabledTrue(newOwner, nftId);
        if (!CollectionUtils.isEmpty(offers)) {
            offers.forEach(offer -> {
                LOG.debug("Cancel offer {} due to a changed nft owner {}", (Object)offer.getOfferId(), (Object)newOwner);
                this.cancelOffer((DeedTenantOffer)offer);
            });
        }
    }

    public List<DeedTenantOfferDTO> getPendingTransactions() {
        return this.offerRepository.findByOfferTransactionStatusInOrderByCreatedDateAsc(Arrays.asList(TransactionStatus.IN_PROGRESS)).stream().map(DeedTenantOfferMapper::toDTO).collect(Collectors.toList());
    }

    public void updateRentingOfferStatusFromBlockchain(String offerId, Map<BlockchainOfferStatus, DeedOfferBlockchainState> minedEvents) throws Exception {
        DeedTenantOffer offer = this.offerRepository.findById(offerId).orElse(null);
        if (offer == null) {
            throw new IllegalArgumentException("Wrong Offer technical internal identifier " + offerId);
        }
        if (minedEvents.isEmpty()) {
            this.saveOfferTransactionAsError(offer.getId());
        } else {
            if (minedEvents.size() > 1) {
                LOG.warn("Mined events for a single transaction seems to hold more than one Offer event. This is not supported yet. Will use the first retrieved event");
            }
            Map.Entry<BlockchainOfferStatus, DeedOfferBlockchainState> entry = minedEvents.entrySet().iterator().next();
            this.updateRentingOfferStatusFromBlockchain(offerId, entry.getValue(), entry.getKey());
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public DeedTenantOffer updateOfferFromBlockchain(DeedOfferBlockchainState blockchainOffer, boolean isBlockchainScan) throws Exception {
        DeedTenantOffer offer = this.getParentOfferByBlockchainId(blockchainOffer.getId().longValue());
        try {
            DeedTenantOffer deedTenantOffer = this.updateOfferFromBlockchain(offer, blockchainOffer);
            return deedTenantOffer;
        }
        finally {
            if (isBlockchainScan && offer != null) {
                LOG.debug("Delete acquired UI Refresh Lock on Offer after blockchain scan finished {}", (Object)offer.getParentId());
                String parentId = offer.getId();
                this.releaseExplicitOfferRefreshLock(parentId);
            }
        }
    }

    public void markOfferAsAcquired(long offerId, Instant leaseEndDate) throws Exception {
        LOG.debug("Mark offer {} as acquired by a Tenant and cancel all offer before the new Lease end date", (Object)offerId);
        DeedTenantOffer offer = this.getOfferByBlockchainOfferId(offerId, true);
        if (offer != null) {
            this.updateRentingOfferStatusFromBlockchain(offer.getId(), null, BlockchainOfferStatus.OFFER_ACQUIRED);
            List<DeedTenantOffer> offers = this.offerRepository.findByNftId(offer.getNftId());
            if (!CollectionUtils.isEmpty(offers)) {
                offers.forEach(parentOffer -> {
                    if (this.isOfferOngoing((DeedTenantOffer)parentOffer, offer, leaseEndDate)) {
                        this.cancelOffer((DeedTenantOffer)parentOffer);
                    }
                });
            }
        }
    }

    public void saveOfferTransactionAsError(String offerId) throws Exception {
        DeedTenantOffer offer = this.offerRepository.findById(offerId).orElse(null);
        if (offer == null) {
            throw new IllegalArgumentException("Wrong Offer id");
        }
        if (this.isChangelog(offer)) {
            DeedTenantOffer parentOffer = this.offerRepository.findById(offer.getParentId()).orElse(null);
            this.cancelChangeLog(parentOffer, offer);
        } else if (!this.blockchainService.isOfferEnabled(offer.getOfferId())) {
            this.offerRepository.delete(offer);
        } else {
            LOG.warn("Don't know what to do with a parent offer {} with blockchain Id {} that exists on blockchain and that is meant to have a valid transaction", (Object)offer.getId(), (Object)offer.getOfferId());
        }
    }

    protected void releaseExplicitOfferRefreshLock(String parentOfferId) {
        StampedLock lock = this.blockchainRefreshLocks.remove(parentOfferId);
        if (lock != null && this.blockchainRefreshStamp.containsKey(parentOfferId)) {
            Long stamp = this.blockchainRefreshStamp.remove(parentOfferId);
            try {
                lock.unlock(stamp);
            }
            catch (Exception e) {
                LOG.debug("Wasn't able to release Blockchain Refresh lock of parent offer with id {}. Unlocked anyway and released.", (Object)parentOfferId, (Object)e);
            }
        }
    }

    private boolean acquireExplicitOfferRefreshLock(String offerId) {
        try {
            StampedLock lock = this.blockchainRefreshLocks.computeIfAbsent(offerId, key -> new StampedLock());
            long stamp = lock.tryWriteLock();
            if (stamp > 0L) {
                this.blockchainRefreshStamp.put(offerId, stamp);
                return true;
            }
            lock.tryWriteLock(3L, TimeUnit.SECONDS);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }
        catch (Exception e) {
            LOG.debug("Refresh Lock on Offer {} not acquired after 3 seconds, proceed to refresh offer drom DB", (Object)offerId);
        }
        return false;
    }

    private DeedTenantOffer getParentOfferFromChangelog(DeedTenantOffer offer) {
        if (!this.isChangelog(offer)) {
            return offer;
        }
        DeedTenantOffer parentOffer = this.offerRepository.findById(offer.getParentId()).orElse(null);
        if (parentOffer == null) {
            return offer;
        }
        parentOffer.setDescription(offer.getDescription());
        parentOffer.setPaymentPeriodicity(offer.getPaymentPeriodicity());
        return parentOffer;
    }

    private String refreshOfferConcurrently(DeedTenantOffer offer, String walletAddress) throws Exception {
        if (offer == null) {
            throw new ObjectNotFoundException();
        }
        String offerId = this.isChangelog(offer) ? offer.getParentId() : offer.getId();
        boolean isRefreshPermitted = this.acquireExplicitOfferRefreshLock(offerId);
        if (isRefreshPermitted) {
            try {
                LOG.info("Refreshing Changed offer with id {} on blockchain on request of wallet {}", (Object)offerId, (Object)walletAddress);
                this.refreshOffer(offer);
            }
            catch (ObjectNotFoundException e) {
                LOG.debug("It seems that there was a concurrent refresh with scheduled tasks, the offer was already refreshed. Original error: {}", (Object)e.getMessage());
            }
        } else {
            LOG.debug("Refresh already made by another process of changed offer with id {}. Try just to retrieve the newer version.", (Object)offerId);
        }
        return offerId;
    }

    private void refreshOffer(DeedTenantOffer offer) throws Exception {
        if (this.isOfferNotConfirmedYet(offer)) {
            Map offerTransactionEvents;
            if (this.isOfferTransactionInProgress(offer) && !(offerTransactionEvents = this.blockchainService.getOfferTransactionEvents(offer.getOfferTransactionHash())).isEmpty()) {
                this.updateRentingOfferStatusFromBlockchain(offer.getId(), offerTransactionEvents);
            }
        } else {
            long blockNumber = this.blockchainService.getLastBlock();
            DeedOfferBlockchainState blockchainOffer = this.blockchainService.getOfferById(BigInteger.valueOf(offer.getOfferId()), BigInteger.valueOf(blockNumber), offer.getOfferTransactionHash());
            if (this.isOfferDeleted(blockchainOffer)) {
                this.updateRentingOfferStatusFromBlockchain(StringUtils.isBlank((CharSequence)offer.getDeleteId()) ? offer.getId() : offer.getDeleteId(), blockchainOffer, BlockchainOfferStatus.OFFER_DELETED);
            } else {
                Set<String> acquisitionIds;
                if (StringUtils.isNotBlank((CharSequence)offer.getUpdateId())) {
                    this.updateRentingOfferStatusFromBlockchain(blockchainOffer, offer.getUpdateId(), BlockchainOfferStatus.OFFER_UPDATED);
                }
                if (!CollectionUtils.isEmpty(acquisitionIds = offer.getAcquisitionIds())) {
                    for (String acquisitionId : acquisitionIds) {
                        this.updateRentingOfferStatusFromBlockchain(blockchainOffer, acquisitionId, BlockchainOfferStatus.OFFER_ACQUIRED);
                    }
                }
            }
        }
    }

    private void updateRentingOfferStatusFromBlockchain(DeedOfferBlockchainState blockchainOffer, String offerChangeLogId, BlockchainOfferStatus blockchainStatus) throws Exception {
        DeedTenantOffer changeLogOffer;
        if (StringUtils.isNotBlank((CharSequence)offerChangeLogId) && this.isChangelog(changeLogOffer = (DeedTenantOffer)this.offerRepository.findById(offerChangeLogId).orElse(null)) && this.isOfferTransactionInProgress(changeLogOffer) && this.blockchainService.isTransactionMined(changeLogOffer.getOfferTransactionHash())) {
            if (this.blockchainService.isTransactionConfirmed(changeLogOffer.getOfferTransactionHash())) {
                this.updateRentingOfferStatusFromBlockchain(offerChangeLogId, blockchainOffer, blockchainStatus);
            } else {
                this.cancelOffer(changeLogOffer);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void updateRentingOfferStatusFromBlockchain(String updateOrDeleteOrParentId, DeedOfferBlockchainState blockchainOffer, BlockchainOfferStatus status) throws Exception {
        DeedTenantOffer offer = this.offerRepository.findById(updateOrDeleteOrParentId).orElse(null);
        if (offer == null) {
            throw new ObjectNotFoundException("Offer with id " + updateOrDeleteOrParentId + "wasn't found");
        }
        long blockchainOfferId = blockchainOffer == null ? offer.getOfferId() : blockchainOffer.getId().longValue();
        LOG.debug("Attempt to refresh Changed offer with internal id {} and status {} retrieved from blockchain with id {}", new Object[]{updateOrDeleteOrParentId, status, blockchainOfferId});
        boolean isCreated = status == BlockchainOfferStatus.OFFER_CREATED && !this.isBlockchainOfferIdKnown(blockchainOfferId);
        boolean isDeleted = status == BlockchainOfferStatus.OFFER_DELETED;
        boolean isAcquired = status == BlockchainOfferStatus.OFFER_ACQUIRED;
        boolean isUpdated = status == BlockchainOfferStatus.OFFER_UPDATED || !isDeleted && !isAcquired && !isCreated && offer.getOfferId() > 0L;
        DeedTenantOffer offerToChange = null;
        try {
            if (isCreated) {
                if (blockchainOffer != null) {
                    offerToChange = this.createOfferByBlockchainChange(blockchainOffer, offer);
                }
            } else if (isDeleted) {
                offerToChange = this.deleteOfferByBlockchainChange(blockchainOffer, offer);
            } else if (isAcquired) {
                offerToChange = this.markOfferAcquiredByBlockchainChange(offer);
            } else if (isUpdated) {
                offerToChange = this.updateOfferBlockchainChange(blockchainOffer, offer);
            } else {
                LOG.warn("Wasn't able to determine offer {} event type. Simply disable it for now", (Object)offer.getOfferId());
                offer.setEnabled(false);
                offer.setOfferTransactionStatus(TransactionStatus.NONE);
                this.saveOffer(offer);
            }
        }
        finally {
            if (offerToChange != null) {
                if (isCreated) {
                    this.listenerService.publishEvent(OFFER_CREATED_CONFIRMED_EVENT, (Object)DeedTenantOfferMapper.toDTO(offerToChange));
                } else if (isUpdated) {
                    this.listenerService.publishEvent(OFFER_UPDATED_CONFIRMED_EVENT, (Object)DeedTenantOfferMapper.toDTO(offerToChange));
                } else if (isDeleted) {
                    this.listenerService.publishEvent(OFFER_DELETED_CONFIRMED_EVENT, (Object)DeedTenantOfferMapper.toDTO(offerToChange));
                } else if (isAcquired) {
                    this.listenerService.publishEvent(OFFER_ACQUISITION_CONFIRMED_EVENT, (Object)DeedTenantOfferMapper.toDTO(offerToChange));
                }
            }
        }
    }

    private void createOfferAcquisitionChangeLog(DeedTenantOffer parentOffer, String transactionHash) {
        DeedTenantOffer offerChangeLog = DeedTenantOfferMapper.toOfferChangeLog(parentOffer, transactionHash);
        offerChangeLog = this.saveOffer(offerChangeLog, false);
        HashSet<String> acquisitionIds = CollectionUtils.isEmpty(parentOffer.getAcquisitionIds()) ? new HashSet<String>() : new HashSet<String>(parentOffer.getAcquisitionIds());
        acquisitionIds.add(offerChangeLog.getId());
        parentOffer.setAcquisitionIds(acquisitionIds);
        parentOffer = this.saveOffer(parentOffer);
        this.listenerService.publishEvent(OFFER_ACQUISITION_PROGRESS_EVENT, (Object)parentOffer);
        LOG.debug("Mark Offer {} as acquisition in progress, acquisition ids for current NFT = {}", (Object)parentOffer.getOfferId(), acquisitionIds);
    }

    private DeedTenantOffer createOfferByBlockchainChange(DeedOfferBlockchainState blockchainOffer, DeedTenantOffer offer) throws Exception {
        boolean invalidMinedOffer;
        boolean offerDeleted = this.isOfferDeleted(blockchainOffer);
        boolean offerIdKnown = this.isBlockchainOfferIdKnown(blockchainOffer.getId().longValue());
        boolean bl = invalidMinedOffer = offerDeleted || offerIdKnown;
        if (invalidMinedOffer && StringUtils.isNotBlank((CharSequence)offer.getOfferTransactionHash()) && !this.blockchainService.isTransactionConfirmed(offer.getOfferTransactionHash())) {
            offer.setOfferTransactionStatus(TransactionStatus.ERROR);
            offer.setEnabled(false);
            offer.setViewAddresses(Collections.emptyList());
            LOG.warn("Disabled invalid Offer with id {}. offerIdKnown= {}, offerDeleted={}", new Object[]{blockchainOffer.getId().longValue(), offerDeleted, offerIdKnown});
            return this.saveOffer(offer);
        }
        if (invalidMinedOffer) {
            LOG.warn("Invalid Offer status retrieved from blockchain with id {}. offerIdKnown= {}, offerDeleted={}. Ignore adding it for now and wait for periodic check again", new Object[]{blockchainOffer.getId().longValue(), offerDeleted, offerIdKnown});
            return null;
        }
        offer.setEnabled(true);
        offer.setOfferTransactionStatus(TransactionStatus.VALIDATED);
        offer.setCreatedDate(Instant.now());
        return this.updateOfferFromBlockchainStatus(offer, blockchainOffer, false);
    }

    private DeedTenantOffer deleteOfferByBlockchainChange(DeedOfferBlockchainState blockchainOffer, DeedTenantOffer offer) {
        LOG.debug("Refreshing deleted offer with id {} switch blockchain status", (Object)offer.getOfferId());
        DeedTenantOffer offerToChange = this.getParentOfferFromChangelog(offer);
        offerToChange.setEnabled(false);
        offerToChange.setLastCheckedBlock(this.getLastCheckedBlockNumber(blockchainOffer, offerToChange));
        return this.saveOfferAndDeleteChangeLog(offerToChange, offer.getId());
    }

    private DeedTenantOffer markOfferAcquiredByBlockchainChange(DeedTenantOffer offer) {
        LOG.debug("Refreshing acuired offer with id {} switch blockchain status", (Object)offer.getOfferId());
        DeedTenantOffer parentOffer = this.getParentOfferFromChangelog(offer);
        parentOffer.setAcquired(true);
        return this.saveOfferAndDeleteChangeLogs(parentOffer);
    }

    private DeedTenantOffer updateOfferBlockchainChange(DeedOfferBlockchainState blockchainOffer, DeedTenantOffer offer) throws Exception {
        long lastCheckedBlock = offer.getLastCheckedBlock();
        DeedTenantOffer offerToChange = this.getParentOfferFromChangelog(offer);
        boolean isChangelogAlreadyApplied = offerToChange.getLastCheckedBlock() > lastCheckedBlock;
        boolean isDeedOfferDeleted = this.isOfferDeleted(blockchainOffer);
        if (isDeedOfferDeleted) {
            LOG.debug("Offer Changelog with id {} of parent offer id {} (offerId = {}) will not be applied since the offer is already deleted", new Object[]{offer.getId(), offer.getParentId(), offer.getOfferId()});
        } else if (isChangelogAlreadyApplied) {
            LOG.debug("Offer Changelog with id {} of parent offer id {} (offerId = {}) was already applied, delete it", new Object[]{offer.getId(), offer.getParentId(), offer.getOfferId()});
        } else {
            offerToChange = this.updateOfferFromBlockchainStatus(offerToChange, blockchainOffer, true);
        }
        return this.saveOfferAndDeleteChangeLog(offerToChange, offer.getId());
    }

    private void cancelOffer(DeedTenantOffer offer) {
        if (offer == null) {
            return;
        }
        if (this.isChangelog(offer)) {
            DeedTenantOffer parentOffer = this.offerRepository.findById(offer.getParentId()).orElse(null);
            this.cancelChangeLog(parentOffer, offer);
        } else {
            offer.setEnabled(false);
            offer = this.saveOfferAndDeleteChangeLogs(offer);
            this.listenerService.publishEvent(OFFER_CANCELED_EVENT, (Object)offer);
        }
    }

    private void cancelChangeLog(DeedTenantOffer parentOffer, DeedTenantOffer offer) {
        if (parentOffer == null) {
            this.offerRepository.delete(offer);
        } else {
            this.saveOfferAndDeleteChangeLog(parentOffer, offer.getId());
        }
    }

    private boolean isChangelog(DeedTenantOffer offer) {
        return offer != null && StringUtils.isNotBlank((CharSequence)offer.getParentId());
    }

    private DeedTenantOffer saveOfferAndDeleteChangeLog(DeedTenantOffer parentOffer, String changeLogId) {
        if (this.isChangelog(parentOffer) || parentOffer == null) {
            throw new IllegalArgumentException("Attempt to save changelog offer: " + parentOffer);
        }
        LOG.debug("Delete offer changelog after applying it with id {} for parent id {}", (Object)changeLogId, (Object)parentOffer.getId());
        if (StringUtils.equals((CharSequence)parentOffer.getUpdateId(), (CharSequence)changeLogId)) {
            parentOffer.setUpdateId(null);
            parentOffer = this.saveOffer(parentOffer);
            this.offerRepository.deleteById(changeLogId);
            return parentOffer;
        }
        if (StringUtils.equals((CharSequence)parentOffer.getDeleteId(), (CharSequence)changeLogId)) {
            return this.saveOfferAndDeleteChangeLogs(parentOffer);
        }
        if (parentOffer.getAcquisitionIds() != null && parentOffer.getAcquisitionIds().contains(changeLogId)) {
            return this.saveOfferAndDeleteChangeLogs(parentOffer);
        }
        return this.saveOffer(parentOffer);
    }

    private DeedTenantOffer saveOfferAndDeleteChangeLogs(DeedTenantOffer parentOffer) {
        if (this.isChangelog(parentOffer) || parentOffer == null) {
            throw new IllegalArgumentException("Attempt to save changelog offer: " + parentOffer);
        }
        parentOffer.setUpdateId(null);
        parentOffer.setDeleteId(null);
        parentOffer.setAcquisitionIds(Collections.emptySet());
        parentOffer = this.saveOffer(parentOffer);
        this.offerRepository.deleteByParentId(parentOffer.getId());
        return parentOffer;
    }

    private DeedTenantOffer updateOfferFromBlockchainStatus(DeedTenantOffer offer, DeedOfferBlockchainState blockchainOffer, boolean isUpdate) throws Exception {
        this.checkOfferBlockchainStatus(offer, blockchainOffer, isUpdate);
        offer.setOfferTransactionStatus(TransactionStatus.VALIDATED);
        return this.updateOfferFromBlockchain(offer, blockchainOffer);
    }

    private DeedTenantOffer updateOfferFromBlockchain(DeedTenantOffer offer, DeedOfferBlockchainState blockchainOffer) throws Exception {
        if (blockchainOffer == null) {
            throw new IllegalArgumentException("Blockchain Offer is null");
        }
        boolean isDeletedOffer = this.isOfferDeleted(blockchainOffer);
        long blockchainOfferId = blockchainOffer.getId().longValue();
        long nftId = blockchainOffer.getDeedId().longValue();
        String ownerAddress = blockchainOffer.getCreator();
        long blockNumber = blockchainOffer.getBlockNumber().longValue();
        if (offer == null) {
            if (isDeletedOffer) {
                LOG.debug("The retrieved offer doesn't exist neither on database nor on blockchain, this seems to be an outdated event about deleted Offer, ignore it");
                return null;
            }
            if (this.isOfferTransactionHashDuplicated(blockchainOffer.getTransactionHash(), null)) {
                LOG.debug("The retrieved offer {} creation event already known, ignore this event", (Object)blockchainOfferId);
                return null;
            }
            offer = new DeedTenantOffer();
            offer.setOfferType(OfferType.RENTING);
            offer.setPaymentPeriodicity(RentalPaymentPeriodicity.ONE_MONTH);
            offer.setOfferTransactionStatus(TransactionStatus.VALIDATED);
            offer.setCreatedDate(Instant.now());
            offer.setModifiedDate(offer.getCreatedDate());
            DeedTenant deedTenant = this.getDeedTenant(nftId, ownerAddress);
            this.setOfferNftInformation(offer, deedTenant);
            LOG.info("Add offer created directly on blockchain with id {} on deed {} as enabled = {}", new Object[]{blockchainOfferId, nftId, offer.isEnabled()});
        } else {
            long lastCheckedBlock = offer.getLastCheckedBlock();
            if (blockNumber > 0L && lastCheckedBlock > 0L && lastCheckedBlock >= blockNumber) {
                if (isDeletedOffer && offer.isEnabled() && !this.blockchainService.isOfferEnabled(blockchainOfferId)) {
                    LOG.debug("It seems that Offer {} has been already deleted in blockchain in block {}. Mark it as disabled.", (Object)offer.getOfferId(), (Object)lastCheckedBlock);
                    offer.setEnabled(false);
                    this.saveOffer(offer);
                } else {
                    LOG.debug("Cancel Offer Status Refresh which is already updated in block {} (last checked block {}). Ignore updating.", (Object)blockNumber, (Object)lastCheckedBlock);
                }
                return offer;
            }
            offer.setModifiedDate(Instant.now());
        }
        offer.setLastCheckedBlock(this.getLastCheckedBlockNumber(blockchainOffer, offer));
        this.copyBlockchainAttributes(offer, blockchainOffer);
        offer = this.saveOffer(offer);
        LOG.debug("Refresh done for Offer {} from blockchain with id {}", (Object)offer.getId(), (Object)offer.getOfferId());
        return offer;
    }

    private void copyBlockchainAttributes(DeedTenantOffer offer, DeedOfferBlockchainState blockchainOffer) throws Exception {
        long blockchainOfferId = blockchainOffer.getId().longValue();
        long nftId = blockchainOffer.getDeedId().longValue();
        if (blockchainOfferId <= 0L || nftId <= 0L) {
            LOG.debug("It seems that we have a bug here, since we attempt copy a deleted blockchain offer. Ignore updating.");
            return;
        }
        if (StringUtils.isBlank((CharSequence)offer.getOfferTransactionHash()) && StringUtils.isNotBlank((CharSequence)blockchainOffer.getTransactionHash())) {
            offer.setOfferTransactionHash(StringUtils.lowerCase((String)blockchainOffer.getTransactionHash()));
        }
        offer.setOfferId(blockchainOfferId);
        offer.setNftId(nftId);
        offer.setOwner(StringUtils.lowerCase((String)blockchainOffer.getCreator()));
        offer.setAmount(blockchainOffer.getPrice().divide(BigInteger.valueOf(10L).pow(18)).doubleValue());
        offer.setAllDurationAmount(blockchainOffer.getAllDurationPrice().divide(BigInteger.valueOf(10L).pow(18)).doubleValue());
        offer.setStartDate(Instant.ofEpochSecond(blockchainOffer.getOfferStartDate().longValue()));
        offer.setEnabled(this.blockchainService.isOfferEnabled(offer.getOfferId()));
        long expirationTimeInSeconds = blockchainOffer.getOfferExpirationDate().longValue();
        offer.setExpirationDate(expirationTimeInSeconds == 0L ? DeedTenantOfferMapper.MAX_DATE_VALUE : Instant.ofEpochSecond(expirationTimeInSeconds));
        if (StringUtils.equals((CharSequence)Address.DEFAULT.getValue(), (CharSequence)blockchainOffer.getAuthorizedTenant())) {
            offer.setHostAddress(null);
        } else {
            offer.setHostAddress(StringUtils.lowerCase((String)blockchainOffer.getAuthorizedTenant()));
        }
        offer.setOwnerMintingPercentage(blockchainOffer.getOwnerMintingPercentage().intValue());
        this.setDuration(offer, blockchainOffer.getMonths().intValue());
        this.setNoticePeriod(offer, blockchainOffer.getNoticePeriod().intValue());
        this.setExpirationDuration(offer, blockchainOffer.getOfferExpirationDays().intValue());
    }

    private boolean isOfferOngoing(DeedTenantOffer parentOffer, DeedTenantOffer offer, Instant leaseEndDate) {
        if (!parentOffer.isEnabled() || parentOffer.isAcquired() || parentOffer.getOfferTransactionStatus() == TransactionStatus.ERROR || this.isChangelog(parentOffer) || offer != null && StringUtils.equals((CharSequence)parentOffer.getId(), (CharSequence)offer.getId())) {
            return false;
        }
        if (this.isOfferStartsAfter(parentOffer, leaseEndDate)) {
            LOG.debug("Offer {} of Acquired NFT {} will not be marqued as 'acquisition in progress' since its start date is far in the future after acquired lease end", (Object)parentOffer.getId(), (Object)parentOffer.getNftId());
            return false;
        }
        return true;
    }

    private boolean isOfferStartsAfter(DeedTenantOffer offer, Instant leaseEndDate) {
        return offer.getStartDate() != null && (offer.getStartDate().isAfter(leaseEndDate) || offer.getStartDate().equals(leaseEndDate));
    }

    private void checkTransactionHashIsUnknown(String transactionHash) throws ObjectAlreadyExistsException {
        boolean isKnown = this.isOfferTransactionHashDuplicated(transactionHash, null);
        if (isKnown) {
            throw new ObjectAlreadyExistsException("Offer with same Transaction Hash " + transactionHash + " already exists");
        }
    }

    private boolean isOfferTransactionHashDuplicated(String transactionHash, String offerId) {
        DeedTenantOffer offer = this.getOfferByTransactionHash(transactionHash);
        return offer != null && (StringUtils.isBlank((CharSequence)offerId) || !StringUtils.equals((CharSequence)offerId, (CharSequence)offer.getId()));
    }

    private DeedTenantOffer getOfferByTransactionHash(String transactionHash) {
        return this.offerRepository.findByOfferTransactionHash(StringUtils.lowerCase((String)transactionHash));
    }

    private boolean isBlockchainOfferIdKnown(long blockchainOfferId) {
        return !CollectionUtils.isEmpty(this.offerRepository.findByOfferIdAndParentIdIsNull(blockchainOfferId));
    }

    private void setExpirationDuration(DeedTenantOffer offer, int expirationDurationDays) {
        offer.setExpirationDays(expirationDurationDays);
    }

    private void setNoticePeriod(DeedTenantOffer offer, int noticePeriodDuration) {
        offer.setNoticePeriod(noticePeriodDuration);
    }

    private void setDuration(DeedTenantOffer offer, int rentalDurationInMonths) {
        offer.setMonths(rentalDurationInMonths);
    }

    private DeedTenantOffer saveOffer(DeedTenantOffer offer) {
        return this.saveOffer(offer, true);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private DeedTenantOffer saveOffer(DeedTenantOffer offer, boolean checkTransactionHashDuplication) {
        DeedTenantOffer existingOffer;
        if (offer.getExpirationDays() == 0) {
            offer.setExpirationDate(DeedTenantOfferMapper.MAX_DATE_VALUE);
        }
        if (offer.isAcquired() || !offer.isEnabled()) {
            offer.setEnabled(false);
            offer.setViewAddresses(Collections.emptyList());
        } else if (offer.getOfferId() != 0L && offer.getOfferTransactionStatus() == TransactionStatus.VALIDATED) {
            offer.setViewAddresses(Collections.singletonList("ALL"));
        } else {
            offer.setViewAddresses(Collections.singletonList(StringUtils.lowerCase((String)offer.getOwner())));
        }
        String id = offer.getId();
        if (StringUtils.isBlank((CharSequence)offer.getOfferTransactionHash())) {
            throw new IllegalStateException("Offer " + id + "/" + offer.getOfferId() + " transaction hash is mandatory");
        }
        boolean isNewOffer = StringUtils.isBlank((CharSequence)id);
        if (!isNewOffer && (existingOffer = (DeedTenantOffer)this.offerRepository.findById(id).orElse(null)) == null) {
            throw new IllegalStateException("Offer to update doesn't exists");
        }
        boolean isChangelog = this.isChangelog(offer);
        if (isNewOffer && !isChangelog && offer.getOfferId() > 0L && this.isBlockchainOfferIdKnown(offer.getOfferId())) {
            throw new IllegalStateException("Offer with same identifier already exists, must not save twice the same Parent OfferId");
        }
        if (isNewOffer && checkTransactionHashDuplication && this.isOfferTransactionHashDuplicated(offer.getOfferTransactionHash(), id)) {
            throw new IllegalStateException("Offer with same transaction hash already exists");
        }
        String parentId = offer.getParentId();
        try {
            DeedTenantOffer deedTenantOffer = (DeedTenantOffer)this.offerRepository.save(offer);
            return deedTenantOffer;
        }
        finally {
            if (isChangelog && isNewOffer) {
                LOG.debug("Delete acquired UI Refresh Lock on Offer {} after a new changelog has been added", (Object)parentId);
                this.releaseExplicitOfferRefreshLock(parentId);
            }
        }
    }

    private long getLastCheckedBlockNumber(DeedOfferBlockchainState blockchainOffer, DeedTenantOffer offer) {
        return Math.max(blockchainOffer.getBlockNumber().longValue(), offer.getLastCheckedBlock());
    }

    private String getNftNotExistsMessage(long nftId) {
        return "Deed Tenant with id " + nftId + " doesn't exists";
    }

    private String getOfferNotExistsMessage(String id) {
        return "Deed Renting Offer with id " + id + " doesn't exists";
    }

    private String getNotOwnerMessage(String walletAddress, long nftId) {
        return walletAddress + " isn't owner of Deed NFT #" + nftId;
    }

    private boolean isOfferNotConfirmedYet(DeedTenantOffer offer) {
        return offer.getOfferId() == 0L;
    }

    private boolean isOfferTransactionInProgress(DeedTenantOffer offer) {
        return offer.getOfferTransactionStatus() == TransactionStatus.IN_PROGRESS && StringUtils.isNotBlank((CharSequence)offer.getOfferTransactionHash());
    }

    private void checkOfferBlockchainStatus(DeedTenantOffer offer, DeedOfferBlockchainState blockchainOffer, boolean isUpdate) {
        if (isUpdate && offer.getOfferId() != blockchainOffer.getId().longValue()) {
            LOG.warn("HACK Tentative or Bug? The transaction hash has been replaced by a transaction of another OFFER Identifier: {} VS {}. The information will be retrieved from Blockchain anyway to replace it.", (Object)offer.getOfferId(), (Object)blockchainOffer.getId().longValue());
        }
        if (offer.getNftId() != blockchainOffer.getDeedId().longValue()) {
            LOG.warn("HACK Tentative or Bug? The transaction hash has been replaced by a transaction of another DEED: {} VS {}. The information will be retrieved from Blockchain anyway to replace it.", (Object)offer.getNftId(), (Object)blockchainOffer.getDeedId().longValue());
        }
        if (!StringUtils.equalsIgnoreCase((CharSequence)offer.getOwner(), (CharSequence)blockchainOffer.getCreator())) {
            LOG.warn("HACK Tentative or Bug? The transaction owner isn't the same as Offer owner: {} VS {}. The information will be retrieved from Blockchain anyway to replace it.", (Object)blockchainOffer.getCreator(), (Object)offer.getOwner());
        }
    }

    private void setOfferNftInformation(DeedTenantOffer deedTenantOffer, DeedTenant deedTenant) {
        if (deedTenant.getCardType() >= 0) {
            DeedCard cardType = DeedCard.values()[deedTenant.getCardType()];
            deedTenantOffer.setCardType(cardType);
            deedTenantOffer.setMintingPower(cardType.getMintingPower());
        }
        if (deedTenant.getCityIndex() >= 0) {
            deedTenantOffer.setCity(DeedCity.values()[deedTenant.getCityIndex()]);
        }
    }

    private DeedTenant getDeedTenant(long nftId, String managerAddress) {
        DeedTenant deedTenant;
        try {
            deedTenant = this.tenantService.getDeedTenantOrImport(managerAddress, Long.valueOf(nftId));
        }
        catch (UnauthorizedOperationException e) {
            deedTenant = this.tenantService.getDeedTenant(nftId);
        }
        catch (ObjectNotFoundException e) {
            throw new IllegalStateException("Unable to locate nft with id " + nftId, e);
        }
        return deedTenant;
    }

    private DeedTenantOffer getOfferByBlockchainOfferId(long blockchainOfferId, boolean updateIfNotFound) throws Exception {
        DeedTenantOffer offer = this.getParentOfferByBlockchainId(blockchainOfferId);
        if (offer == null) {
            if (updateIfNotFound) {
                DeedOfferBlockchainState blockchainOffer = this.blockchainService.getOfferById(BigInteger.valueOf(blockchainOfferId), null, null);
                if (blockchainOffer == null) {
                    return null;
                }
                return this.updateOfferFromBlockchain(blockchainOffer, false);
            }
            return null;
        }
        return offer;
    }

    private DeedTenantOffer getParentOfferByBlockchainId(long blockchainOfferId) {
        List<DeedTenantOffer> offers = this.offerRepository.findByOfferIdAndParentIdIsNull(blockchainOfferId);
        if (CollectionUtils.isEmpty(offers)) {
            return null;
        }
        if (offers.size() > 1) {
            LOG.warn("It seems that we have more than one parent offer with id {}, this is likely a bug! Retrieve first one only. List = {}", (Object)blockchainOfferId, offers);
        }
        return offers.get(0);
    }

    private boolean isOfferDeleted(DeedOfferBlockchainState blockchainOffer) {
        return blockchainOffer.getId() == null || blockchainOffer.getId().longValue() == 0L || blockchainOffer.getDeedId() == null || blockchainOffer.getDeedId().longValue() == 0L;
    }
}

