/*
 * Decompiled with CFR 0.152.
 */
package io.meeds.dapp.service;

import io.meeds.dapp.constant.Currency;
import io.meeds.dapp.elasticsearch.model.CurrencyExchangeRate;
import io.meeds.dapp.elasticsearch.model.MeedExchangeRate;
import io.meeds.dapp.model.MeedPrice;
import io.meeds.dapp.storage.CurrencyExchangeRateRepository;
import io.meeds.dapp.storage.MeedExchangeRateRepository;
import jakarta.json.Json;
import jakarta.json.JsonObject;
import jakarta.json.JsonReader;
import java.io.DataOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.Reader;
import java.io.StringReader;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.net.URL;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.time.LocalDate;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.function.BiPredicate;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.net.ssl.HttpsURLConnection;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

@Component
public class ExchangeService {
    private static final Logger LOG = LoggerFactory.getLogger(ExchangeService.class);
    private static final LocalDate MEEDS_TOKEN_FIRST_DATE = LocalDate.of(2021, 11, 6);
    private static final Pattern LATEST_BLOCK_NUMBER_ERROR_PATTERN = Pattern.compile(".+ up to block number (\\d+) .+");
    private static final String PAIR_PARAM_NAME = "pair";
    private static final String MESSAGE_PARAM_NAME = "message";
    private static final String BUNDLE_PARAM_NAME = "bundle";
    private static final String DATA_PARAM_NAME = "data";
    private static final String BLOCKS_PARAM_NAME = "blocks";
    private static final String ERRORS_PARAM_NAME = "errors";
    @Value(value="${meeds.exchange.currencyApiKey:}")
    private String currencyApiKey;
    @Value(value="${meeds.exchange.currencyApiUrl:https://api.currencyapi.com/v3/latest}")
    private String currencyApiUrl;
    @Value(value="${meeds.exchange.blockchainApiUrl:https://api.thegraph.com/subgraphs/name/blocklytics/ethereum-blocks}")
    private String blockchainApiUrl;
    @Value(value="${meeds.exchange.lpTokenApiUrl:https://api.thegraph.com/subgraphs/name/sushiswap/exchange}")
    private String lpTokenApiUrl;
    @Value(value="${meeds.exchange.lpTokenAddress:0x960bd61d0b960b107ff5309a2dcced4705567070}")
    private String lpTokenAddress;
    @Autowired
    private CurrencyExchangeRateRepository currencyExchangeRateRepository;
    @Autowired
    private MeedExchangeRateRepository meedExchangeRateRepository;

    public List<MeedPrice> getExchangeRates(Currency currency, LocalDate fromDate, LocalDate toDate) {
        if (fromDate == null) {
            throw new IllegalArgumentException("fromDate mandatory");
        }
        if (toDate == null) {
            throw new IllegalArgumentException("toDate mandatory");
        }
        if (toDate.isBefore(fromDate)) {
            throw new IllegalArgumentException("toDate must be after mandatory");
        }
        List<MeedExchangeRate> exchangeRates = this.meedExchangeRateRepository.findByDateBetween(fromDate, toDate);
        if (exchangeRates == null || exchangeRates.isEmpty()) {
            return Collections.emptyList();
        }
        List<Object> currencyExchangeRates = currency != null && currency != Currency.USD && currency != Currency.ETH ? this.currencyExchangeRateRepository.findByCurrencyAndDateBetween(currency, fromDate, toDate) : Collections.emptyList();
        return exchangeRates.stream().map(exchangeRate -> this.toMeedPrice((MeedExchangeRate)exchangeRate, (List<CurrencyExchangeRate>)currencyExchangeRates, currency)).toList();
    }

    public BigDecimal getExchangeRate(Currency currency) {
        if (currency == null || currency == Currency.USD) {
            return BigDecimal.ONE;
        }
        if (currency == Currency.ETH) {
            List<MeedExchangeRate> meedExchangeRates = this.meedExchangeRateRepository.findByDateBetween(LocalDate.now().minusDays(2L), LocalDate.now());
            if (CollectionUtils.isEmpty(meedExchangeRates)) {
                return BigDecimal.ZERO;
            }
            return meedExchangeRates.stream().max(Comparator.comparing(MeedExchangeRate::getDate)).filter(rate -> rate.getEthUsdPrice() != null && !BigDecimal.ZERO.equals(rate.getEthUsdPrice())).map(rate -> BigDecimal.ONE.divide(rate.getEthUsdPrice(), 18, RoundingMode.HALF_UP)).orElse(BigDecimal.ZERO);
        }
        List<CurrencyExchangeRate> currencyExchangeRates = this.currencyExchangeRateRepository.findByCurrencyAndDateBetween(currency, LocalDate.now().minusDays(2L), LocalDate.now());
        if (CollectionUtils.isEmpty(currencyExchangeRates)) {
            return BigDecimal.ZERO;
        }
        return currencyExchangeRates.stream().max(Comparator.comparing(CurrencyExchangeRate::getDate)).map(CurrencyExchangeRate::getRate).orElse(BigDecimal.ZERO);
    }

    public void computeTodayCurrencyExchangeRate() {
        this.computeCurrencyExchangeRateOfDay(LocalDate.now(ZoneOffset.UTC));
    }

    public void computeMeedExchangeRate() {
        LocalDate indexDate = this.firstMeedTokenDate();
        LocalDate today = LocalDate.now(ZoneOffset.UTC);
        LocalDate untilDate = today.plusDays(1L);
        while (indexDate.isBefore(untilDate)) {
            MeedExchangeRate rate = this.meedExchangeRateRepository.findById(indexDate).orElse(new MeedExchangeRate(indexDate));
            if (!rate.isFinalRate()) {
                boolean todayValue = indexDate.isEqual(today);
                rate.setFinalRate(!todayValue);
                ZonedDateTime time = todayValue ? ZonedDateTime.now(ZoneOffset.UTC) : indexDate.plusDays(1L).atStartOfDay(ZoneOffset.UTC).minusSeconds(1L);
                this.computePrice(rate, time, todayValue, this::addEthPrice);
                this.computePrice(rate, time, todayValue, this::addPairData);
                this.meedExchangeRateRepository.save(rate);
            }
            indexDate = indexDate.plusDays(1L);
        }
    }

    public BigDecimal getMeedUsdPrice() {
        List<MeedPrice> meedExchangeRates = this.getExchangeRates(Currency.USD, LocalDate.now().minusDays(8L), LocalDate.now());
        return meedExchangeRates.stream().filter(object -> object.getDate() != null).max(Comparator.comparing(MeedPrice::getDate)).map(MeedPrice::getCurrencyPrice).orElse(BigDecimal.ZERO);
    }

    protected void computeCurrencyExchangeRate() {
        LocalDate today = LocalDate.now(ZoneOffset.UTC);
        CurrencyExchangeRate todayExchangeRate = this.currencyExchangeRateRepository.findById(today).orElse(null);
        if (todayExchangeRate == null && (todayExchangeRate = this.computeCurrencyExchangeRateOfDay(today)) == null) {
            LOG.warn("Can't find Currency Exchange rate for today, give up update rates");
            return;
        }
        LocalDate indexDate = this.firstMeedTokenDate();
        LocalDate untilDate = LocalDate.now(ZoneOffset.UTC).plusDays(1L);
        while (indexDate.isBefore(untilDate)) {
            CurrencyExchangeRate rate = this.currencyExchangeRateRepository.findById(indexDate).orElse(null);
            if (rate == null) {
                rate = new CurrencyExchangeRate(indexDate, todayExchangeRate.getCurrency(), todayExchangeRate.getRate());
                this.currencyExchangeRateRepository.save(rate);
            }
            indexDate = indexDate.plusDays(1L);
        }
    }

    private CurrencyExchangeRate computeCurrencyExchangeRateOfDay(LocalDate date) {
        CurrencyExchangeRate rate = null;
        String exchangeRateResultString = this.retrieveCurrencyExchangeRate();
        if (StringUtils.isNotBlank((CharSequence)exchangeRateResultString)) {
            try (JsonReader reader = Json.createReader((Reader)new StringReader(exchangeRateResultString));){
                JsonObject exchangeRateResult = reader.readObject();
                BigDecimal euroExchangeRate = exchangeRateResult.getJsonObject(DATA_PARAM_NAME).getJsonObject("EUR").getJsonNumber("value").bigDecimalValue();
                rate = new CurrencyExchangeRate(date, Currency.EUR, euroExchangeRate);
                this.currencyExchangeRateRepository.save(rate);
            }
        }
        return rate;
    }

    private void computePrice(MeedExchangeRate rate, ZonedDateTime date, boolean todayValue, BiPredicate<MeedExchangeRate, String> computeFunction) {
        String blockNumber = this.getBlockNumber(date, todayValue);
        boolean added = computeFunction.test(rate, blockNumber);
        if (!added && todayValue) {
            int previousBlock = Integer.parseInt(blockNumber) - 10;
            added = computeFunction.test(rate, String.valueOf(previousBlock));
        }
        if (!added) {
            if (LOG.isWarnEnabled()) {
                LOG.warn("Error computing Eth Price for date {}. Retrieve empty result.", (Object)date.toLocalDate());
            }
            rate.setFinalRate(false);
        }
    }

    private boolean addEthPrice(MeedExchangeRate rate, String blockNumber) {
        String body = "{\"query\":\"{bundle(id: 1, block:{number:" + blockNumber + "}){ethPrice}}\"}";
        String ethPriceDataJsonString = this.executeQuery(this.lpTokenApiUrl, body);
        if (StringUtils.isNotBlank((CharSequence)ethPriceDataJsonString)) {
            try (JsonReader reader = Json.createReader((Reader)new StringReader(ethPriceDataJsonString));){
                JsonObject ethPriceDataJson = reader.readObject();
                if (ethPriceDataJson.containsKey((Object)DATA_PARAM_NAME) && ethPriceDataJson.getJsonObject(DATA_PARAM_NAME).containsKey((Object)BUNDLE_PARAM_NAME) && !ethPriceDataJson.getJsonObject(DATA_PARAM_NAME).isNull(BUNDLE_PARAM_NAME)) {
                    JsonObject bundleObject = ethPriceDataJson.getJsonObject(DATA_PARAM_NAME).getJsonObject(BUNDLE_PARAM_NAME);
                    BigDecimal ethPrice = new BigDecimal(bundleObject.getJsonString("ethPrice").getString());
                    rate.setEthUsdPrice(ethPrice);
                    boolean bl = true;
                    return bl;
                }
                String lastKnownBlock = this.getLastBlockFromErrorMessage(ethPriceDataJson);
                if (StringUtils.isNotBlank((CharSequence)lastKnownBlock)) {
                    boolean bl = this.addEthPrice(rate, lastKnownBlock);
                    return bl;
                }
            }
        }
        return false;
    }

    private boolean addPairData(MeedExchangeRate rate, String blockNumber) {
        String body;
        String pairDataJsonString;
        if (StringUtils.isNotBlank((CharSequence)blockNumber) && StringUtils.isNotBlank((CharSequence)(pairDataJsonString = this.executeQuery(this.lpTokenApiUrl, body = "{\"query\":\"{pair(id: \\\"" + this.lpTokenAddress + "\\\", block: {number:" + blockNumber + "}) {id,token1Price,reserve0,reserve1}}\"}")))) {
            try (JsonReader reader = Json.createReader((Reader)new StringReader(pairDataJsonString));){
                JsonObject pairDataJson = reader.readObject();
                if (this.hasPairParam(pairDataJson)) {
                    JsonObject pairJsonObject = pairDataJson.getJsonObject(DATA_PARAM_NAME).getJsonObject(PAIR_PARAM_NAME);
                    BigDecimal meedEthPrice = new BigDecimal(pairJsonObject.getJsonString("token1Price").getString());
                    rate.setMeedEthPrice(meedEthPrice);
                    BigDecimal meedsReserve = new BigDecimal(pairJsonObject.getJsonString("reserve0").getString());
                    rate.setMeedReserve(meedsReserve);
                    BigDecimal ethReserve = new BigDecimal(pairJsonObject.getJsonString("reserve1").getString());
                    rate.setEthReserve(ethReserve);
                    boolean bl = true;
                    return bl;
                }
                String lastKnownBlock = this.getLastBlockFromErrorMessage(pairDataJson);
                if (StringUtils.isNotBlank((CharSequence)lastKnownBlock)) {
                    boolean bl = this.addPairData(rate, lastKnownBlock);
                    return bl;
                }
            }
        }
        return false;
    }

    private String getLastBlockFromErrorMessage(JsonObject pairDataJson) {
        Matcher matcher;
        String message;
        if (pairDataJson.containsKey((Object)ERRORS_PARAM_NAME) && !pairDataJson.isNull(ERRORS_PARAM_NAME) && !pairDataJson.getJsonArray(ERRORS_PARAM_NAME).isEmpty() && pairDataJson.getJsonArray(ERRORS_PARAM_NAME).getJsonObject(0).containsKey((Object)MESSAGE_PARAM_NAME) && !pairDataJson.getJsonArray(ERRORS_PARAM_NAME).getJsonObject(0).isNull(MESSAGE_PARAM_NAME) && (message = pairDataJson.getJsonArray(ERRORS_PARAM_NAME).getJsonObject(0).getString(MESSAGE_PARAM_NAME)).contains("up to block number") && (matcher = LATEST_BLOCK_NUMBER_ERROR_PATTERN.matcher(message)).find()) {
            String blockNumber = matcher.group(1);
            LOG.debug("{}. Attempt to retrieve data with block number {}", (Object)message, (Object)blockNumber);
            return blockNumber;
        }
        return null;
    }

    private String getBlockNumber(ZonedDateTime date, boolean lastBlock) {
        Object body = lastBlock ? "{\"query\":\"{blocks(first:1,orderBy:number,orderDirection:desc){number}}\"}" : "{\"query\":\"{blocks(first:1,orderBy:timestamp,orderDirection: desc,where:{timestamp_lte:" + date.toEpochSecond() + "}){number}}\"}";
        String blockNumberDataJsonString = this.executeQuery(this.blockchainApiUrl, (String)body);
        if (StringUtils.isNotBlank((CharSequence)blockNumberDataJsonString)) {
            try (JsonReader reader = Json.createReader((Reader)new StringReader(blockNumberDataJsonString));){
                JsonObject blockNumberDataJson = reader.readObject();
                if (blockNumberDataJson.containsKey((Object)DATA_PARAM_NAME) && blockNumberDataJson.getJsonObject(DATA_PARAM_NAME).containsKey((Object)BLOCKS_PARAM_NAME) && !blockNumberDataJson.getJsonObject(DATA_PARAM_NAME).isNull(BLOCKS_PARAM_NAME)) {
                    String string = blockNumberDataJson.getJsonObject(DATA_PARAM_NAME).getJsonArray(BLOCKS_PARAM_NAME).getJsonObject(0).getString("number");
                    return string;
                }
            }
        }
        return null;
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    protected String executeQuery(String url, String body) {
        String string;
        int responseCode;
        HttpsURLConnection con = this.newURLConnection(url);
        con.setRequestMethod("GET");
        con.setRequestProperty("Content-Type", "application/json");
        if (StringUtils.isNotBlank((CharSequence)body)) {
            con.setDoOutput(true);
            DataOutputStream wr = new DataOutputStream(con.getOutputStream());
            wr.writeBytes(body);
            wr.flush();
            wr.close();
        }
        if ((responseCode = con.getResponseCode()) != 200) return null;
        InputStream inputStream = con.getInputStream();
        try {
            string = IOUtils.toString((InputStream)inputStream, (Charset)StandardCharsets.UTF_8);
            if (inputStream == null) return string;
        }
        catch (Throwable throwable) {
            try {
                if (inputStream == null) throw throwable;
                try {
                    inputStream.close();
                    throw throwable;
                }
                catch (Throwable throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
            catch (Exception e) {
                LOG.warn("An error occurred while retrieving data from URL '{}' with data '{}'", new Object[]{url, body, e});
                return null;
            }
        }
        inputStream.close();
        return string;
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    private String retrieveCurrencyExchangeRate() {
        String string;
        if (StringUtils.isBlank((CharSequence)this.currencyApiKey)) {
            throw new IllegalStateException("API Key is mandatory");
        }
        HttpsURLConnection con = this.newURLConnection(this.getCurrencyApiUrl());
        int responseCode = con.getResponseCode();
        if (responseCode != 200) return null;
        InputStream inputStream = con.getInputStream();
        try {
            string = IOUtils.toString((InputStream)con.getInputStream(), (Charset)StandardCharsets.UTF_8);
            if (inputStream == null) return string;
        }
        catch (Throwable throwable) {
            try {
                if (inputStream == null) throw throwable;
                try {
                    inputStream.close();
                    throw throwable;
                }
                catch (Throwable throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
            catch (Exception e) {
                LOG.warn("An error occurred while retrieving EURO exchange rate", (Throwable)e);
                return null;
            }
        }
        inputStream.close();
        return string;
    }

    private String getCurrencyApiUrl() {
        return this.currencyApiUrl + "?apikey=" + this.currencyApiKey + "&base_currency=USD&currencies=EUR";
    }

    private MeedPrice toMeedPrice(MeedExchangeRate exchangeRate, List<CurrencyExchangeRate> currencyExchangeRates, Currency currency) {
        LocalDate date = exchangeRate.getDate();
        BigDecimal ethPrice = exchangeRate.getMeedEthPrice();
        BigDecimal ethUsdPrice = exchangeRate.getEthUsdPrice();
        BigDecimal currencyPrice = this.getMeedPriceInCurrency(ethPrice, ethUsdPrice, currency, currencyExchangeRates, date);
        return new MeedPrice(date, ethPrice, currencyPrice);
    }

    private BigDecimal getMeedPriceInCurrency(BigDecimal meedEthPrice, BigDecimal ethUsdPrice, Currency currency, List<CurrencyExchangeRate> currencyExchangeRates, LocalDate date) {
        if (meedEthPrice == null) {
            return BigDecimal.ZERO;
        }
        BigDecimal currencyPrice = meedEthPrice;
        if (currency == Currency.USD) {
            currencyPrice = currencyPrice.multiply(ethUsdPrice);
        } else if (currency != Currency.ETH) {
            BigDecimal currencyRate = currencyExchangeRates.stream().filter(rate -> rate.getDate().isEqual(date)).findAny().map(CurrencyExchangeRate::getRate).orElse(BigDecimal.valueOf(1L));
            currencyPrice = currencyPrice.multiply(ethUsdPrice).multiply(currencyRate);
        }
        return currencyPrice;
    }

    private boolean hasPairParam(JsonObject pairDataJson) {
        return pairDataJson.containsKey((Object)DATA_PARAM_NAME) && pairDataJson.getJsonObject(DATA_PARAM_NAME).containsKey((Object)PAIR_PARAM_NAME) && !pairDataJson.getJsonObject(DATA_PARAM_NAME).isNull(PAIR_PARAM_NAME);
    }

    protected HttpsURLConnection newURLConnection(String url) throws IOException {
        return (HttpsURLConnection)new URL(url).openConnection();
    }

    protected LocalDate firstMeedTokenDate() {
        return MEEDS_TOKEN_FIRST_DATE;
    }
}

