/*
 * This file is part of the Meeds project (https://meeds.io/).
 * Copyright (C) 2020 - 2022 Meeds Association contact@meeds.io
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
package io.meeds.tenant.provisioning.scheduling.task;

import java.util.List;
import java.util.concurrent.TimeUnit;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Component;

import io.meeds.deeds.constant.ObjectNotFoundException;
import io.meeds.deeds.elasticsearch.model.DeedTenant;
import io.meeds.deeds.service.BlockchainService;
import io.meeds.tenant.provisioning.service.TenantProvisioningService;

@Component
public class ProvisioningBlockchainTransactionCheckTask {

  private static final Logger       LOG = LoggerFactory.getLogger(ProvisioningBlockchainTransactionCheckTask.class);

  @Autowired
  private TenantProvisioningService tenantProvisioningService;

  @Autowired
  private BlockchainService         blockchainService;

  @Scheduled(fixedDelay = 1, timeUnit = TimeUnit.MINUTES, initialDelay = 3)
  public synchronized void checkPendingTenants() {
    List<DeedTenant> pendingDeedTenants = tenantProvisioningService.getPendingTransactions();
    int pendingDeedTenantsSize = pendingDeedTenants.size();
    if (pendingDeedTenantsSize == 0) {
      return;
    }
    LOG.info("Check {} pending transactions for Deed Tenant Provisioning.", pendingDeedTenantsSize);
    pendingDeedTenants.forEach(deedTenant -> {
      String transactionHash = null;
      try {
        long nftId = deedTenant.getNftId();
        if (StringUtils.isNotBlank(deedTenant.getStartupTransactionHash())) {
          transactionHash = deedTenant.getStartupTransactionHash();
          LOG.info("Check transaction {} status of Deed Tenant Provisioning (Deploy) with id {}.", transactionHash, nftId);
          sendCommandFromTransaction(deedTenant, transactionHash, nftId, true);
        } else if (StringUtils.isNotBlank(deedTenant.getShutdownTransactionHash())) {
          transactionHash = deedTenant.getShutdownTransactionHash();
          LOG.info("Check transaction {} status of Deed Tenant Provisioning (Undeploy) with id {}.", transactionHash, nftId);
          sendCommandFromTransaction(deedTenant, transactionHash, nftId, false);
        }
      } catch (Exception e) {
        LOG.warn("Can't check Provisioning status of transaction {}", transactionHash, e);
      }
    });
  }

  private void sendCommandFromTransaction(DeedTenant deedTenant, String transactionHash, long nftId, boolean start) throws ObjectNotFoundException {
    if (tenantProvisioningService.isTransactionAlreadyConfirmed(transactionHash, nftId)) {
      LOG.info("The mined transaction [{}] for nftId='{}' is already confirmed with a created [TASK]. Ignore adding a new task for it.",
               transactionHash,
               nftId);
    } else {
      checkTransactionAndSendProvisioningCommand(deedTenant, transactionHash, start);
    }
  }

  private void checkTransactionAndSendProvisioningCommand(DeedTenant deedTenant, String transactionHash, boolean start) throws ObjectNotFoundException {
    if (blockchainService.isTransactionMined(transactionHash)) {
      boolean transactionConfirmed = blockchainService.isTransactionConfirmed(transactionHash);
      LOG.info("Deed Tenant with id {} Transaction Provisioning mined. Status = {}",
               deedTenant.getNftId(),
               transactionConfirmed ? "SUCCESS" : "FAIL");
      if (transactionConfirmed) {
        if (start) {
          tenantProvisioningService.startTenant(deedTenant.getNftId(), transactionHash, true);
        } else {
          tenantProvisioningService.stopTenant(deedTenant.getNftId(), transactionHash, true);
        }
      } else {
        tenantProvisioningService.cancelTenantProvisioning(deedTenant.getNftId(), transactionHash);
      }
    } else {
      LOG.info("Deed Tenant with id {} Transaction Provisioning always in progress", deedTenant.getNftId());
    }
  }

}
