/*
 * This file is part of the Meeds project (https://meeds.io/).
 * Copyright (C) 2020 - 2022 Meeds Association contact@meeds.io
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
package io.meeds.tenant.provisioning.service;

import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertNull;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.ArgumentMatchers.argThat;
import static org.mockito.ArgumentMatchers.eq;
import static org.mockito.Mockito.atLeast;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

import java.util.Collections;
import java.util.List;

import org.junit.jupiter.api.Test;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.boot.test.mock.mockito.MockBean;

import io.meeds.deeds.constant.ObjectNotFoundException;
import io.meeds.deeds.constant.TenantProvisioningStatus;
import io.meeds.deeds.elasticsearch.model.DeedTenant;
import io.meeds.deeds.service.BlockchainService;
import io.meeds.deeds.service.ListenerService;
import io.meeds.deeds.storage.DeedTenantManagerRepository;

@SpringBootTest(classes = {
    TenantProvisioningService.class,
})
class TenantProvisioningServiceTest {

  @MockBean
  private DeedTenantManagerRepository deedTenantManagerRepository;

  @MockBean
  private TaskIntegrationService      taskIntegrationService;

  @MockBean
  private BlockchainService           blockchainService;

  @MockBean
  private ListenerService             listenerService;

  @Autowired
  private TenantProvisioningService   tenantProvisioningService;

  @Test
  void testGetTenantsWithIncludeCompleted() {
    TenantProvisioningStatus provisioningStatus = TenantProvisioningStatus.START_CONFIRMED;
    int limit = 5;
    DeedTenant deedTenant = new DeedTenant();
    deedTenant.setNftId(5l);
    List<DeedTenant> deedTenants = Collections.singletonList(deedTenant);

    when(deedTenantManagerRepository.findByTenantProvisioningStatus(eq(provisioningStatus),
                                                                    argThat(matcher -> matcher.getPageSize() == limit
                                                                        && matcher.getPageNumber() == 0))).thenReturn(deedTenants);
    List<DeedTenant> tenants = tenantProvisioningService.getTenants(provisioningStatus, true, limit);
    assertEquals(deedTenants, tenants);
  }

  @Test
  void testGetTenantsWithoutCompleted() {
    TenantProvisioningStatus provisioningStatus = TenantProvisioningStatus.START_CONFIRMED;
    int limit = 5;
    DeedTenant deedTenant = new DeedTenant();
    deedTenant.setNftId(5l);
    List<DeedTenant> deedTenants = Collections.singletonList(deedTenant);

    when(deedTenantManagerRepository.findByTenantProvisioningStatusAndCompletedIsFalse(eq(provisioningStatus),
                                                                                       argThat(matcher -> matcher.getPageSize() == limit
                                                                                           && matcher.getPageNumber() == 0))).thenReturn(deedTenants);
    List<DeedTenant> tenants = tenantProvisioningService.getTenants(provisioningStatus, false, limit);
    assertEquals(deedTenants, tenants);
  }

  @Test
  void testStartTenantWhenConfirmed() throws ObjectNotFoundException {
    long nftId = 5l;
    boolean confirmed = true;
    String transactionHash = "transactionHash";
    tenantProvisioningService.startTenant(nftId, transactionHash, confirmed);
    verify(deedTenantManagerRepository, atLeast(1)).save(argThat(deedTenant -> {
      assertEquals(nftId, deedTenant.getNftId());
      assertEquals(transactionHash, deedTenant.getStartupTransactionHash());
      assertNull(deedTenant.getShutdownTransactionHash());
      return true;
    }));
    verify(taskIntegrationService, times(1)).saveTask(any(DeedTenant.class), eq(transactionHash), eq(true), eq(confirmed));
  }

  @Test
  void testStopTenantWhenConfirmed() throws ObjectNotFoundException {
    long nftId = 5l;
    boolean confirmed = true;
    String transactionHash = "transactionHash";
    tenantProvisioningService.stopTenant(nftId, transactionHash, confirmed);
    verify(deedTenantManagerRepository, atLeast(1)).save(argThat(deedTenant -> {
      assertEquals(nftId, deedTenant.getNftId());
      assertEquals(transactionHash, deedTenant.getShutdownTransactionHash());
      assertNull(deedTenant.getStartupTransactionHash());
      return true;
    }));
    verify(taskIntegrationService, times(1)).saveTask(any(DeedTenant.class), eq(transactionHash), eq(false), eq(confirmed));
  }

}
