/*
 * This file is part of the Meeds project (https://meeds.io/).
 * Copyright (C) 2020 - 2022 Meeds Association contact@meeds.io
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
package io.meeds.tenant.provisioning.listener;

import static io.meeds.deeds.constant.CommonConstants.TENANT_COMMAND_START_EVENT;
import static io.meeds.deeds.constant.CommonConstants.TENANT_COMMAND_STOP_EVENT;

import java.util.Arrays;
import java.util.List;

import org.apache.commons.codec.binary.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import io.meeds.deeds.elasticsearch.model.DeedTenant;
import io.meeds.deeds.listener.EventListener;
import io.meeds.tenant.provisioning.service.TenantProvisioningService;

/**
 * A listener that will be triggered once
 */
@Service
public class TenantCommandListener implements EventListener<DeedTenant> {

  private static final Logger       LOG              = LoggerFactory.getLogger(TenantCommandListener.class);

  public static final String        LISTENER_NAME    = "TenantCommandListener";

  private static final List<String> SUPPORTED_EVENTS = Arrays.asList(TENANT_COMMAND_START_EVENT, TENANT_COMMAND_STOP_EVENT);

  @Autowired
  private TenantProvisioningService tenantProvisioningService;

  @Override
  public String getName() {
    return LISTENER_NAME;
  }

  @Override
  public List<String> getSupportedEvents() {
    return SUPPORTED_EVENTS;
  }

  @Override
  public void onEvent(String eventName, DeedTenant deedTenant) {
    LOG.info("A commad [{}] has been [RECEIVED] for deed {}", eventName, deedTenant.getNftId());
    try {
      if (StringUtils.equals(TENANT_COMMAND_START_EVENT, eventName)) {
        tenantProvisioningService.startTenant(deedTenant.getNftId(), deedTenant.getStartupTransactionHash(), false);
      } else {
        tenantProvisioningService.stopTenant(deedTenant.getNftId(), deedTenant.getShutdownTransactionHash(), false);
      }
    } catch (Exception e) {
      LOG.warn("Error commanding [{}] Deed Tenant [{}]", eventName, deedTenant, e);
    }
  }

}
