/*
 * This file is part of the Meeds project (https://meeds.io/).
 * Copyright (C) 2020 - 2022 Meeds Association contact@meeds.io
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
package io.meeds.tenant.provisioning.listener;

import static io.meeds.deeds.constant.CommonConstants.DEED_OWNERSHIP_TRANSFERRED_BLOCKCHAIN_EVENT;
import static io.meeds.deeds.constant.CommonConstants.DEED_TENANT_OWNERSHIP_TRANSFERRED_EVENT;

import java.util.Collections;
import java.util.List;

import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import io.meeds.deeds.constant.CommonConstants.DeedOwnershipTransferEvent;
import io.meeds.deeds.elasticsearch.model.DeedTenant;
import io.meeds.deeds.listener.EventListener;
import io.meeds.deeds.service.ListenerService;
import io.meeds.deeds.service.TenantService;

@Service
public class DeedOwnershipTransferredListener implements EventListener<DeedOwnershipTransferEvent> {

  public static final String        LISTENER_NAME    = "DeedOwnershipTransferredProvisioningListener";

  private static final List<String> SUPPORTED_EVENTS = Collections.singletonList(DEED_OWNERSHIP_TRANSFERRED_BLOCKCHAIN_EVENT);

  @Autowired
  private TenantService             tenantService;

  @Autowired
  private ListenerService           listenerService;

  @Override
  public String getName() {
    return LISTENER_NAME;
  }

  @Override
  public List<String> getSupportedEvents() {
    return SUPPORTED_EVENTS;
  }

  @Override
  public void onEvent(String eventName, DeedOwnershipTransferEvent ownershipTransferEvent) {
    String oldOwner = ownershipTransferEvent.getFrom();
    String newOwner = ownershipTransferEvent.getTo();
    if (StringUtils.equalsIgnoreCase(oldOwner, newOwner)) {
      return;
    }
    DeedTenant deedTenant = tenantService.getDeedTenant(ownershipTransferEvent.getNftId());
    if (deedTenant != null) {
      setNewOwner(oldOwner, newOwner, deedTenant);
      setNewDeedManager(oldOwner, newOwner, deedTenant);
      tenantService.saveDeedTenant(deedTenant);

      listenerService.publishEvent(DEED_TENANT_OWNERSHIP_TRANSFERRED_EVENT, ownershipTransferEvent);
    }
  }

  private void setNewDeedManager(String oldOwner, String newOwner, DeedTenant deedTenant) {
    String deedManagerAddress = deedTenant.getManagerAddress();
    if (StringUtils.isBlank(deedManagerAddress) || StringUtils.equalsIgnoreCase(deedManagerAddress, oldOwner)) {
      deedTenant.setManagerAddress(newOwner);
    }
  }

  private void setNewOwner(String oldOwner, String newOwner, DeedTenant deedTenant) {
    String deedOwnerAddress = deedTenant.getOwnerAddress();
    if (StringUtils.isBlank(deedOwnerAddress) || StringUtils.equalsIgnoreCase(deedOwnerAddress, oldOwner)) {
      deedTenant.setOwnerAddress(newOwner);
    }
  }

}
