/*
 * This file is part of the Meeds project (https://meeds.io/).
 * Copyright (C) 2020 - 2022 Meeds Association contact@meeds.io
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
package io.meeds.tenant.provisioning.listener;

import static io.meeds.tenant.provisioning.service.TenantProvisioningService.DEED_EVENT_TENANT_CANCEL_COMMAND;
import static io.meeds.tenant.provisioning.service.TenantProvisioningService.DEED_EVENT_TENANT_START_COMMAND_CREATED;
import static io.meeds.tenant.provisioning.service.TenantProvisioningService.DEED_EVENT_TENANT_STOP_COMMAND_CREATED;

import java.util.Arrays;
import java.util.List;

import org.apache.commons.codec.binary.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import io.meeds.deeds.constant.ObjectNotFoundException;
import io.meeds.deeds.elasticsearch.model.DeedTenant;
import io.meeds.deeds.listener.EventListener;
import io.meeds.tenant.provisioning.model.AutomaticStatus;
import io.meeds.tenant.provisioning.service.DeedEmailService;

@Service
public class TenantStatusChangeListener implements EventListener<DeedTenant> {
  private static final Logger       LOG                        = LoggerFactory.getLogger(TenantStatusChangeListener.class);

  public static final String        LISTENER_NAME              = "TenantStatusChangeListener";

  private static final List<String> SUPPORTED_EVENTS           = Arrays.asList(DEED_EVENT_TENANT_CANCEL_COMMAND,
                                                                               DEED_EVENT_TENANT_START_COMMAND_CREATED,
                                                                               DEED_EVENT_TENANT_STOP_COMMAND_CREATED);

  @Autowired
  private DeedEmailService deedEmailService;

  @Override
  public String getName() {
    return LISTENER_NAME;
  }

  @Override
  public List<String> getSupportedEvents() {
    return SUPPORTED_EVENTS;
  }

  @Override
  public void onEvent(String eventName, DeedTenant deedTenant) {
    AutomaticStatus tenantStatus = null;
    if (StringUtils.equals(eventName, DEED_EVENT_TENANT_START_COMMAND_CREATED)) {
      tenantStatus = deedTenant.getTenantProvisioningStatus().isPending() ? AutomaticStatus.START_IN_PROGRESS : AutomaticStatus.START_CONFIRMED;
    } else if (StringUtils.equals(eventName, DEED_EVENT_TENANT_STOP_COMMAND_CREATED)) {
      tenantStatus = deedTenant.getTenantProvisioningStatus().isPending() ? AutomaticStatus.STOP_IN_PROGRESS : AutomaticStatus.STOP_CONFIRMED;
    } else if (StringUtils.equals(eventName, DEED_EVENT_TENANT_CANCEL_COMMAND)) {
      tenantStatus = AutomaticStatus.TRANSACTION_ERROR;
    }
    try {
      if (tenantStatus != null) {
        deedEmailService.sendEmail(deedTenant.getNftId(), tenantStatus);
      }
    } catch (ObjectNotFoundException e) {
      LOG.warn("Can't send automatic mail, reason: {}", e.getMessage());
    } catch (Exception e) {
      LOG.warn("Error sending automatic mail for event [{}] for Deed Tenant [{}]", eventName, deedTenant.getNftId(), e);
    }
  }

}
