/**
 * This file is part of the Meeds project (https://meeds.io/).
 *
 * Copyright (C) 2020 - 2025 Meeds Association contact@meeds.io
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
package org.exoplatform.social.core.jpa.search.listener;

import org.exoplatform.commons.search.index.IndexingService;
import org.exoplatform.commons.utils.CommonsUtils;
import org.exoplatform.commons.utils.IdentifierUtil;
import org.exoplatform.services.log.ExoLogger;
import org.exoplatform.services.log.Log;
import org.exoplatform.social.core.identity.model.Identity;
import org.exoplatform.social.core.jpa.search.ProfileIndexingServiceConnector;
import org.exoplatform.social.core.profile.ProfileLifeCycleEvent;
import org.exoplatform.social.core.profile.ProfileListenerPlugin;

public class ProfileESListenerImpl extends ProfileListenerPlugin {

  private static final Log LOG = ExoLogger.getLogger(ProfileESListenerImpl.class);

  @Override
  public void avatarUpdated(ProfileLifeCycleEvent event) {
    if (event.getProfile().getIdentity().isEnable()) {
      IndexingService indexingService = CommonsUtils.getService(IndexingService.class);
      String id = event.getProfile().getIdentity().getId();

      LOG.debug("Notifying indexing service for profile avatar update id={}", id);

      indexingService.reindex(ProfileIndexingServiceConnector.TYPE, id);
    } else {
      LOG.debug("Ignore notifying indexing service for profile avatar update for disabled id={}", event.getProfile().getIdentity().getId());
    }

  }

  @Override
  public void bannerUpdated(ProfileLifeCycleEvent event) {
    LOG.debug("Profile banner of user {} has been updated", event.getProfile().getIdentity().getId());
  }

  @Override
  public void contactSectionUpdated(ProfileLifeCycleEvent event) {
    if (event.getProfile().getIdentity().isEnable()) {
      IndexingService indexingService = CommonsUtils.getService(IndexingService.class);
      String id = event.getProfile().getIdentity().getId();

      LOG.debug("Notifying indexing service for the profile contact update id={}", id);

      indexingService.reindex(ProfileIndexingServiceConnector.TYPE, id);
    } else {
      LOG.debug("Ignore notifying indexing service for profile contact update for disabled id={}", event.getProfile().getIdentity().getId());
    }
  }

  @Override
  public void aboutMeUpdated(ProfileLifeCycleEvent event) {
    if (event.getProfile().getIdentity().isEnable()) {
      String id = event.getProfile().getIdentity().getId();

      LOG.debug("Notifying indexing service for the profile aboutme update id={}", id);

      CommonsUtils.getService(IndexingService.class).reindex(ProfileIndexingServiceConnector.TYPE, id);
    } else {
      LOG.debug("Ignore notifying indexing service for profile aboutme update for disabled id={}", event.getProfile().getIdentity().getId());
    }
  }

  @Override
  public void experienceSectionUpdated(ProfileLifeCycleEvent event) {
    if (event.getProfile().getIdentity().isEnable()) {
      String id = event.getProfile().getIdentity().getId();

      LOG.debug("Notifying indexing service for the profile experience update id={}", id);

      CommonsUtils.getService(IndexingService.class).reindex(ProfileIndexingServiceConnector.TYPE, id);
    } else {
      LOG.debug("Ignore notifying indexing service for profile experience update for disabled id={}", event.getProfile().getIdentity().getId());
    }
  }

  @Override
  public void createProfile(ProfileLifeCycleEvent event) {
    if (event.getProfile().getIdentity().isEnable()) {
      IndexingService indexingService = CommonsUtils.getService(IndexingService.class);
      String id = event.getProfile().getIdentity().getId();

      LOG.debug("Notifying indexing service for the profile creation id={}", id);

      indexingService.index(ProfileIndexingServiceConnector.TYPE, id);
    } else {
      LOG.debug("Ignore notifying indexing service for profile creation for disabled id={}", event.getProfile().getIdentity().getId());
    }
  }

  @Override
  public void technicalUpdated(ProfileLifeCycleEvent event) {
    if (event.getProfile().getIdentity().isEnable()) {
      IndexingService indexingService = CommonsUtils.getService(IndexingService.class);
      String id = event.getProfile().getIdentity().getId();

      LOG.debug("Notifying indexing service for the profile update id={}, technical property changed", id);

      indexingService.reindex(ProfileIndexingServiceConnector.TYPE, id);
    } else {
      LOG.debug("Ignore notifying indexing service for the profile update for disabled id={}, technical property changed", event.getProfile().getIdentity().getId());
    }
  }
}
