/**
 * This file is part of the Meeds project (https://meeds.io/).
 *
 * Copyright (C) 2020 - 2025 Meeds Association contact@meeds.io
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
package org.exoplatform.commons.notification.impl.jpa.email.entity;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.List;

import org.apache.commons.lang3.StringUtils;

import jakarta.persistence.Column;
import jakarta.persistence.Entity;
import jakarta.persistence.FetchType;
import jakarta.persistence.GeneratedValue;
import jakarta.persistence.GenerationType;
import jakarta.persistence.Id;
import jakarta.persistence.NamedQuery;
import jakarta.persistence.OneToMany;
import jakarta.persistence.SequenceGenerator;
import jakarta.persistence.Table;

@Entity(name = "NotificationsMailNotifEntity")
@Table(name = "NTF_EMAIL_NOTIFS")
@NamedQuery(name = "NotificationsMailNotifEntity.getNotifsByPluginAndDay", query = "SELECT distinct(m) FROM NotificationsMailNotifEntity m " +
    "JOIN m.digests d " +
    "LEFT OUTER JOIN FETCH m.parameters p " +
    "WHERE DAY(m.creationDate)= :day " +
    "AND MONTH(m.creationDate)= :month " +
    "AND YEAR(m.creationDate)= :year " +
    "AND m.type= :pluginId " +
    "AND d.type= 'daily' " +
    "ORDER BY m.order ASC, m.creationDate DESC")
@NamedQuery(name = "NotificationsMailNotifEntity.getNotifsByPluginAndWeek", query = "SELECT distinct(m) FROM NotificationsMailNotifEntity m " +
    "JOIN m.digests d " +
    "LEFT OUTER JOIN FETCH m.parameters p " +
    "WHERE m.creationDate> :date " +
    "AND m.type= :pluginId " +
    "AND d.type= 'weekly' " +
    "ORDER BY m.order ASC, m.creationDate DESC")
@NamedQuery(name = "NotificationsMailNotifEntity.getAllNotificationsWithoutDigests", query = "SELECT distinct(m) FROM NotificationsMailNotifEntity m " +
    "WHERE m.digests IS EMPTY")
public class MailNotifEntity {
  @Id
  @Column(name = "EMAIL_NOTIF_ID")
  @SequenceGenerator(name = "SEQ_NTF_EMAIL_NOTIF", sequenceName = "SEQ_NTF_EMAIL_NOTIF", allocationSize = 1)
  @GeneratedValue(strategy = GenerationType.AUTO, generator = "SEQ_NTF_EMAIL_NOTIF")
  private long                  id;

  @Column(name = "SENDER")
  private String                sender;

  @Column(name = "TYPE")
  private String                type;

  @Column(name = "CREATION_DATE")
  private Calendar              creationDate;

  @Column(name = "SENDING_ORDER")
  private int                   order;

  @OneToMany(fetch = FetchType.LAZY, mappedBy = "notification")
  private List<MailDigestEntity> digests = new ArrayList<>();

  @OneToMany(fetch = FetchType.LAZY, mappedBy = "notification")
  private List<MailParamEntity>  parameters = new ArrayList<>();

  public long getId() {
    return id;
  }

  public String getSender() {
    return sender;
  }

  public MailNotifEntity setSender(String sender) {
    if (StringUtils.isBlank(sender)) {
      this.sender = null;
    } else {
      this.sender = sender;
    }
    return this;
  }

  public String getType() {
    return type;
  }

  public MailNotifEntity setType(String type) {
    this.type = type;
    return this;
  }

  public Calendar getCreationDate() {
    return creationDate;
  }

  public MailNotifEntity setCreationDate(Calendar creationDate) {
    this.creationDate = creationDate;
    return this;
  }

  public int getOrder() {
    return order;
  }

  public MailNotifEntity setOrder(int order) {
    this.order = order;
    return this;
  }

  public List<MailParamEntity> getParameters() {
    return parameters;
  }

  public List<MailDigestEntity> getDigests() {
    return digests;
  }
}
