/*
 * Decompiled with CFR 0.152.
 */
package io.meeds.wallet.service;

import io.meeds.wallet.model.ContractDetail;
import io.meeds.wallet.model.TransactionDetail;
import io.meeds.wallet.model.TransactionStatistics;
import io.meeds.wallet.model.Wallet;
import io.meeds.wallet.model.WalletType;
import io.meeds.wallet.service.WalletAccountService;
import io.meeds.wallet.service.WalletContractService;
import io.meeds.wallet.service.WalletTransactionService;
import io.meeds.wallet.storage.TransactionStorage;
import io.meeds.wallet.utils.WalletUtils;
import java.time.LocalDate;
import java.time.Year;
import java.time.YearMonth;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.time.format.TextStyle;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.exoplatform.commons.utils.CommonsUtils;
import org.exoplatform.container.xml.InitParams;
import org.exoplatform.services.listener.ListenerService;
import org.exoplatform.services.log.ExoLogger;
import org.exoplatform.services.log.Log;
import org.exoplatform.social.core.space.spi.SpaceService;

public class WalletTransactionServiceImpl
implements WalletTransactionService {
    private static final Log LOG = ExoLogger.getLogger(WalletTransactionServiceImpl.class);
    private static final String YEAR_PERIODICITY = "year";
    private static final String MONTH_PERIODICITY = "month";
    private static final long DEFAULT_MAX_PARALLEL_PENDING_TRANSACTIONS = 1L;
    private static final long DEFAULT_MAX_SENDING_TRANSACTIONS_ATTEMPTS = 3L;
    private WalletAccountService accountService;
    private WalletContractService contractService;
    private TransactionStorage transactionStorage;
    private SpaceService spaceService;
    private ListenerService listenerService;
    private long pendingTransactionMaxDays;
    private long maxParallelPendingTransactions;
    private long maxAttemptsToSend;

    public WalletTransactionServiceImpl(WalletAccountService accountService, TransactionStorage transactionStorage, WalletContractService contractService, InitParams params) {
        this.transactionStorage = transactionStorage;
        this.accountService = accountService;
        this.contractService = contractService;
        if (params != null) {
            String value;
            if (params.containsKey((Object)"transaction.pending.maxDays")) {
                value = params.getValueParam("transaction.pending.maxDays").getValue();
                this.pendingTransactionMaxDays = Long.parseLong(value);
            }
            if (params.containsKey((Object)"transaction.pending.maxToSend")) {
                value = params.getValueParam("transaction.pending.maxToSend").getValue();
                this.maxParallelPendingTransactions = Long.parseLong(value);
            }
            if (params.containsKey((Object)"transaction.pending.maxSendingAttempts")) {
                value = params.getValueParam("transaction.pending.maxSendingAttempts").getValue();
                this.maxAttemptsToSend = Long.parseLong(value);
            }
        }
        if (this.maxParallelPendingTransactions <= 0L) {
            LOG.warn("Invalid value {} for parameter {}, using default value {}", new Object[]{this.maxParallelPendingTransactions, "transaction.pending.maxToSend", 1L});
            this.maxParallelPendingTransactions = 1L;
        }
        if (this.maxAttemptsToSend <= 0L) {
            LOG.warn("Invalid value {} for parameter {}, using default value {}", new Object[]{this.maxAttemptsToSend, "transaction.pending.maxSendingAttempts", 3L});
            this.maxAttemptsToSend = 3L;
        }
    }

    public List<TransactionDetail> getPendingWalletTransactionsNotSent(String address) {
        return this.transactionStorage.getPendingWalletTransactionsNotSent(address, WalletUtils.getNetworkId());
    }

    public List<TransactionDetail> getPendingWalletTransactionsSent(String address) {
        return this.transactionStorage.getPendingWalletTransactionsSent(address, WalletUtils.getNetworkId());
    }

    public List<TransactionDetail> getPendingEtherTransactions(String address) {
        return this.transactionStorage.getPendingEtherTransactions(address, WalletUtils.getNetworkId());
    }

    public long countContractPendingTransactionsSent() {
        return this.transactionStorage.countContractPendingTransactionsSent(WalletUtils.getNetworkId());
    }

    public long countContractPendingTransactionsToSend() {
        return this.transactionStorage.countContractPendingTransactionsToSend(WalletUtils.getNetworkId());
    }

    public List<TransactionDetail> getTransactions(String address, String contractAddress, String contractMethodName, String hash, int limit, boolean onlyPending, boolean administration, String currentUser) throws IllegalAccessException {
        if (administration && !WalletUtils.isUserRewardingAdmin((String)currentUser)) {
            throw new IllegalAccessException(currentUser + " user is not allowed to get administrative transactions");
        }
        if (this.contractService.isContract(address)) {
            if (WalletUtils.isUserRewardingAdmin((String)currentUser)) {
                return this.getContractTransactions(address, contractMethodName, limit, currentUser);
            }
            throw new IllegalAccessException(currentUser + " user is not allowed to get all contract transactions");
        }
        if (StringUtils.isNotBlank((CharSequence)address)) {
            return this.getWalletTransactions(address, contractAddress, contractMethodName, hash, limit, onlyPending, administration, currentUser);
        }
        if (administration) {
            return this.getTransactions(limit, currentUser);
        }
        throw new IllegalStateException(currentUser + " user is not allowed to get all contract transactions");
    }

    public TransactionStatistics getTransactionStatistics(String address, String periodicity, String selectedDate, Locale locale) {
        Locale userLocale;
        if (StringUtils.isBlank((CharSequence)address)) {
            throw new IllegalArgumentException("Wallet address is mandatory");
        }
        if (StringUtils.isBlank((CharSequence)periodicity)) {
            throw new IllegalArgumentException("Periodicity is mandatory");
        }
        TransactionStatistics transactionStatistics = new TransactionStatistics();
        List periodList = null;
        Locale locale2 = userLocale = locale == null ? Locale.getDefault() : locale;
        if (StringUtils.equalsIgnoreCase((CharSequence)periodicity, (CharSequence)YEAR_PERIODICITY)) {
            ArrayList<YearMonth> monthsList = new ArrayList<YearMonth>();
            String[] selectedDateParts = StringUtils.isBlank((CharSequence)selectedDate) ? null : selectedDate.split("-");
            Year selectedYear = selectedDateParts == null ? Year.now() : Year.of(Integer.parseInt(selectedDateParts[0]));
            for (int i = 1; i <= 12; ++i) {
                monthsList.add(YearMonth.of(selectedYear.getValue(), i));
            }
            transactionStatistics.setPeriodicityLabel(String.valueOf(selectedYear.getValue()));
            transactionStatistics.setLabels(monthsList.stream().map(month -> StringUtils.capitalize((String)month.getMonth().getDisplayName(TextStyle.FULL, userLocale))).collect(Collectors.toList()));
            periodList = monthsList.stream().map(yearMonth -> yearMonth.atDay(1)).collect(Collectors.toList());
        } else if (StringUtils.equalsIgnoreCase((CharSequence)periodicity, (CharSequence)MONTH_PERIODICITY)) {
            String[] selectedDateParts = StringUtils.isBlank((CharSequence)selectedDate) ? null : selectedDate.split("-");
            YearMonth selectedMonth = selectedDateParts == null ? YearMonth.now() : YearMonth.of(Integer.parseInt(selectedDateParts[0]), Integer.parseInt(selectedDateParts[1]));
            int maxDayOfMonth = selectedMonth.lengthOfMonth();
            List dayList = IntStream.rangeClosed(1, maxDayOfMonth).boxed().collect(Collectors.toList());
            String monthLabel = StringUtils.capitalize((String)selectedMonth.getMonth().getDisplayName(TextStyle.FULL, userLocale)) + " " + selectedMonth.getYear();
            transactionStatistics.setPeriodicityLabel(monthLabel);
            transactionStatistics.setLabels(dayList.stream().map(day -> String.format("%02d", day)).collect(Collectors.toList()));
            periodList = dayList.stream().map(selectedMonth::atDay).collect(Collectors.toList());
        } else {
            throw new IllegalArgumentException("Uknown periodicity parameter: " + periodicity);
        }
        for (LocalDate startDate : periodList) {
            ZonedDateTime startDateTime = startDate.atStartOfDay(ZoneId.systemDefault());
            ZonedDateTime endDateTime = this.getEndDate(startDate, periodicity);
            String contractAddress = WalletUtils.getContractAddress();
            double receivedContractAmount = this.transactionStorage.countReceivedContractAmount(contractAddress, address, startDateTime, endDateTime);
            transactionStatistics.getIncome().add(String.valueOf(receivedContractAmount));
            double sentContractAmount = this.transactionStorage.countSentContractAmount(contractAddress, address, startDateTime, endDateTime);
            transactionStatistics.getOutcome().add(String.valueOf(sentContractAmount));
        }
        return transactionStatistics;
    }

    public TransactionDetail getTransactionByHash(String hash, String currentUser) {
        TransactionDetail transactionDetail = this.transactionStorage.getTransactionByHash(hash);
        if (transactionDetail != null) {
            this.retrieveWalletsDetails(transactionDetail, currentUser);
        }
        return transactionDetail;
    }

    public long getNonce(String fromAddress) {
        if (this.transactionStorage.countPendingTransactionAsSender(WalletUtils.getNetworkId(), fromAddress) == 0L) {
            return 0L;
        }
        long maxUsedNonce = this.transactionStorage.getMaxUsedNonce(WalletUtils.getNetworkId(), fromAddress);
        return maxUsedNonce + 1L;
    }

    public long getNonce(String fromAddress, String currentUser) throws IllegalAccessException {
        Wallet wallet = this.accountService.getWalletByAddress(fromAddress);
        if (wallet == null || !this.accountService.isWalletOwner(wallet, currentUser)) {
            throw new IllegalAccessException("User '" + currentUser + "' is attempting to access last transaction nonce for wallet of user " + String.valueOf(wallet));
        }
        return this.getNonce(fromAddress);
    }

    public TransactionDetail getTransactionByHash(String hash) {
        TransactionDetail transactionDetail = this.transactionStorage.getTransactionByHash(hash);
        if (transactionDetail != null) {
            this.retrieveWalletsDetails(transactionDetail);
        }
        return transactionDetail;
    }

    public TransactionDetail getPendingTransactionByHash(String hash) {
        TransactionDetail transactionDetail = this.transactionStorage.getPendingTransactionByHash(hash);
        if (transactionDetail != null) {
            this.retrieveWalletsDetails(transactionDetail);
        }
        return transactionDetail;
    }

    public void saveTransactionDetail(TransactionDetail transactionDetail, boolean broadcastMinedTransaction) {
        this.transactionStorage.saveTransactionDetail(transactionDetail);
        if (broadcastMinedTransaction) {
            this.broadcastTransactionMinedEvent(transactionDetail);
        }
    }

    public long countPendingTransactionsWithSameNonce(String transactionHash, String fromAddress, long nonce) {
        return this.transactionStorage.countPendingTransactionsWithSameNonce(WalletUtils.getNetworkId(), transactionHash, fromAddress, nonce);
    }

    public void cancelTransactionsWithSameNonce(TransactionDetail replacingTransaction) {
        List<TransactionDetail> transactions = this.transactionStorage.getPendingTransactionsWithSameNonce(replacingTransaction.getNetworkId(), replacingTransaction.getHash(), replacingTransaction.getFrom(), replacingTransaction.getNonce());
        if (CollectionUtils.isNotEmpty(transactions)) {
            transactions.forEach(replacedTransaction -> {
                replacedTransaction.setDropped(true);
                replacedTransaction.setPending(false);
                replacedTransaction.setNonce(0L);
                this.saveTransactionDetail((TransactionDetail)replacedTransaction, true);
                WalletUtils.broadcastTransactionReplacedEvent((TransactionDetail)replacedTransaction, (TransactionDetail)replacingTransaction);
            });
        }
    }

    public void saveTransactionDetail(TransactionDetail transactionDetail, String currentUser) throws IllegalAccessException {
        if (StringUtils.isBlank((CharSequence)currentUser)) {
            throw new IllegalArgumentException("username is mandatory");
        }
        String senderAddress = StringUtils.isBlank((CharSequence)transactionDetail.getBy()) ? transactionDetail.getFrom() : transactionDetail.getBy();
        Wallet senderWallet = this.accountService.getWalletByAddress(senderAddress);
        if (senderWallet == null || !this.accountService.isWalletOwner(senderWallet, currentUser)) {
            throw new IllegalAccessException("User '" + currentUser + "' is attempting to save a new transaction for wallet of user " + String.valueOf(senderWallet));
        }
        Wallet issuerWallet = this.accountService.getWalletByTypeAndId(WalletType.USER.getId(), currentUser);
        transactionDetail.setIssuer(issuerWallet);
        transactionDetail.setNetworkId(WalletUtils.getNetworkId());
        this.transactionStorage.saveTransactionDetail(transactionDetail);
    }

    public long getPendingTransactionMaxDays() {
        return this.pendingTransactionMaxDays;
    }

    public List<TransactionDetail> getTransactionsToSend() {
        List<TransactionDetail> transactionsToSend = this.transactionStorage.getTransactionsToSend(WalletUtils.getNetworkId());
        transactionsToSend.forEach(this::retrieveWalletsDetails);
        return transactionsToSend;
    }

    public boolean canSendTransactionToBlockchain(String fromAddress) {
        return this.transactionStorage.countPendingTransactionSent(WalletUtils.getNetworkId(), fromAddress) < this.getMaxParallelPendingTransactions();
    }

    public long getMaxAttemptsToSend() {
        return this.maxAttemptsToSend;
    }

    public long getMaxParallelPendingTransactions() {
        return this.maxParallelPendingTransactions;
    }

    public long countTransactions() {
        return this.transactionStorage.countTransactions();
    }

    private List<TransactionDetail> getTransactions(int limit, String currentUser) {
        List<TransactionDetail> transactionDetails = this.transactionStorage.getTransactions(WalletUtils.getNetworkId(), limit);
        transactionDetails.stream().forEach(transactionDetail -> this.retrieveWalletsDetails((TransactionDetail)transactionDetail, currentUser));
        return transactionDetails;
    }

    private List<TransactionDetail> getContractTransactions(String contractAddress, String contractMethodName, int limit, String currentUser) throws IllegalAccessException {
        ContractDetail contractDetail = this.contractService.getContractDetail(contractAddress);
        if (contractDetail == null) {
            throw new IllegalStateException("Can't find contract with address " + contractAddress);
        }
        if (!WalletUtils.isUserRewardingAdmin((String)currentUser)) {
            throw new IllegalAccessException("User " + currentUser + " attempts to access contract transactions with address " + contractAddress);
        }
        List<TransactionDetail> transactionDetails = this.transactionStorage.getContractTransactions(contractAddress, contractMethodName, limit);
        transactionDetails.stream().forEach(transactionDetail -> this.retrieveWalletsDetails((TransactionDetail)transactionDetail, currentUser));
        return transactionDetails;
    }

    private List<TransactionDetail> getWalletTransactions(String address, String contractAddress, String contractMethodName, String hash, int limit, boolean pending, boolean administration, String currentUser) throws IllegalAccessException {
        Wallet wallet = this.accountService.getWalletByAddress(address);
        if (wallet == null) {
            return Collections.emptyList();
        }
        if (!WalletUtils.canAccessWallet((Wallet)wallet, (String)currentUser)) {
            throw new IllegalAccessException("Can't access wallet with address " + address);
        }
        List<TransactionDetail> transactionDetails = this.transactionStorage.getWalletTransactions(WalletUtils.getNetworkId(), address, contractAddress, contractMethodName, hash, limit, pending, administration);
        transactionDetails.stream().forEach(transactionDetail -> this.retrieveWalletsDetails((TransactionDetail)transactionDetail, currentUser));
        return transactionDetails;
    }

    private void retrieveWalletsDetails(TransactionDetail transactionDetail) {
        if (transactionDetail == null || StringUtils.isBlank((CharSequence)transactionDetail.getFrom())) {
            return;
        }
        if (transactionDetail.getFromWallet() == null) {
            Wallet senderWallet = this.accountService.getWalletByAddress(transactionDetail.getFrom());
            transactionDetail.setFromWallet(senderWallet);
            WalletUtils.hideWalletOwnerPrivateInformation((Wallet)senderWallet);
        }
        if (transactionDetail.getToWallet() == null && StringUtils.isNotBlank((CharSequence)transactionDetail.getTo())) {
            Wallet receiverWallet = this.accountService.getWalletByAddress(transactionDetail.getTo());
            WalletUtils.hideWalletOwnerPrivateInformation((Wallet)receiverWallet);
            transactionDetail.setToWallet(receiverWallet);
        }
        if (transactionDetail.getByWallet() == null && StringUtils.isNotBlank((CharSequence)transactionDetail.getBy())) {
            Wallet senderWalletBy = this.accountService.getWalletByAddress(transactionDetail.getBy());
            WalletUtils.hideWalletOwnerPrivateInformation((Wallet)senderWalletBy);
            transactionDetail.setByWallet(senderWalletBy);
        }
        if (transactionDetail.getIssuer() == null && transactionDetail.getIssuerId() > 0L) {
            Wallet issuerWallet = this.accountService.getWalletByIdentityId(transactionDetail.getIssuerId());
            transactionDetail.setIssuer(issuerWallet);
        }
    }

    private void retrieveWalletsDetails(TransactionDetail transactionDetail, String currentUser) {
        if (transactionDetail == null || StringUtils.isBlank((CharSequence)transactionDetail.getFrom())) {
            return;
        }
        this.retrieveWalletsDetails(transactionDetail);
        if (StringUtils.isNotBlank((CharSequence)transactionDetail.getBy())) {
            if (!this.displayTransactionsLabel(transactionDetail.getByWallet(), currentUser)) {
                transactionDetail.setLabel(null);
            }
        } else if (!this.displayTransactionsLabel(transactionDetail.getFromWallet(), currentUser)) {
            transactionDetail.setLabel(null);
        }
    }

    private boolean displayTransactionsLabel(Wallet senderWallet, String currentUserId) {
        if (senderWallet == null || WalletUtils.isAdminAccount((String)senderWallet.getAddress())) {
            return WalletUtils.isUserRewardingAdmin((String)currentUserId);
        }
        String accountId = senderWallet.getId();
        String accountType = senderWallet.getType();
        if (StringUtils.isBlank((CharSequence)accountId) || StringUtils.isBlank((CharSequence)accountType)) {
            return WalletUtils.isUserRewardingAdmin((String)currentUserId);
        }
        if (WalletType.isSpace((String)senderWallet.getType())) {
            return this.getSpaceService().canManageSpace(WalletUtils.getSpace((String)accountId), currentUserId);
        }
        return StringUtils.equalsIgnoreCase((CharSequence)accountId, (CharSequence)currentUserId);
    }

    private void broadcastTransactionMinedEvent(TransactionDetail transactionDetail) {
        try {
            Map transaction = WalletUtils.transactionToMap((TransactionDetail)transactionDetail);
            this.getListenerService().broadcast("exo.wallet.transaction.mined", null, (Object)transaction);
        }
        catch (Exception e) {
            LOG.warn("Error while broadcasting transaction mined event: {}", new Object[]{transactionDetail, e});
        }
    }

    private SpaceService getSpaceService() {
        if (this.spaceService == null) {
            this.spaceService = (SpaceService)CommonsUtils.getService(SpaceService.class);
        }
        return this.spaceService;
    }

    private ListenerService getListenerService() {
        if (this.listenerService == null) {
            this.listenerService = (ListenerService)CommonsUtils.getService(ListenerService.class);
        }
        return this.listenerService;
    }

    private ZonedDateTime getEndDate(LocalDate selectedDay, String periodicity) {
        return StringUtils.equalsIgnoreCase((CharSequence)periodicity, (CharSequence)YEAR_PERIODICITY) ? YearMonth.of(selectedDay.getYear(), selectedDay.getMonthValue()).atEndOfMonth().plusDays(1L).atStartOfDay(ZoneId.systemDefault()) : selectedDay.plusDays(1L).atStartOfDay(ZoneId.systemDefault());
    }
}

