/*
 * This file is part of the Meeds project (https://meeds.io/).
 * Copyright (C) 2020 Meeds Association
 * contact@meeds.io
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package io.meeds.gamification.dao;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.stream.Collectors;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;

import org.exoplatform.commons.persistence.impl.GenericDAOJPAImpl;

import io.meeds.gamification.constant.IdentityType;
import io.meeds.gamification.constant.RealizationStatus;
import io.meeds.gamification.entity.RealizationEntity;
import io.meeds.gamification.model.PiechartLeaderboard;
import io.meeds.gamification.model.ProfileReputation;
import io.meeds.gamification.model.StandardLeaderboard;
import io.meeds.gamification.model.filter.RealizationFilter;

import jakarta.persistence.NoResultException;
import jakarta.persistence.Query;
import jakarta.persistence.Tuple;
import jakarta.persistence.TypedQuery;

public class RealizationDAO extends GenericDAOJPAImpl<RealizationEntity, Long> {

  private static final String        DATE_PARAM_NAME         = "date";

  private static final String        TO_DATE_PARAM_NAME      = "toDate";

  private static final String        FROM_DATE_PARAM_NAME    = "fromDate";

  private static final String        SPACE_ID_PARAM_NAME     = "spaceId";

  private static final String        EARNER_ID_PARAM_NAME    = "earnerId";

  private static final String        EARNER_IDS_PARAM_NAME   = "earnerIds";

  private static final String        REVIEWER_IDS_PARAM_NAME = "earnerIds";

  private static final String        PROGRAM_IDS_PARAM_NAME  = "programIds";

  private static final String        RULE_IDS_PARAM_NAME     = "ruleIds";

  private static final String        SPACE_IDS_PARAM_NAME    = "spaceIds";

  private static final String        EARNER_TYPE_PARAM_NAME  = "earnerType";

  public static final String         STATUS_PARAM_NAME       = "status";
  
  public static final String         STATUSES_PARAM_NAME     = "statuses";

  private static final String        RULE_ID_PARAM_NAME      = "ruleId";

  private static final String        RECEIVER_ID_PARAM_NAME  = "receiverId";

  private static final String        OBJECT_ID_PARAM_NAME    = "objectId";

  private static final String        OBJECT_TYPE_PARAM_NAME  = "objectType";

  private final Map<String, Boolean> filterNamedQueries      = new ConcurrentHashMap<>();

  public int getLeaderboardRank(IdentityType earnerType, String earnerIdentityId) {
    Query query = getEntityManager().createNamedQuery("RealizationEntity.getLeaderboardRank");
    query.setParameter(EARNER_TYPE_PARAM_NAME, earnerType)
         .setParameter(EARNER_ID_PARAM_NAME, Long.parseLong(earnerIdentityId))
         .setParameter(STATUS_PARAM_NAME, RealizationStatus.ACCEPTED);
    try {
      Object result = query.getSingleResult();
      return result == null ? 0 : Integer.parseInt(result.toString());
    } catch (NoResultException e) {
      return 0;
    }
  }

  public int getLeaderboardRankByDatesAndProgramIds(IdentityType earnerType,
                                                    String earnerIdentityId,
                                                    Date fromDate,
                                                    Date toDate,
                                                    Long ...programIds) {
    Query query = getEntityManager().createNamedQuery("RealizationEntity.getLeaderboardRankByDatesAndProgramIds");
    query.setParameter(FROM_DATE_PARAM_NAME, fromDate)
         .setParameter(TO_DATE_PARAM_NAME, toDate)
         .setParameter(PROGRAM_IDS_PARAM_NAME, Arrays.asList(programIds))
         .setParameter(EARNER_TYPE_PARAM_NAME, earnerType)
         .setParameter(EARNER_ID_PARAM_NAME, Long.parseLong(earnerIdentityId))
         .setParameter(STATUS_PARAM_NAME, RealizationStatus.ACCEPTED);
    try {
      Object result = query.getSingleResult();
      return result == null ? 0 : Integer.parseInt(result.toString());
    } catch (NoResultException e) {
      return 0;
    }
  }

  public int getLeaderboardRankByProgramIds(IdentityType earnerType, String earnerIdentityId, Long... programIds) {
    Query query = getEntityManager().createNamedQuery("RealizationEntity.getLeaderboardRankByProgramIds");
    query.setParameter(PROGRAM_IDS_PARAM_NAME,  Arrays.asList(programIds))
         .setParameter(EARNER_TYPE_PARAM_NAME, earnerType)
         .setParameter(EARNER_ID_PARAM_NAME, Long.parseLong(earnerIdentityId))
         .setParameter(STATUS_PARAM_NAME, RealizationStatus.ACCEPTED);
    try {
      Object result = query.getSingleResult();
      return result == null ? 0 : Integer.parseInt(result.toString());
    } catch (NoResultException e) {
      return 0;
    }
  }

  public int getLeaderboardRankByDates(IdentityType earnerType, String earnerIdentityId, Date fromDate, Date toDate) {
    Query query = getEntityManager().createNamedQuery("RealizationEntity.getLeaderboardRankByDates");
    query.setParameter(FROM_DATE_PARAM_NAME, fromDate)
         .setParameter(TO_DATE_PARAM_NAME, toDate)
         .setParameter(EARNER_TYPE_PARAM_NAME, earnerType)
         .setParameter(EARNER_ID_PARAM_NAME, Long.parseLong(earnerIdentityId))
         .setParameter(STATUS_PARAM_NAME, RealizationStatus.ACCEPTED);
    try {
      Object result = query.getSingleResult();
      return result == null ? 0 : Integer.parseInt(result.toString());
    } catch (NoResultException e) {
      return 0;
    }
  }

  public List<StandardLeaderboard> getLeaderboard(IdentityType earnerType, int offset, int limit) {
    TypedQuery<StandardLeaderboard> query = getEntityManager().createNamedQuery("RealizationEntity.getLeaderboard",
                                                                                StandardLeaderboard.class);
    query.setParameter(EARNER_TYPE_PARAM_NAME, earnerType).setParameter(STATUS_PARAM_NAME, RealizationStatus.ACCEPTED);
    query.setFirstResult(offset);
    query.setMaxResults(limit);
    return query.getResultList();
  }

  public List<StandardLeaderboard> getLeaderboardByProgramIds(IdentityType earnerType,
                                                              int offset,
                                                              int limit,
                                                              Long... programIds) {
    TypedQuery<StandardLeaderboard> query = getEntityManager().createNamedQuery("RealizationEntity.getLeaderboardByProgramIds",
                                                                                StandardLeaderboard.class);
    query.setParameter(PROGRAM_IDS_PARAM_NAME, Arrays.asList(programIds))
         .setParameter(EARNER_TYPE_PARAM_NAME, earnerType)
         .setParameter(STATUS_PARAM_NAME, RealizationStatus.ACCEPTED);
    query.setFirstResult(offset);
    query.setMaxResults(limit);
    return query.getResultList();
  }

  public List<StandardLeaderboard> getLeaderboardByDates(Date fromDate, Date toDate, IdentityType earnerType, int offset, int limit) {
    TypedQuery<StandardLeaderboard> query = getEntityManager().createNamedQuery("RealizationEntity.getLeaderboardByDate",
                                                                                StandardLeaderboard.class);
    query.setParameter(FROM_DATE_PARAM_NAME, fromDate)
         .setParameter(TO_DATE_PARAM_NAME, toDate)
         .setParameter(EARNER_TYPE_PARAM_NAME, earnerType)
         .setParameter(STATUS_PARAM_NAME, RealizationStatus.ACCEPTED);
    query.setFirstResult(offset);
    query.setMaxResults(limit);
    return query.getResultList();
  }

  public List<StandardLeaderboard> getLeaderboardByDatesAndProgramIds(Date fromDate,
                                                                     Date toDate, IdentityType earnerType,
                                                                     int offset,
                                                                     int limit,
                                                                     Long ...programIds) {
    TypedQuery<StandardLeaderboard> query =
                                          getEntityManager().createNamedQuery("RealizationEntity.getLeaderboardByDateAndProgramIds",
                                                                              StandardLeaderboard.class);
    query.setParameter(FROM_DATE_PARAM_NAME, fromDate)
         .setParameter(TO_DATE_PARAM_NAME, toDate)
         .setParameter(EARNER_TYPE_PARAM_NAME, earnerType)
         .setParameter(PROGRAM_IDS_PARAM_NAME, Arrays.asList(programIds))
         .setParameter(STATUS_PARAM_NAME, RealizationStatus.ACCEPTED);
    query.setFirstResult(offset);
    query.setMaxResults(limit);
    return query.getResultList();
  }

  public long getScoreByIdentityId(String earnerIdentityId) {
    TypedQuery<Long> query = getEntityManager().createNamedQuery("RealizationEntity.getScoreByIdentityId", Long.class);
    query.setParameter(EARNER_ID_PARAM_NAME, Long.parseLong(earnerIdentityId)).setParameter(STATUS_PARAM_NAME, RealizationStatus.ACCEPTED);
    try {
      Long result = query.getSingleResult();
      return result == null ? 0 : result;
    } catch (NoResultException e) {
      return 0;
    }
  }

  public List<PiechartLeaderboard> getLeaderboardStatsByIdentityIdAndDates(String earnerId, Long spaceId, Date fromDate, Date toDate) {
    boolean isSpaceQuery = spaceId != null && spaceId > 0;
    String queryName = isSpaceQuery ? "RealizationEntity.getLeaderboardStatsByIdentityIdAndDatesAndSpaceId" : "RealizationEntity.getLeaderboardStatsByIdentityIdAndDates";
    TypedQuery<PiechartLeaderboard> query = getEntityManager().createNamedQuery(queryName, PiechartLeaderboard.class);
    query.setParameter(EARNER_ID_PARAM_NAME, Long.parseLong(earnerId))
         .setParameter(FROM_DATE_PARAM_NAME, fromDate)
         .setParameter(TO_DATE_PARAM_NAME, toDate)
         .setParameter(STATUS_PARAM_NAME, RealizationStatus.ACCEPTED);
    if (isSpaceQuery) {
      query.setParameter(SPACE_ID_PARAM_NAME, spaceId);
    }
    return query.getResultList();
  }

  public List<PiechartLeaderboard> getLeaderboardStatsByIdentityId(String earnerId, Long spaceId) {
    boolean isSpaceQuery = spaceId != null && spaceId > 0;
    String queryName = isSpaceQuery ? "RealizationEntity.getLeaderboardStatsByIdentityIdAndSpaceId" : "RealizationEntity.getLeaderboardStatsByIdentityId";
    TypedQuery<PiechartLeaderboard> query = getEntityManager().createNamedQuery(queryName, PiechartLeaderboard.class);
    query.setParameter(EARNER_ID_PARAM_NAME, Long.parseLong(earnerId))
         .setParameter(STATUS_PARAM_NAME, RealizationStatus.ACCEPTED);
    if (isSpaceQuery) {
      query.setParameter(SPACE_ID_PARAM_NAME, spaceId);
    }
    return query.getResultList();
  }

  /**
   * Compute for a given user the score earned for each doman
   * 
   * @param earnerId ProfileReputation
   * @return a list of objects of type
   */
  public List<ProfileReputation> getScorePerProgramByIdentityId(String earnerId) {
    TypedQuery<ProfileReputation> query = getEntityManager().createNamedQuery("RealizationEntity.getScorePerProgramByIdentityId",
                                                                              ProfileReputation.class);
    query.setParameter(EARNER_ID_PARAM_NAME, Long.parseLong(earnerId)).setParameter(STATUS_PARAM_NAME, RealizationStatus.ACCEPTED);
    return query.getResultList();

  }

  public long getScoreByIdentityIdAndBetweenDates(String earnerId, Date fromDate, Date toDate, Long ...programIds) {
    boolean isProgramQuery = programIds != null && programIds.length > 0;
    String queryName = isProgramQuery ? "RealizationEntity.getScoreByIdentityIdAndBetweenDatesAndProgramIds" : "RealizationEntity.getScoreByIdentityIdAndBetweenDates";
    TypedQuery<Long> query = getEntityManager().createNamedQuery(queryName, Long.class);
    query.setParameter(EARNER_ID_PARAM_NAME, Long.parseLong(earnerId))
         .setParameter(FROM_DATE_PARAM_NAME, fromDate == null ? new Date(0L) : fromDate)
         .setParameter(TO_DATE_PARAM_NAME, toDate == null ? new Date() : toDate)
         .setParameter(STATUS_PARAM_NAME, RealizationStatus.ACCEPTED);
    if (isProgramQuery) {
      query.setParameter(PROGRAM_IDS_PARAM_NAME, Arrays.asList(programIds));
    }
    Long count = query.getSingleResult();
    return count == null ? 0 : count;
  }

  public Map<Long, Long> getScoreByIdentityIdsAndBetweenDates(List<String> earnersId, Date fromDate, Date toDate) {
    TypedQuery<Tuple> query = getEntityManager().createNamedQuery("RealizationEntity.getScoreByIdentityIdsAndBetweenDates",
                                                                  Tuple.class);
    query.setParameter("earnersId", earnersId.stream().map(Long::parseLong).toList())
         .setParameter(FROM_DATE_PARAM_NAME, fromDate)
         .setParameter(TO_DATE_PARAM_NAME, toDate)
         .setParameter(STATUS_PARAM_NAME, RealizationStatus.ACCEPTED);

    return query.getResultList()
                .stream()
                .collect(Collectors.toMap(tuple -> tuple.get(0, Long.class), tuple -> tuple.get(1, Long.class)));
  }

  public int countRealizationsByRuleIdAndEarnerId(String earnerIdentityId, long ruleId) {
    TypedQuery<Long> query = getEntityManager().createNamedQuery("RealizationEntity.countRealizationsByRuleIdAndEarnerId",
                                                                 Long.class);
    query.setParameter(RULE_ID_PARAM_NAME, ruleId)
         .setParameter(EARNER_ID_PARAM_NAME, Long.parseLong(earnerIdentityId))
         .setParameter(STATUS_PARAM_NAME, RealizationStatus.ACCEPTED);
    try {
      Long count = query.getSingleResult();
      return count == null ? 0 : count.intValue();
    } catch (NoResultException e) {
      return 0;
    }
  }
  
  public List<Long> getParticipantsBetweenDates(Date fromDate, Date toDate) {
    TypedQuery<Long> query = getEntityManager().createNamedQuery("RealizationEntity.getParticipantsBetweenDates", Long.class);
    query.setParameter(FROM_DATE_PARAM_NAME, fromDate);
    query.setParameter(TO_DATE_PARAM_NAME, toDate);
    query.setParameter(EARNER_TYPE_PARAM_NAME, IdentityType.USER);
    query.setParameter(STATUS_PARAM_NAME, RealizationStatus.ACCEPTED);
    try {
      return query.getResultList();
    } catch (NoResultException e) {
      return Collections.emptyList();
    }
  }
 
  public long countParticipantsBetweenDates(Date fromDate, Date toDate) {
    TypedQuery<Long> query = getEntityManager().createNamedQuery("RealizationEntity.countParticipantsBetweenDates", Long.class);
    query.setParameter(FROM_DATE_PARAM_NAME, fromDate);
    query.setParameter(TO_DATE_PARAM_NAME, toDate);
    query.setParameter(EARNER_TYPE_PARAM_NAME, IdentityType.USER);
    query.setParameter(STATUS_PARAM_NAME, RealizationStatus.ACCEPTED);
    try {
      Long count = query.getSingleResult();
      return count == null ? 0 : count.intValue();
    } catch (NoResultException e) {
      return 0;
    }
  }

  public int countRealizationsInPeriod(String earnerIdentityId, long ruleId, Date sinceDate) {
    TypedQuery<Long> query = getEntityManager().createNamedQuery("RealizationEntity.countRealizationsInPeriod", Long.class);
    query.setParameter(DATE_PARAM_NAME, sinceDate)
         .setParameter(RULE_ID_PARAM_NAME, ruleId)
         .setParameter(EARNER_ID_PARAM_NAME, Long.parseLong(earnerIdentityId))
         .setParameter(STATUS_PARAM_NAME, List.of(RealizationStatus.ACCEPTED, RealizationStatus.PENDING));
    try {
      Long count = query.getSingleResult();
      return count == null ? 0 : count.intValue();
    } catch (NoResultException e) {
      return 0;
    }
  }

  public Long findLastRealizationByRuleIdAndEarnerIdAndReceiverAndObjectId(long ruleId,
                                                                           String earnerId,
                                                                           String receiverId,
                                                                           String objectId,
                                                                           String objectType) {
    TypedQuery<Long> query =
                           getEntityManager().createNamedQuery("RealizationEntity.findRealizationsByRuleIdAndEarnerIdAndReceiverAndObjectId",
                                                               Long.class);
    query.setParameter(RULE_ID_PARAM_NAME, ruleId)
         .setParameter(EARNER_ID_PARAM_NAME, Long.parseLong(earnerId))
         .setParameter(RECEIVER_ID_PARAM_NAME, receiverId)
         .setParameter(OBJECT_ID_PARAM_NAME, objectId)
         .setParameter(OBJECT_TYPE_PARAM_NAME, objectType)
         .setParameter(STATUS_PARAM_NAME, List.of(RealizationStatus.ACCEPTED, RealizationStatus.PENDING));
    query.setMaxResults(1);

    List<Long> resultList = query.getResultList();
    return CollectionUtils.isEmpty(resultList) ? null : resultList.getFirst();
  }

  public List<Long> getRealizationsByObjectIdAndObjectType(String objectId, String objectType) {
    TypedQuery<Tuple> query = getEntityManager().createNamedQuery("RealizationEntity.getRealizationsByObjectIdAndObjectType", Tuple.class);
    query.setParameter(OBJECT_ID_PARAM_NAME, objectId);
    query.setParameter(OBJECT_TYPE_PARAM_NAME, objectType);
    try {
      return query.getResultList().stream().map(t -> t.get(0, Long.class)).toList();
    } catch (NoResultException e) {
      return Collections.emptyList();
    }
  }

  /**
   * Find realizations by filter with offset, limit.
   * 
   * @param realizationFilter : data Transfert Object {@link RealizationFilter}
   * @param offset : the starting index, when supplied. Starts at 0.
   * @param limit : how many realizations we should load from DB
   * @return a list of object of type {@link RealizationEntity}
   */
  public List<Long> findRealizationsByFilter(RealizationFilter realizationFilter, int offset, int limit) {
    TypedQuery<Tuple> query = buildQueryFromFilter(realizationFilter, Tuple.class, false);
    if (limit > 0) {
      query.setMaxResults(limit);
    }
    if (offset > 0) {
      query.setFirstResult(offset);
    }
    return query.getResultList().stream().map(t -> t.get(0, Long.class)).toList();
  }

  public int countRealizationsByFilter(RealizationFilter filter) {
    TypedQuery<Long> query = buildQueryFromFilter(filter, Long.class, true);
    return query.getSingleResult().intValue();
  }

  private <T> TypedQuery<T> buildQueryFromFilter(RealizationFilter filter, Class<T> clazz, boolean count) {
    List<String> suffixes = new ArrayList<>();
    List<String> predicates = new ArrayList<>();
    buildPredicates(filter, suffixes, predicates);

    TypedQuery<T> query;
    String queryName = getQueryFilterName(suffixes, count);
    if (filterNamedQueries.containsKey(queryName)) {
      query = getEntityManager().createNamedQuery(queryName, clazz);
    } else {
      String queryContent = getQueryFilterContent(filter, predicates, count);
      query = getEntityManager().createQuery(queryContent, clazz);
      getEntityManager().getEntityManagerFactory().addNamedQuery(queryName, query);
      filterNamedQueries.put(queryName, true);
    }
    addQueryFilterParameters(filter, query);
    return query;
  }

  private void buildPredicates(RealizationFilter filter, List<String> suffixes, List<String> predicates) {
    if (filter.getEarnerType() != null) {
      suffixes.add("EarnerType");
      predicates.add("g.earnerType = :" + EARNER_TYPE_PARAM_NAME);
    }

    if (filter.getFromDate() != null && filter.getToDate() != null) {
      suffixes.add("Interval");
      predicates.add("g.createdDate >= :" + FROM_DATE_PARAM_NAME + " AND g.createdDate < :" + TO_DATE_PARAM_NAME);
    }
    if (CollectionUtils.isNotEmpty(filter.getEarnerIds())) {
      suffixes.add("EarnerIds");
      predicates.add("g.earnerId IN (:" + EARNER_IDS_PARAM_NAME + ")");
    }
    if (CollectionUtils.isNotEmpty(filter.getProgramIds())) {
      suffixes.add("ProgramIds");
      predicates.add("g.domainEntity.id IN (:" + PROGRAM_IDS_PARAM_NAME + ")");
    }
    if (CollectionUtils.isNotEmpty(filter.getRuleIds())) {
      suffixes.add("RuleIds");
      predicates.add("g.ruleEntity.id IN (:" + RULE_IDS_PARAM_NAME + ")");
    }
    if (CollectionUtils.isNotEmpty(filter.getSpacesIds())) {
      suffixes.add("SpaceIds");
      predicates.add("g.domainEntity.audienceId IN (:" + SPACE_IDS_PARAM_NAME + ")");
    }
    if (CollectionUtils.isNotEmpty(filter.getReviewerIds())) {
      suffixes.add("ReviewerIds");
      predicates.add("g.reviewerId IN (:" + REVIEWER_IDS_PARAM_NAME + ")");
    }
    if (filter.getStatuses() != null) {
      suffixes.add("Status");
      predicates.add("g.status IN (:" + STATUSES_PARAM_NAME + ")");
    }

    suffixes.add(getSortField(filter));
    suffixes.add(filter.isSortDescending() ? "Descending" : "Ascending");
  }

  private String getQueryFilterName(List<String> suffixes, boolean count) {
    String queryName;
    if (suffixes.isEmpty()) {
      queryName = count ? "RealizationEntity.countRealizations" : "RealizationEntity.findRealizations";
    } else {
      queryName = (count ? "RealizationEntity.countRealizations" : "RealizationEntity.findRealizations") + "By"
          + StringUtils.join(suffixes, "And");
    }
    return queryName;
  }

  private String getQueryFilterContent(RealizationFilter filter, List<String> predicates, boolean count) {
    String querySelect = count ? "SELECT COUNT(g) FROM RealizationEntity g " :
                               "SELECT DISTINCT g.id, g." + getSortRealizationField(filter) + " FROM RealizationEntity g ";
    String queryContent;
    if (predicates.isEmpty()) {
      queryContent = querySelect;
    } else {
      queryContent = querySelect + " WHERE " + StringUtils.join(predicates, " AND ");
    }

    if (!count) {
      String sortDirection = filter.isSortDescending() ? "DESC" : "ASC";
      String sortField = getSortField(filter);
      if (StringUtils.equals(sortField, DATE_PARAM_NAME)) {
        queryContent += " ORDER BY g.id " + sortDirection;
      } else {
        queryContent += " ORDER BY g." + sortField + " " + sortDirection + " ,g.id DESC ";
      }
    }
    return queryContent;
  }

  private <T> void addQueryFilterParameters(RealizationFilter filter, TypedQuery<T> query) {
    if (filter.getEarnerType() != null) {
      query.setParameter(EARNER_TYPE_PARAM_NAME, filter.getEarnerType());
    }
    if (filter.getFromDate() != null && filter.getToDate() != null) {
      query.setParameter(FROM_DATE_PARAM_NAME, filter.getFromDate());
      query.setParameter(TO_DATE_PARAM_NAME, filter.getToDate());
    }
    if (CollectionUtils.isNotEmpty(filter.getEarnerIds())) {
      query.setParameter(EARNER_IDS_PARAM_NAME, filter.getEarnerIds().stream().map(Long::parseLong).toList());
    }
    if (CollectionUtils.isNotEmpty(filter.getProgramIds())) {
      query.setParameter(PROGRAM_IDS_PARAM_NAME, filter.getProgramIds());
    }
    if (CollectionUtils.isNotEmpty(filter.getRuleIds())) {
      query.setParameter(RULE_IDS_PARAM_NAME, filter.getRuleIds());
    }
    if (CollectionUtils.isNotEmpty(filter.getSpacesIds())) {
      query.setParameter(SPACE_IDS_PARAM_NAME, filter.getSpacesIds());
    }
    if (CollectionUtils.isNotEmpty(filter.getReviewerIds())) {
      query.setParameter(REVIEWER_IDS_PARAM_NAME, filter.getReviewerIds());
    }
    if (filter.getStatuses() != null) {
      query.setParameter(STATUSES_PARAM_NAME, filter.getStatuses());
    }
  }

  private String getSortRealizationField(RealizationFilter filter) {
    if (StringUtils.isBlank(filter.getSortField()) || StringUtils.equals(DATE_PARAM_NAME, filter.getSortField())) {
      return "createdDate";
    }
    return switch (filter.getSortField()) {
    case STATUS_PARAM_NAME, "type", EARNER_ID_PARAM_NAME, "receiver", OBJECT_ID_PARAM_NAME, OBJECT_TYPE_PARAM_NAME, EARNER_TYPE_PARAM_NAME, "actionScore", "globalScore", DATE_PARAM_NAME: {
      yield filter.getSortField();
    }
    default:
      throw new IllegalArgumentException("Unexpected Sort Field value: " + filter.getSortField());
    };
  }

  private String getSortField(RealizationFilter filter) {
    if (StringUtils.isBlank(filter.getSortField())) {
      return DATE_PARAM_NAME;
    }
    return switch (filter.getSortField()) {
    case STATUS_PARAM_NAME, "type", EARNER_ID_PARAM_NAME, "receiver", OBJECT_ID_PARAM_NAME, OBJECT_TYPE_PARAM_NAME, EARNER_TYPE_PARAM_NAME, "actionScore", "globalScore", DATE_PARAM_NAME: {
      yield filter.getSortField();
    }
    default:
      throw new IllegalArgumentException("Unexpected Sort Field value: " + filter.getSortField());
    };
  }

}
