/**
 * This file is part of the Meeds project (https://meeds.io/).
 *
 * Copyright (C) 2020 - 2025 Meeds Association contact@meeds.io
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
package org.gatein.pc.test.controller.unit;

import org.gatein.pc.api.Portlet;
import org.gatein.pc.api.PortletContext;
import org.gatein.pc.api.PortletInvoker;
import org.gatein.pc.api.PortletInvokerException;
import org.gatein.pc.api.spi.PortletInvocationContext;
import org.gatein.pc.controller.event.EventControllerContext;
import org.gatein.pc.controller.impl.AbstractControllerContext;
import org.gatein.pc.controller.impl.event.EventControllerContextImpl;
import org.gatein.pc.controller.impl.state.StateControllerContextImpl;
import org.gatein.pc.controller.state.PageNavigationalState;
import org.gatein.pc.controller.state.StateControllerContext;
import org.gatein.pc.api.invocation.PortletInvocation;
import org.gatein.pc.api.invocation.response.PortletInvocationResponse;
import org.gatein.pc.portlet.impl.spi.AbstractServerContext;
import org.gatein.pc.test.unit.PortletTestServlet;
import org.gatein.common.io.Serialization;

import jakarta.servlet.RequestDispatcher;
import jakarta.servlet.ServletContext;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.util.Collection;

public class PortletControllerContextImpl extends AbstractControllerContext
{

   /** . */
   private final PortletInvoker portletInvoker;
   
   /** . */
   private final EventControllerContext eventControllerContext;

   /** . */
   private final StateControllerContext stateControllerContext;

   /** . */
   private final Serialization<PageNavigationalState> serialization;

   public PortletControllerContextImpl(HttpServletRequest req, HttpServletResponse resp, ServletContext servletContext)
      throws IOException, ClassNotFoundException
   {
      super(req, resp);

      //
      this.portletInvoker = (PortletInvoker)servletContext.getAttribute("ConsumerPortletInvoker");
      this.eventControllerContext = new EventControllerContextImpl(portletInvoker);
      this.stateControllerContext = new StateControllerContextImpl();
      this.serialization = new PageNavigationalStateSerialization(stateControllerContext);
   }

   public PortletInvoker getPortletInvoker()
   {
      return portletInvoker;
   }

   protected Portlet getPortlet(String windowId) throws PortletInvokerException
   {
      return portletInvoker.getPortlet(PortletContext.createPortletContext(windowId));
   }

   public StateControllerContext getStateControllerContext()
   {
      return stateControllerContext;
   }

   public PortletInvocationResponse invoke(PortletInvocation invocation) throws PortletInvokerException
   {

      // Override ServerContext
      invocation.setServerContext(new AbstractServerContext(
         getClientRequest(), getClientResponse()
      ) {

         @Override
         public void dispatch(ServletContext target, HttpServletRequest request, HttpServletResponse response, Callable callable) throws Exception
         {
            RequestDispatcher dispatcher = target.getRequestDispatcher("/portlet");
            PortletTestServlet.callback.set(callable);
            try
            {
               dispatcher.include(getClientRequest(), getClientResponse());
            }
            finally
            {
               PortletTestServlet.callback.set(null);
            }
         }
      });

      //
      return portletInvoker.invoke(invocation);
   }

   public EventControllerContext getEventControllerContext()
   {
      return eventControllerContext;
   }

   public Collection<Portlet> getPortlets() throws PortletInvokerException
   {
      return portletInvoker.getPortlets();
   }

   public PortletInvocationContext createPortletInvocationContext(String windowId, PageNavigationalState pageNavigationalState)
   {
      return new ControllerPortletInvocationContext(serialization, req, resp, windowId, pageNavigationalState);
   }

   public Serialization<PageNavigationalState> getPageNavigationalStateSerialization()
   {
      return serialization;
   }
}
