/**
 * This file is part of the Meeds project (https://meeds.io/).
 *
 * Copyright (C) 2020 - 2025 Meeds Association contact@meeds.io
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
package org.gatein.pc.portlet;

import org.gatein.pc.api.Portlet;
import org.gatein.pc.api.PortletContext;
import org.gatein.pc.api.PortletInvoker;
import org.gatein.pc.api.PortletInvokerException;
import org.gatein.pc.api.PortletStateType;
import org.gatein.pc.api.PortletStatus;
import org.gatein.pc.api.invocation.PortletInvocation;
import org.gatein.pc.api.invocation.response.PortletInvocationResponse;
import org.gatein.pc.api.state.DestroyCloneFailure;
import org.gatein.pc.api.state.PropertyChange;
import org.gatein.pc.api.state.PropertyMap;

import java.util.List;
import java.util.Set;
import java.util.concurrent.atomic.AtomicReference;

/**
 * A base class for  {@link org.gatein.pc.api.PortletInvoker} interface interceptors. The subclasses extends it and
 * override the intercepted methods. The next interceptor in the chain is wired in the field {@link #next} of the
 * interceptor. When the interceptor wants to give control to the next interceptor, it must invoke the same method on
 * the super class. If no next interceptor is configured the invocation of the parent method will throw an {@link
 * IllegalStateException}.
 *
 */
public class PortletInvokerInterceptor implements PortletInvoker
{

   /** . */
   private final AtomicReference<PortletInvoker> next = new AtomicReference<PortletInvoker>();

   public PortletInvokerInterceptor()
   {
   }

   public PortletInvokerInterceptor(PortletInvoker next)
   {
      this.next.set(next);
   }

   public PortletInvoker getNext()
   {
      return next.get();
   }

   public void setNext(PortletInvoker next)
   {
      this.next.set(next);
   }

   public PortletInvokerInterceptor append(PortletInvoker invoker)
   {
      PortletInvoker ref = next.get();
      if (ref == null)
      {
         next.set(invoker);
      }
      else if (ref instanceof PortletInvokerInterceptor)
      {
         ((PortletInvokerInterceptor)ref).append(invoker);
      }
      else
      {
         throw new IllegalStateException("End reached");
      }
      return this;
   }

   public Set<Portlet> getPortlets() throws PortletInvokerException
   {
      return safeGetNext().getPortlets();
   }

   public Portlet getPortlet(PortletContext portletContext) throws IllegalArgumentException, PortletInvokerException
   {
      return safeGetNext().getPortlet(portletContext);
   }

   public PortletStatus getStatus(PortletContext portletContext) throws IllegalArgumentException, PortletInvokerException
   {
      return safeGetNext().getStatus(portletContext);
   }

   public PortletInvocationResponse invoke(PortletInvocation invocation) throws IllegalArgumentException, PortletInvokerException
   {
      return safeGetNext().invoke(invocation);
   }

   public PortletContext createClone(PortletStateType stateType, PortletContext portletContext) throws IllegalArgumentException, PortletInvokerException, UnsupportedOperationException
   {
      return safeGetNext().createClone(stateType, portletContext);
   }

   public List<DestroyCloneFailure> destroyClones(List<PortletContext> portletContexts) throws IllegalArgumentException, PortletInvokerException, UnsupportedOperationException
   {
      return safeGetNext().destroyClones(portletContexts);
   }

   public PropertyMap getProperties(PortletContext portletContext, Set<String> keys) throws IllegalArgumentException, PortletInvokerException, UnsupportedOperationException
   {
      return safeGetNext().getProperties(portletContext, keys);
   }

   public PropertyMap getProperties(PortletContext portletContext) throws IllegalArgumentException, PortletInvokerException, UnsupportedOperationException
   {
      return safeGetNext().getProperties(portletContext);
   }

   public PortletContext setProperties(PortletContext portletContext, PropertyChange[] changes) throws IllegalArgumentException, PortletInvokerException, UnsupportedOperationException
   {
      return safeGetNext().setProperties(portletContext, changes);
   }

   /**
    * Attempt to get the next invoker, the method never returns a null value and rather throws an {@link
    * IllegalStateException} if the next invoker cannot be obtained.
    *
    * @return the non null next invoker
    */
   private PortletInvoker safeGetNext()
   {
      PortletInvoker next = this.next.get();

      //
      if (next == null)
      {
         throw new IllegalStateException("No next invoker");
      }

      //
      return next;
   }

   public PortletContext exportPortlet(PortletStateType stateType, PortletContext originalPortletContext)
      throws PortletInvokerException, IllegalArgumentException
   {
      return safeGetNext().exportPortlet(stateType, originalPortletContext);
   }

   public PortletContext importPortlet(PortletStateType stateType, PortletContext originalPortletContext)
      throws PortletInvokerException, IllegalArgumentException
   {
      return safeGetNext().importPortlet(stateType, originalPortletContext);
   }

   public boolean isActive()
   {
      return safeGetNext().isActive();
   }
}
