/**
 * This file is part of the Meeds project (https://meeds.io/).
 *
 * Copyright (C) 2020 - 2024 Meeds Association contact@meeds.io
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
package io.meeds.ide.portlet;

import java.io.IOException;
import java.util.Map.Entry;
import java.util.Set;

import javax.portlet.PortletConfig;
import javax.portlet.PortletException;
import javax.portlet.PortletPreferences;
import javax.portlet.PortletRequestDispatcher;
import javax.portlet.RenderRequest;
import javax.portlet.RenderResponse;

import org.exoplatform.commons.ObjectAlreadyExistsException;
import org.exoplatform.commons.api.portlet.GenericDispatchedViewPortlet;
import org.exoplatform.container.ExoContainerContext;
import org.exoplatform.portal.config.UserACL;
import org.exoplatform.portal.config.model.Application;
import org.exoplatform.portal.config.model.ApplicationState;
import org.exoplatform.portal.mop.service.LayoutService;
import org.exoplatform.portal.pom.spi.portlet.Portlet;
import org.exoplatform.portal.webui.application.UIPortlet;
import org.exoplatform.social.core.identity.model.Identity;
import org.exoplatform.social.webui.Utils;

import io.meeds.ide.model.Widget;
import io.meeds.ide.service.WidgetService;

import static io.meeds.layout.util.JsonUtils.fromJsonString;
import static io.meeds.social.portlet.CMSPortlet.DATA_INIT_PREFERENCE_NAME;

public class WidgetPortlet extends GenericDispatchedViewPortlet {

  private static final String WIDGET_ID_PARAM           = "widgetId";

  private static final String PORTLET_INSTANCE_ID_PARAM = "portletInstanceId";

  private String              editDispatchedPath;

  private LayoutService       layoutService;

  private UserACL             userAcl;

  @Override
  public void init(PortletConfig config) throws PortletException {
    super.init(config);
    this.editDispatchedPath = config.getInitParameter("portlet-edit-dispatched-file-path");
  }

  @Override
  public void doView(RenderRequest request, RenderResponse response) throws PortletException, IOException {
    checkPreferences(request);
    super.doView(request, response);
  }

  @Override
  protected void doEdit(RenderRequest request, RenderResponse response) throws PortletException, IOException {
    checkPreferences(request);
    PortletRequestDispatcher prd = getPortletContext().getRequestDispatcher(editDispatchedPath);
    PortletPreferences preferences = request.getPreferences();
    if (preferences != null) {
      Set<Entry<String, String[]>> preferencesEntries = preferences.getMap().entrySet();
      for (Entry<String, String[]> entry : preferencesEntries) {
        request.setAttribute(entry.getKey(), entry.getValue());
      }
    }
    prd.include(request, response);
  }

  private void checkPreferences(RenderRequest request) throws PortletException {
    PortletPreferences preferences = request.getPreferences();
    if (preferences.getValue(DATA_INIT_PREFERENCE_NAME, null) != null) {
      try {
        Widget widget = new Widget();
        Widget imported = fromJsonString(preferences.getValue(DATA_INIT_PREFERENCE_NAME, null), Widget.class);
        if (imported != null) {
          widget.setJs(imported.getJs());
          widget.setHtml(imported.getHtml());
          widget.setCss(imported.getCss());
        }
        WidgetService widgetService = ExoContainerContext.getService(WidgetService.class);
        widget = widgetService.createWidget(widget, getUserAcl().getSuperUser());
        String storageId = UIPortlet.getCurrentUIPortlet().getStorageId();
        Application applicationModel = getLayoutService().getApplicationModel(storageId);
        ApplicationState state = applicationModel.getState();
        Portlet prefs = getLayoutService().load(state);
        prefs.setValue(WIDGET_ID_PARAM, String.valueOf(widget.getId()));
        prefs.setValue(DATA_INIT_PREFERENCE_NAME, null);
        layoutService.save(state, prefs);
        preferences.setValue(WIDGET_ID_PARAM, String.valueOf(widget.getId()));
        preferences.setValue(DATA_INIT_PREFERENCE_NAME, null);
      } catch (ObjectAlreadyExistsException e) {
        throw new PortletException("Widget already exists", e);
      } catch (IllegalAccessException e) {
        throw new PortletException("User not allowed to change Widget settings", e);
      }
    }
    if (preferences.getValue(PORTLET_INSTANCE_ID_PARAM, null) != null && preferences.getValue(WIDGET_ID_PARAM, null) == null) {
      long portletInstanceId = Long.parseLong(preferences.getValue(PORTLET_INSTANCE_ID_PARAM, null));
      Identity identity = Utils.getViewerIdentity();
      try {
        WidgetService widgetService = ExoContainerContext.getService(WidgetService.class);
        Widget widget = widgetService.getWidgetByPortletId(portletInstanceId);
        if (widget == null) {
          widget = new Widget();
          widget.setPortletId(portletInstanceId);
          widget = widgetService.createWidget(widget, identity.getRemoteId());
        }
        preferences.setValue(WIDGET_ID_PARAM, String.valueOf(widget.getId()));
        String storageId = UIPortlet.getCurrentUIPortlet().getStorageId();
        Application applicationModel = getLayoutService().getApplicationModel(storageId);
        ApplicationState state = applicationModel.getState();
        Portlet prefs = getLayoutService().load(state);
        prefs.setValue(WIDGET_ID_PARAM, String.valueOf(widget.getId()));
        layoutService.save(state, prefs);
      } catch (IllegalAccessException e) {
        throw new PortletException("User not allowed to change Widget settings", e);
      } catch (ObjectAlreadyExistsException e) {
        throw new PortletException("Associated Widget already exists for portlet with instance id " + portletInstanceId, e);
      }
    }
  }

  private LayoutService getLayoutService() {
    if (layoutService == null) {
      layoutService = ExoContainerContext.getService(LayoutService.class);
    }
    return layoutService;
  }

  private UserACL getUserAcl() {
    if (userAcl == null) {
      userAcl = ExoContainerContext.getService(UserACL.class);
    }
    return userAcl;
  }
}
