/*
 * This file is part of the Meeds project (https://meeds.io/).
 * Copyright (C) 2020 Meeds Association
 * contact@meeds.io
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package org.exoplatform.perkstore.statistic;

import static org.exoplatform.perkstore.statistic.StatisticUtils.*;

import java.lang.reflect.Method;
import java.util.Map;

import org.apache.commons.lang3.StringUtils;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.reflect.MethodSignature;

import org.exoplatform.services.log.ExoLogger;
import org.exoplatform.services.log.Log;

/**
 * See {@link ExoPerkStoreStatistic}
 */
@Aspect
public class ExoPerkStoreStatisticAspect {
  private static final Log LOG = ExoLogger.getLogger(ExoPerkStoreStatisticAspect.class);

  /**
   * Encapsulate method annotated with {@link ExoPerkStoreStatistic} to add
   * statistic log entry
   * 
   * @param point processing point
   * @return result of processing point after its execution
   * @throws Throwable if processing point throws an exception
   */
  @Around("execution(* *(..)) && @annotation(org.exoplatform.perkstore.statistic.ExoPerkStoreStatistic)")
  public Object around(ProceedingJoinPoint point) throws Throwable {
    ExoPerkStoreStatisticService statisticService = (ExoPerkStoreStatisticService) point.getThis();
    MethodSignature methodSignature = (MethodSignature) point.getSignature();
    Method method = methodSignature.getMethod();
    ExoPerkStoreStatistic annotation = method.getAnnotation(ExoPerkStoreStatistic.class);
    boolean local = annotation.local();
    String service = annotation.service();
    String operation = annotation.operation();

    String errorMessage = null;
    long startTime = System.currentTimeMillis();
    Object result = null;
    try {
      result = point.proceed();
      return result;
    } catch (RuntimeException e) {
      errorMessage = e.getMessage();
      throw e;
    } finally {
      long duration = System.currentTimeMillis() - startTime;
      try {
        Map<String, Object> parameters = statisticService.getStatisticParameters(operation, result, point.getArgs());
        if (parameters != null) {
          if (local) {
            parameters.put(LOCAL_SERVICE, service);
          } else {
            parameters.put(REMOTE_SERVICE, service);
          }
          if (!parameters.containsKey(OPERATION)) {
            parameters.put(OPERATION, operation);
          }
          if (!parameters.containsKey(DURATION)) {
            parameters.put(DURATION, duration);
          }
          if (parameters.containsKey(ERROR_MSG) || StringUtils.isNotBlank(errorMessage)) {
            if (!parameters.containsKey(ERROR_MSG)) {
              parameters.put(ERROR_MSG, errorMessage);
            }
            parameters.put(STATUS, "ko");
            parameters.put(STATUS_CODE, "500");
          } else {
            if (!parameters.containsKey(STATUS)) {
              parameters.put(STATUS, "ok");
            }
            if (!parameters.containsKey(STATUS_CODE)) {
              parameters.put(STATUS_CODE, "200");
            }
          }
          addStatisticEntry(parameters);
        }
      } catch (Throwable e) {
        LOG.warn("Error adding statistic log entry in method {} for statistic type {}", method.getName(), operation, e);
      }
    }
  }

}
