/*
 * Decompiled with CFR 0.152.
 */
package io.mongock.driver.core.lock;

import io.mongock.driver.api.lock.LockCheckException;
import io.mongock.driver.api.lock.LockManager;
import io.mongock.driver.core.lock.LockDaemon;
import io.mongock.driver.core.lock.LockEntry;
import io.mongock.driver.core.lock.LockPersistenceException;
import io.mongock.driver.core.lock.LockRepository;
import io.mongock.driver.core.lock.LockStatus;
import io.mongock.utils.TimeService;
import io.mongock.utils.annotation.NotThreadSafe;
import java.time.Instant;
import java.util.Date;
import java.util.UUID;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@NotThreadSafe
public class DefaultLockManager
implements LockManager {
    private static final Logger logger = LoggerFactory.getLogger(DefaultLockManager.class);
    private static final String GOING_TO_SLEEP_MSG = "Mongock is going to sleep to wait for the lock:  {} ms({} minutes)";
    private static final String EXPIRATION_ARG_ERROR_MSG = "Lock expiration period must be greater than %d ms";
    private static final String MAX_TRIES_ERROR_TEMPLATE = "Quit trying lock after %s millis due to LockPersistenceException: \n\tcurrent lock:  %s\n\tnew lock: %s\n\tacquireLockQuery: %s\n\tdb error detail: %s";
    private static final String LOCK_HELD_BY_OTHER_PROCESS = "Lock held by other process. Cannot ensure lock.\n\tcurrent lock:  %s\n\tnew lock: %s\n\tacquireLockQuery: %s\n\tdb error detail: %s";
    private static final long MIN_LOCK_ACQUIRED_FOR_MILLIS = 3000L;
    private static final long DEFAULT_LOCK_ACQUIRED_FOR_MILLIS = 60000L;
    private static final long DEFAULT_QUIT_TRY_AFTER_MILLIS = 180000L;
    private static final double LOCK_REFRESH_MARGIN_PERCENTAGE = 0.33;
    private static final long MIN_LOCK_REFRESH_MARGIN_MILLIS = 1000L;
    private static final long DEFAULT_LOCK_REFRESH_MARGIN_MILLIS = 19800L;
    private static final long MINIMUM_WAITING_TO_TRY_AGAIN = 500L;
    private static final long DEFAULT_TRY_FREQUENCY_MILLIS = 1000L;
    private final LockRepository repository;
    private final TimeService timeUtils;
    private final String owner;
    private LockDaemon lockDaemon;
    private final long lockAcquiredForMillis;
    private final long lockTryFrequencyMillis;
    private final long lockRefreshMarginMillis;
    private final long lockQuitTryingAfterMillis;
    private volatile Date lockExpiresAt = null;
    private volatile Instant shouldStopTryingAt;

    public static DefaultLockManagerBuilder builder() {
        return new DefaultLockManagerBuilder();
    }

    public DefaultLockManager(LockRepository repository, TimeService timeUtils, long lockAcquiredForMillis, long lockQuitTryingAfterMillis, long lockTryFrequencyMillis, long lockRefreshMarginMillis) {
        this.repository = repository;
        this.timeUtils = timeUtils;
        this.lockAcquiredForMillis = lockAcquiredForMillis;
        this.lockQuitTryingAfterMillis = lockQuitTryingAfterMillis;
        this.lockTryFrequencyMillis = lockTryFrequencyMillis;
        this.lockRefreshMarginMillis = lockRefreshMarginMillis;
        this.owner = UUID.randomUUID().toString();
    }

    public void acquireLockDefault() throws LockCheckException {
        this.acquireLock(this.getDefaultKey());
    }

    private void acquireLock(String lockKey) throws LockCheckException {
        boolean keepLooping = true;
        do {
            try {
                logger.info("Mongock trying to acquire the lock");
                Date newLockExpiresAt = this.timeUtils.currentDatePlusMillis(this.lockAcquiredForMillis);
                this.repository.insertUpdate(new LockEntry(lockKey, LockStatus.LOCK_HELD.name(), this.owner, newLockExpiresAt));
                logger.info("Mongock acquired the lock until: {}", (Object)newLockExpiresAt);
                this.updateStatus(newLockExpiresAt);
                this.lockDaemon.activate();
                keepLooping = false;
            }
            catch (LockPersistenceException ex) {
                this.handleLockException(true, ex);
            }
        } while (keepLooping);
    }

    public void ensureLockDefault() throws LockCheckException {
        this.ensureLock(this.getDefaultKey());
    }

    private void ensureLock(String lockKey) throws LockCheckException {
        boolean keepLooping = true;
        do {
            if (this.needsRefreshLock()) {
                try {
                    logger.info("Mongock trying to refresh the lock");
                    Date lockExpiresAtTemp = this.timeUtils.currentDatePlusMillis(this.lockAcquiredForMillis);
                    LockEntry lockEntry = new LockEntry(lockKey, LockStatus.LOCK_HELD.name(), this.owner, lockExpiresAtTemp);
                    this.repository.updateIfSameOwner(lockEntry);
                    this.updateStatus(lockExpiresAtTemp);
                    logger.info("Mongock refreshed the lock until: {}", (Object)lockExpiresAtTemp);
                    this.lockDaemon.activate();
                    keepLooping = false;
                }
                catch (LockPersistenceException ex) {
                    this.handleLockException(false, ex);
                }
                continue;
            }
            keepLooping = false;
        } while (keepLooping);
    }

    public void releaseLockDefault() {
        this.releaseLock(this.getDefaultKey());
    }

    public void close() {
        this.releaseLockDefault();
    }

    private synchronized void releaseLock(String lockKey) {
        if (this.lockDaemon != null) {
            this.lockDaemon.cancel();
        }
        if (this.lockExpiresAt == null) {
            return;
        }
        logger.info("Mongock releasing the lock");
        this.repository.removeByKeyAndOwner(lockKey, this.getOwner());
        this.lockExpiresAt = null;
        this.shouldStopTryingAt = Instant.now();
        logger.info("Mongock released the lock");
    }

    public long getLockTryFrequency() {
        return this.lockTryFrequencyMillis;
    }

    private void handleLockException(boolean acquiringLock, LockPersistenceException ex) {
        LockEntry currentLock = this.repository.findByKey(this.getDefaultKey());
        if (this.isAcquisitionTimerOver()) {
            this.updateStatus(null);
            throw new LockCheckException(String.format(MAX_TRIES_ERROR_TEMPLATE, this.lockQuitTryingAfterMillis, currentLock != null ? currentLock.toString() : "none", ex.getNewLockEntity(), ex.getAcquireLockQuery(), ex.getDbErrorDetail()));
        }
        if (this.isLockOwnedByOtherProcess(currentLock)) {
            Date currentLockExpiresAt = currentLock.getExpiresAt();
            logger.warn("Lock is taken by other process until: {}", (Object)currentLockExpiresAt);
            if (!acquiringLock) {
                throw new LockCheckException(String.format(LOCK_HELD_BY_OTHER_PROCESS, currentLock.toString(), ex.getNewLockEntity(), ex.getAcquireLockQuery(), ex.getDbErrorDetail()));
            }
            this.waitForLock(currentLockExpiresAt);
        }
    }

    private boolean isLockOwnedByOtherProcess(LockEntry currentLock) {
        return currentLock != null && !currentLock.isOwner(this.owner);
    }

    private void waitForLock(Date expiresAtMillis) {
        Date current = this.timeUtils.currentTime();
        long currentLockWillExpireInMillis = expiresAtMillis.getTime() - current.getTime();
        long sleepingMillis = this.lockTryFrequencyMillis;
        if (this.lockTryFrequencyMillis > currentLockWillExpireInMillis) {
            logger.info("The configured time frequency[{} millis] is higher than the current lock's expiration", (Object)this.lockTryFrequencyMillis);
            sleepingMillis = currentLockWillExpireInMillis > 500L ? currentLockWillExpireInMillis : 500L;
        }
        logger.info("Mongock will try to acquire the lock in {} mills", (Object)sleepingMillis);
        try {
            logger.info(GOING_TO_SLEEP_MSG, (Object)sleepingMillis, (Object)this.timeUtils.millisToMinutes(sleepingMillis));
            Thread.sleep(sleepingMillis);
        }
        catch (InterruptedException ex) {
            logger.error("ERROR acquiring the lock", (Throwable)ex);
            Thread.currentThread().interrupt();
        }
    }

    public String getOwner() {
        return this.owner;
    }

    public boolean isLockHeld() {
        return this.lockExpiresAt != null && this.timeUtils.currentTime().compareTo(this.lockExpiresAt) < 1;
    }

    public long getMillisUntilRefreshRequired() {
        if (this.lockExpiresAt != null) {
            return this.lockExpiresAt.getTime() - this.timeUtils.currentTime().getTime() - this.lockRefreshMarginMillis;
        }
        return this.lockAcquiredForMillis - this.lockRefreshMarginMillis;
    }

    private boolean needsRefreshLock() {
        Date expirationWithMargin;
        if (this.lockExpiresAt == null) {
            return true;
        }
        Date currentTime = this.timeUtils.currentTime();
        return currentTime.compareTo(expirationWithMargin = new Date(this.lockExpiresAt.getTime() - this.lockRefreshMarginMillis)) >= 0;
    }

    private void updateStatus(Date lockExpiresAt) {
        this.lockExpiresAt = lockExpiresAt;
        this.finishAcquisitionTimer();
    }

    protected void initialize() {
        if (this.shouldStopTryingAt == null) {
            this.shouldStopTryingAt = this.timeUtils.nowPlusMillis(this.lockQuitTryingAfterMillis);
        }
        if (this.lockDaemon == null) {
            this.lockDaemon = new LockDaemon(this);
            this.lockDaemon.start();
        }
    }

    private void finishAcquisitionTimer() {
        this.shouldStopTryingAt = null;
    }

    private boolean isAcquisitionTimerOver() {
        return this.timeUtils.isPast(this.shouldStopTryingAt);
    }

    public static class DefaultLockManagerBuilder {
        private LockRepository lockRepository;
        private long lockAcquiredForMillis = 60000L;
        private long lockTryFrequencyMillis = 1000L;
        private long lockRefreshMarginMillis = 19800L;
        private long lockQuitTryingAfterMillis = 180000L;
        private TimeService timeService = new TimeService();

        public DefaultLockManagerBuilder setLockQuitTryingAfterMillis(long millis) {
            if (millis <= 0L) {
                throw new IllegalArgumentException("Lock-quit-trying-after must be grater than 0 ");
            }
            this.lockQuitTryingAfterMillis = millis;
            return this;
        }

        public DefaultLockManagerBuilder setLockTryFrequencyMillis(long millis) {
            if (millis < 500L) {
                throw new IllegalArgumentException(String.format("Lock-try-frequency must be grater than %d", 500L));
            }
            this.lockTryFrequencyMillis = millis;
            return this;
        }

        public DefaultLockManagerBuilder setLockAcquiredForMillis(long millis) {
            if (millis < 3000L) {
                throw new IllegalArgumentException(String.format(DefaultLockManager.EXPIRATION_ARG_ERROR_MSG, 3000L));
            }
            this.lockAcquiredForMillis = millis;
            long marginTemp = (long)((double)this.lockAcquiredForMillis * 0.33);
            this.lockRefreshMarginMillis = Math.max(marginTemp, 1000L);
            return this;
        }

        public DefaultLockManagerBuilder setLockRepository(LockRepository lockRepository) {
            this.lockRepository = lockRepository;
            return this;
        }

        public DefaultLockManagerBuilder setTimeUtils(TimeService timeService) {
            this.timeService = timeService;
            return this;
        }

        public DefaultLockManager build() {
            DefaultLockManager lockManager = new DefaultLockManager(this.lockRepository, this.timeService, this.lockAcquiredForMillis, this.lockQuitTryingAfterMillis, this.lockTryFrequencyMillis, this.lockRefreshMarginMillis);
            lockManager.initialize();
            return lockManager;
        }
    }
}

