/*
 * Copyright 2014 Red Hat, Inc.
 *
 * Red Hat licenses this file to you under the Apache License, version 2.0
 * (the "License"); you may not use this file except in compliance with the
 * License.  You may obtain a copy of the License at:
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

package io.vertx.axle.core.http;

import java.util.Map;
import org.reactivestreams.Publisher;
import io.reactivex.Flowable;
import java.util.function.Consumer;
import java.util.concurrent.CompletionStage;
import io.vertx.core.http.StreamPriority;
import io.vertx.core.http.HttpMethod;
import io.vertx.core.AsyncResult;
import io.vertx.core.Handler;

/**
 * Represents a server-side HTTP response.
 * <p>
 * An instance of this is created and associated to every instance of
 * {@link io.vertx.axle.core.http.HttpServerRequest} that.
 * <p>
 * It allows the developer to control the HTTP response that is sent back to the
 * client for a particular HTTP request.
 * <p>
 * It contains methods that allow HTTP headers and trailers to be set, and for a body to be written out to the response.
 * <p>
 * It also allows files to be streamed by the kernel directly from disk to the
 * outgoing HTTP connection, bypassing user space altogether (where supported by
 * the underlying operating system). This is a very efficient way of
 * serving files from the server since buffers do not have to be read one by one
 * from the file and written to the outgoing socket.
 * <p>
 * It implements {@link io.vertx.axle.core.streams.WriteStream} so it can be used with
 * {@link io.vertx.axle.core.streams.Pump} to pump data with flow control.
 *
 * <p/>
 * NOTE: This class has been automatically generated from the {@link io.vertx.core.http.HttpServerResponse original} non RX-ified interface using Vert.x codegen.
 */

@io.vertx.lang.axle.Gen(io.vertx.core.http.HttpServerResponse.class)
public class HttpServerResponse implements io.vertx.axle.core.streams.WriteStream<io.vertx.axle.core.buffer.Buffer> {

  @Override
  public String toString() {
    return delegate.toString();
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) return true;
    if (o == null || getClass() != o.getClass()) return false;
    HttpServerResponse that = (HttpServerResponse) o;
    return delegate.equals(that.delegate);
  }
  
  @Override
  public int hashCode() {
    return delegate.hashCode();
  }

  public static final io.vertx.lang.axle.TypeArg<HttpServerResponse> __TYPE_ARG = new io.vertx.lang.axle.TypeArg<>(    obj -> new HttpServerResponse((io.vertx.core.http.HttpServerResponse) obj),
    HttpServerResponse::getDelegate
  );

  private final io.vertx.core.http.HttpServerResponse delegate;
  
  public HttpServerResponse(io.vertx.core.http.HttpServerResponse delegate) {
    this.delegate = delegate;
  }

  /**
  * Empty constructor used by CDI, do not use this constructor directly.
  **/
  HttpServerResponse() {    this.delegate = null;
  }

  public io.vertx.core.http.HttpServerResponse getDelegate() {
    return delegate;
  }

  /**
   * Same as {@link io.vertx.axle.core.http.HttpServerResponse#end} but with an <code>handler</code> called when the operation completes
   * @param handler 
   */
  private void __end(Handler<AsyncResult<Void>> handler) { 
    delegate.end(handler);
  }

  /**
   * Same as {@link io.vertx.axle.core.http.HttpServerResponse#end} but with an <code>handler</code> called when the operation completes
   * @return 
   */
  public CompletionStage<Void> end() { 
    return io.vertx.axle.AsyncResultCompletionStage.toCompletionStage(handler -> {
      __end(handler);
    });
  }

  /**
   * This will return <code>true</code> if there are more bytes in the write queue than the value set using {@link io.vertx.axle.core.http.HttpServerResponse#setWriteQueueMaxSize}
   * @return true if write queue is full
   */
  public boolean writeQueueFull() { 
    boolean ret = delegate.writeQueueFull();
    return ret;
  }

  private io.vertx.axle.core.http.HttpServerResponse __exceptionHandler(Handler<Throwable> handler) { 
    delegate.exceptionHandler(handler);
    return this;
  }

  public io.vertx.axle.core.http.HttpServerResponse exceptionHandler(Consumer<Throwable> handler) {
    return __exceptionHandler(handler != null ? handler::accept : null);
  }

  /**
   * Same as {@link io.vertx.axle.core.http.HttpServerResponse#write} but with an <code>handler</code> called when the operation completes
   * @param data 
   * @param handler 
   * @return 
   */
  private io.vertx.axle.core.http.HttpServerResponse __write(io.vertx.axle.core.buffer.Buffer data, Handler<AsyncResult<Void>> handler) { 
    delegate.write(data.getDelegate(), handler);
    return this;
  }

  /**
   * Same as {@link io.vertx.axle.core.http.HttpServerResponse#write} but with an <code>handler</code> called when the operation completes
   * @param data 
   * @return 
   */
  public CompletionStage<Void> write(io.vertx.axle.core.buffer.Buffer data) { 
    return io.vertx.axle.AsyncResultCompletionStage.toCompletionStage(handler -> {
      __write(data, handler);
    });
  }

  public io.vertx.axle.core.http.HttpServerResponse setWriteQueueMaxSize(int maxSize) { 
    delegate.setWriteQueueMaxSize(maxSize);
    return this;
  }

  private io.vertx.axle.core.http.HttpServerResponse __drainHandler(Handler<Void> handler) { 
    delegate.drainHandler(handler);
    return this;
  }

  public io.vertx.axle.core.http.HttpServerResponse drainHandler(Consumer<Void> handler) {
    return __drainHandler(handler != null ? handler::accept : null);
  }

  /**
   * @return the HTTP status code of the response. The default is <code>200</code> representing <code>OK</code>.
   */
  public int getStatusCode() { 
    int ret = delegate.getStatusCode();
    return ret;
  }

  /**
   * Set the status code. If the status message hasn't been explicitly set, a default status message corresponding
   * to the code will be looked-up and used.
   * @param statusCode 
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.axle.core.http.HttpServerResponse setStatusCode(int statusCode) { 
    delegate.setStatusCode(statusCode);
    return this;
  }

  /**
   * @return the HTTP status message of the response. If this is not specified a default value will be used depending on what {@link io.vertx.axle.core.http.HttpServerResponse#setStatusCode} has been set to.
   */
  public String getStatusMessage() { 
    String ret = delegate.getStatusMessage();
    return ret;
  }

  /**
   * Set the status message
   * @param statusMessage 
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.axle.core.http.HttpServerResponse setStatusMessage(String statusMessage) { 
    delegate.setStatusMessage(statusMessage);
    return this;
  }

  /**
   * If <code>chunked</code> is <code>true</code>, this response will use HTTP chunked encoding, and each call to write to the body
   * will correspond to a new HTTP chunk sent on the wire.
   * <p>
   * If chunked encoding is used the HTTP header <code>Transfer-Encoding</code> with a value of <code>Chunked</code> will be
   * automatically inserted in the response.
   * <p>
   * If <code>chunked</code> is <code>false</code>, this response will not use HTTP chunked encoding, and therefore the total size
   * of any data that is written in the respone body must be set in the <code>Content-Length</code> header <b>before</b> any
   * data is written out.
   * <p>
   * An HTTP chunked response is typically used when you do not know the total size of the request body up front.
   * @param chunked 
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.axle.core.http.HttpServerResponse setChunked(boolean chunked) { 
    delegate.setChunked(chunked);
    return this;
  }

  /**
   * @return is the response chunked?
   */
  public boolean isChunked() { 
    boolean ret = delegate.isChunked();
    return ret;
  }

  /**
   * @return The HTTP headers
   */
  public io.vertx.axle.core.MultiMap headers() { 
    if (cached_0 != null) {
      return cached_0;
    }
    io.vertx.axle.core.MultiMap ret = io.vertx.axle.core.MultiMap.newInstance(delegate.headers());
    cached_0 = ret;
    return ret;
  }

  /**
   * Put an HTTP header
   * @param name the header name
   * @param value the header value.
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.axle.core.http.HttpServerResponse putHeader(String name, String value) { 
    delegate.putHeader(name, value);
    return this;
  }

  /**
   * @return The HTTP trailers
   */
  public io.vertx.axle.core.MultiMap trailers() { 
    if (cached_1 != null) {
      return cached_1;
    }
    io.vertx.axle.core.MultiMap ret = io.vertx.axle.core.MultiMap.newInstance(delegate.trailers());
    cached_1 = ret;
    return ret;
  }

  /**
   * Put an HTTP trailer
   * @param name the trailer name
   * @param value the trailer value
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.axle.core.http.HttpServerResponse putTrailer(String name, String value) { 
    delegate.putTrailer(name, value);
    return this;
  }

  /**
   * Set a close handler for the response, this is called when the underlying connection is closed and the response
   * was still using the connection.
   * <p>
   * For HTTP/1.x it is called when the connection is closed before <code>end()</code> is called, therefore it is not
   * guaranteed to be called.
   * <p>
   * For HTTP/2 it is called when the related stream is closed, and therefore it will be always be called.
   * @param handler the handler
   * @return a reference to this, so the API can be used fluently
   */
  private io.vertx.axle.core.http.HttpServerResponse __closeHandler(Handler<Void> handler) { 
    delegate.closeHandler(handler);
    return this;
  }

  /**
   * Set a close handler for the response, this is called when the underlying connection is closed and the response
   * was still using the connection.
   * <p>
   * For HTTP/1.x it is called when the connection is closed before <code>end()</code> is called, therefore it is not
   * guaranteed to be called.
   * <p>
   * For HTTP/2 it is called when the related stream is closed, and therefore it will be always be called.
   * @param handler the handler
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.axle.core.http.HttpServerResponse closeHandler(Consumer<Void> handler) {
    return __closeHandler(handler != null ? handler::accept : null);
  }

  /**
   * Set an end handler for the response. This will be called when the response is disposed to allow consistent cleanup
   * of the response.
   * @param handler the handler
   * @return a reference to this, so the API can be used fluently
   */
  private io.vertx.axle.core.http.HttpServerResponse __endHandler(Handler<Void> handler) { 
    delegate.endHandler(handler);
    return this;
  }

  /**
   * Set an end handler for the response. This will be called when the response is disposed to allow consistent cleanup
   * of the response.
   * @param handler the handler
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.axle.core.http.HttpServerResponse endHandler(Consumer<Void> handler) {
    return __endHandler(handler != null ? handler::accept : null);
  }

  /**
   * Same as {@link io.vertx.axle.core.http.HttpServerResponse#write} but with an <code>handler</code> called when the operation completes
   * @param chunk 
   * @param enc 
   * @param handler 
   * @return 
   */
  private io.vertx.axle.core.http.HttpServerResponse __write(String chunk, String enc, Handler<AsyncResult<Void>> handler) { 
    delegate.write(chunk, enc, handler);
    return this;
  }

  /**
   * Same as {@link io.vertx.axle.core.http.HttpServerResponse#write} but with an <code>handler</code> called when the operation completes
   * @param chunk 
   * @param enc 
   * @return 
   */
  public CompletionStage<Void> write(String chunk, String enc) { 
    return io.vertx.axle.AsyncResultCompletionStage.toCompletionStage(handler -> {
      __write(chunk, enc, handler);
    });
  }

  /**
   * Same as {@link io.vertx.axle.core.http.HttpServerResponse#write} but with an <code>handler</code> called when the operation completes
   * @param chunk 
   * @param handler 
   * @return 
   */
  private io.vertx.axle.core.http.HttpServerResponse __write(String chunk, Handler<AsyncResult<Void>> handler) { 
    delegate.write(chunk, handler);
    return this;
  }

  /**
   * Same as {@link io.vertx.axle.core.http.HttpServerResponse#write} but with an <code>handler</code> called when the operation completes
   * @param chunk 
   * @return 
   */
  public CompletionStage<Void> write(String chunk) { 
    return io.vertx.axle.AsyncResultCompletionStage.toCompletionStage(handler -> {
      __write(chunk, handler);
    });
  }

  /**
   * Used to write an interim 100 Continue response to signify that the client should send the rest of the request.
   * Must only be used if the request contains an "Expect:100-Continue" header
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.axle.core.http.HttpServerResponse writeContinue() { 
    delegate.writeContinue();
    return this;
  }

  /**
   * Same as {@link io.vertx.axle.core.http.HttpServerResponse#end} but with an <code>handler</code> called when the operation completes
   * @param chunk 
   * @param handler 
   */
  private void __end(String chunk, Handler<AsyncResult<Void>> handler) { 
    delegate.end(chunk, handler);
  }

  /**
   * Same as {@link io.vertx.axle.core.http.HttpServerResponse#end} but with an <code>handler</code> called when the operation completes
   * @param chunk 
   * @return 
   */
  public CompletionStage<Void> end(String chunk) { 
    return io.vertx.axle.AsyncResultCompletionStage.toCompletionStage(handler -> {
      __end(chunk, handler);
    });
  }

  /**
   * Same as {@link io.vertx.axle.core.http.HttpServerResponse#end} but with an <code>handler</code> called when the operation completes
   * @param chunk 
   * @param enc 
   * @param handler 
   */
  private void __end(String chunk, String enc, Handler<AsyncResult<Void>> handler) { 
    delegate.end(chunk, enc, handler);
  }

  /**
   * Same as {@link io.vertx.axle.core.http.HttpServerResponse#end} but with an <code>handler</code> called when the operation completes
   * @param chunk 
   * @param enc 
   * @return 
   */
  public CompletionStage<Void> end(String chunk, String enc) { 
    return io.vertx.axle.AsyncResultCompletionStage.toCompletionStage(handler -> {
      __end(chunk, enc, handler);
    });
  }

  /**
   * Same as {@link io.vertx.axle.core.http.HttpServerResponse#end} but with an <code>handler</code> called when the operation completes
   * @param chunk 
   * @param handler 
   */
  private void __end(io.vertx.axle.core.buffer.Buffer chunk, Handler<AsyncResult<Void>> handler) { 
    delegate.end(chunk.getDelegate(), handler);
  }

  /**
   * Same as {@link io.vertx.axle.core.http.HttpServerResponse#end} but with an <code>handler</code> called when the operation completes
   * @param chunk 
   * @return 
   */
  public CompletionStage<Void> end(io.vertx.axle.core.buffer.Buffer chunk) { 
    return io.vertx.axle.AsyncResultCompletionStage.toCompletionStage(handler -> {
      __end(chunk, handler);
    });
  }

  /**
   * Like {@link io.vertx.axle.core.http.HttpServerResponse#sendFile} but providing a handler which will be notified once the file has been completely
   * written to the wire.
   * @param filename path to the file to serve
   * @param resultHandler handler that will be called on completion
   * @return a reference to this, so the API can be used fluently
   */
  private io.vertx.axle.core.http.HttpServerResponse __sendFile(String filename, Handler<AsyncResult<Void>> resultHandler) { 
    delegate.sendFile(filename, resultHandler);
    return this;
  }

  /**
   * Like {@link io.vertx.axle.core.http.HttpServerResponse#sendFile} but providing a handler which will be notified once the file has been completely
   * written to the wire.
   * @param filename path to the file to serve
   * @return a reference to this, so the API can be used fluently
   */
  public CompletionStage<Void> sendFile(String filename) { 
    return io.vertx.axle.AsyncResultCompletionStage.toCompletionStage(handler -> {
      __sendFile(filename, handler);
    });
  }

  /**
   * Like {@link io.vertx.axle.core.http.HttpServerResponse#sendFile} but providing a handler which will be notified once the file has been completely
   * written to the wire.
   * @param filename path to the file to serve
   * @param offset the offset to serve from
   * @param resultHandler handler that will be called on completion
   * @return a reference to this, so the API can be used fluently
   */
  private io.vertx.axle.core.http.HttpServerResponse __sendFile(String filename, long offset, Handler<AsyncResult<Void>> resultHandler) { 
    delegate.sendFile(filename, offset, resultHandler);
    return this;
  }

  /**
   * Like {@link io.vertx.axle.core.http.HttpServerResponse#sendFile} but providing a handler which will be notified once the file has been completely
   * written to the wire.
   * @param filename path to the file to serve
   * @param offset the offset to serve from
   * @return a reference to this, so the API can be used fluently
   */
  public CompletionStage<Void> sendFile(String filename, long offset) { 
    return io.vertx.axle.AsyncResultCompletionStage.toCompletionStage(handler -> {
      __sendFile(filename, offset, handler);
    });
  }

  /**
   * Like {@link io.vertx.axle.core.http.HttpServerResponse#sendFile} but providing a handler which will be notified once the file has been
   * completely written to the wire.
   * @param filename path to the file to serve
   * @param offset the offset to serve from
   * @param length the length to serve to
   * @param resultHandler handler that will be called on completion
   * @return a reference to this, so the API can be used fluently
   */
  private io.vertx.axle.core.http.HttpServerResponse __sendFile(String filename, long offset, long length, Handler<AsyncResult<Void>> resultHandler) { 
    delegate.sendFile(filename, offset, length, resultHandler);
    return this;
  }

  /**
   * Like {@link io.vertx.axle.core.http.HttpServerResponse#sendFile} but providing a handler which will be notified once the file has been
   * completely written to the wire.
   * @param filename path to the file to serve
   * @param offset the offset to serve from
   * @param length the length to serve to
   * @return a reference to this, so the API can be used fluently
   */
  public CompletionStage<Void> sendFile(String filename, long offset, long length) { 
    return io.vertx.axle.AsyncResultCompletionStage.toCompletionStage(handler -> {
      __sendFile(filename, offset, length, handler);
    });
  }

  /**
   * Close the underlying TCP connection corresponding to the request.
   */
  public void close() { 
    delegate.close();
  }

  /**
   * @return has the response already ended?
   */
  public boolean ended() { 
    boolean ret = delegate.ended();
    return ret;
  }

  /**
   * @return has the underlying TCP connection corresponding to the request already been closed?
   */
  public boolean closed() { 
    boolean ret = delegate.closed();
    return ret;
  }

  /**
   * @return have the headers for the response already been written?
   */
  public boolean headWritten() { 
    boolean ret = delegate.headWritten();
    return ret;
  }

  /**
   * Provide a handler that will be called just before the headers are written to the wire.<p>
   * This provides a hook allowing you to add any more headers or do any more operations before this occurs.
   * @param handler the handler
   * @return a reference to this, so the API can be used fluently
   */
  private io.vertx.axle.core.http.HttpServerResponse __headersEndHandler(Handler<Void> handler) { 
    delegate.headersEndHandler(handler);
    return this;
  }

  /**
   * Provide a handler that will be called just before the headers are written to the wire.<p>
   * This provides a hook allowing you to add any more headers or do any more operations before this occurs.
   * @param handler the handler
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.axle.core.http.HttpServerResponse headersEndHandler(Consumer<Void> handler) {
    return __headersEndHandler(handler != null ? handler::accept : null);
  }

  /**
   * Provides a handler that will be called after the last part of the body is written to the wire.
   * The handler is called asynchronously of when the response has been received by the client.
   * This provides a hook allowing you to do more operations once the request has been sent over the wire
   * such as resource cleanup.
   * @param handler the handler
   * @return a reference to this, so the API can be used fluently
   */
  private io.vertx.axle.core.http.HttpServerResponse __bodyEndHandler(Handler<Void> handler) { 
    delegate.bodyEndHandler(handler);
    return this;
  }

  /**
   * Provides a handler that will be called after the last part of the body is written to the wire.
   * The handler is called asynchronously of when the response has been received by the client.
   * This provides a hook allowing you to do more operations once the request has been sent over the wire
   * such as resource cleanup.
   * @param handler the handler
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.axle.core.http.HttpServerResponse bodyEndHandler(Consumer<Void> handler) {
    return __bodyEndHandler(handler != null ? handler::accept : null);
  }

  /**
   * @return the total number of bytes written for the body of the response.
   */
  public long bytesWritten() { 
    long ret = delegate.bytesWritten();
    return ret;
  }

  /**
   * @return the id of the stream of this response,  for HTTP/1.x
   */
  public int streamId() { 
    int ret = delegate.streamId();
    return ret;
  }

  /**
   * Like {@link io.vertx.axle.core.http.HttpServerResponse#push} with no headers.
   * @param method 
   * @param host 
   * @param path 
   * @param handler 
   * @return 
   */
  private io.vertx.axle.core.http.HttpServerResponse __push(HttpMethod method, String host, String path, Handler<AsyncResult<io.vertx.axle.core.http.HttpServerResponse>> handler) { 
    delegate.push(method, host, path, new Handler<AsyncResult<io.vertx.core.http.HttpServerResponse>>() {
      public void handle(AsyncResult<io.vertx.core.http.HttpServerResponse> ar) {
        if (ar.succeeded()) {
          handler.handle(io.vertx.core.Future.succeededFuture(io.vertx.axle.core.http.HttpServerResponse.newInstance(ar.result())));
        } else {
          handler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    });
    return this;
  }

  /**
   * Like {@link io.vertx.axle.core.http.HttpServerResponse#push} with no headers.
   * @param method 
   * @param host 
   * @param path 
   * @return 
   */
  public CompletionStage<io.vertx.axle.core.http.HttpServerResponse> push(HttpMethod method, String host, String path) { 
    return io.vertx.axle.AsyncResultCompletionStage.toCompletionStage(handler -> {
      __push(method, host, path, handler);
    });
  }

  /**
   * Like {@link io.vertx.axle.core.http.HttpServerResponse#push} with the host copied from the current request.
   * @param method 
   * @param path 
   * @param headers 
   * @param handler 
   * @return 
   */
  private io.vertx.axle.core.http.HttpServerResponse __push(HttpMethod method, String path, io.vertx.axle.core.MultiMap headers, Handler<AsyncResult<io.vertx.axle.core.http.HttpServerResponse>> handler) { 
    delegate.push(method, path, headers.getDelegate(), new Handler<AsyncResult<io.vertx.core.http.HttpServerResponse>>() {
      public void handle(AsyncResult<io.vertx.core.http.HttpServerResponse> ar) {
        if (ar.succeeded()) {
          handler.handle(io.vertx.core.Future.succeededFuture(io.vertx.axle.core.http.HttpServerResponse.newInstance(ar.result())));
        } else {
          handler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    });
    return this;
  }

  /**
   * Like {@link io.vertx.axle.core.http.HttpServerResponse#push} with the host copied from the current request.
   * @param method 
   * @param path 
   * @param headers 
   * @return 
   */
  public CompletionStage<io.vertx.axle.core.http.HttpServerResponse> push(HttpMethod method, String path, io.vertx.axle.core.MultiMap headers) { 
    return io.vertx.axle.AsyncResultCompletionStage.toCompletionStage(handler -> {
      __push(method, path, headers, handler);
    });
  }

  /**
   * Like {@link io.vertx.axle.core.http.HttpServerResponse#push} with the host copied from the current request.
   * @param method 
   * @param path 
   * @param handler 
   * @return 
   */
  private io.vertx.axle.core.http.HttpServerResponse __push(HttpMethod method, String path, Handler<AsyncResult<io.vertx.axle.core.http.HttpServerResponse>> handler) { 
    delegate.push(method, path, new Handler<AsyncResult<io.vertx.core.http.HttpServerResponse>>() {
      public void handle(AsyncResult<io.vertx.core.http.HttpServerResponse> ar) {
        if (ar.succeeded()) {
          handler.handle(io.vertx.core.Future.succeededFuture(io.vertx.axle.core.http.HttpServerResponse.newInstance(ar.result())));
        } else {
          handler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    });
    return this;
  }

  /**
   * Like {@link io.vertx.axle.core.http.HttpServerResponse#push} with the host copied from the current request.
   * @param method 
   * @param path 
   * @return 
   */
  public CompletionStage<io.vertx.axle.core.http.HttpServerResponse> push(HttpMethod method, String path) { 
    return io.vertx.axle.AsyncResultCompletionStage.toCompletionStage(handler -> {
      __push(method, path, handler);
    });
  }

  /**
   * Push a response to the client.<p/>
   *
   * The <code>handler</code> will be notified with a <i>success</i> when the push can be sent and with
   * a <i>failure</i> when the client has disabled push or reset the push before it has been sent.<p/>
   *
   * The <code>handler</code> may be queued if the client has reduced the maximum number of streams the server can push
   * concurrently.<p/>
   *
   * Push can be sent only for peer initiated streams and if the response is not ended.
   * @param method the method of the promised request
   * @param host the host of the promised request
   * @param path the path of the promised request
   * @param headers the headers of the promised request
   * @param handler the handler notified when the response can be written
   * @return a reference to this, so the API can be used fluently
   */
  private io.vertx.axle.core.http.HttpServerResponse __push(HttpMethod method, String host, String path, io.vertx.axle.core.MultiMap headers, Handler<AsyncResult<io.vertx.axle.core.http.HttpServerResponse>> handler) { 
    delegate.push(method, host, path, headers.getDelegate(), new Handler<AsyncResult<io.vertx.core.http.HttpServerResponse>>() {
      public void handle(AsyncResult<io.vertx.core.http.HttpServerResponse> ar) {
        if (ar.succeeded()) {
          handler.handle(io.vertx.core.Future.succeededFuture(io.vertx.axle.core.http.HttpServerResponse.newInstance(ar.result())));
        } else {
          handler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    });
    return this;
  }

  /**
   * Push a response to the client.<p/>
   *
   * The <code>handler</code> will be notified with a <i>success</i> when the push can be sent and with
   * a <i>failure</i> when the client has disabled push or reset the push before it has been sent.<p/>
   *
   * The <code>handler</code> may be queued if the client has reduced the maximum number of streams the server can push
   * concurrently.<p/>
   *
   * Push can be sent only for peer initiated streams and if the response is not ended.
   * @param method the method of the promised request
   * @param host the host of the promised request
   * @param path the path of the promised request
   * @param headers the headers of the promised request
   * @return a reference to this, so the API can be used fluently
   */
  public CompletionStage<io.vertx.axle.core.http.HttpServerResponse> push(HttpMethod method, String host, String path, io.vertx.axle.core.MultiMap headers) { 
    return io.vertx.axle.AsyncResultCompletionStage.toCompletionStage(handler -> {
      __push(method, host, path, headers, handler);
    });
  }

  /**
   * Reset this HTTP/2 stream with the error code <code>0</code>.
   */
  public void reset() { 
    delegate.reset();
  }

  /**
   * Reset this HTTP/2 stream with the error <code>code</code>.
   * @param code the error code
   */
  public void reset(long code) { 
    delegate.reset(code);
  }

  /**
   * Write an HTTP/2 frame to the response, allowing to extend the HTTP/2 protocol.<p>
   *
   * The frame is sent immediatly and is not subject to flow control.
   * @param type the 8-bit frame type
   * @param flags the 8-bit frame flags
   * @param payload the frame payload
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.axle.core.http.HttpServerResponse writeCustomFrame(int type, int flags, io.vertx.axle.core.buffer.Buffer payload) { 
    delegate.writeCustomFrame(type, flags, payload.getDelegate());
    return this;
  }

  /**
   * Like {@link io.vertx.axle.core.http.HttpServerResponse#writeCustomFrame} but with an {@link io.vertx.axle.core.http.HttpFrame}.
   * @param frame the frame to write
   * @return 
   */
  public io.vertx.axle.core.http.HttpServerResponse writeCustomFrame(io.vertx.axle.core.http.HttpFrame frame) { 
    delegate.writeCustomFrame(frame.getDelegate());
    return this;
  }

  /**
   * Sets the priority of the associated stream
   * <p/>
   * This is not implemented for HTTP/1.x.
   * @param streamPriority the priority for this request's stream
   * @return 
   */
  public io.vertx.axle.core.http.HttpServerResponse setStreamPriority(StreamPriority streamPriority) { 
    delegate.setStreamPriority(streamPriority);
    return this;
  }

  /**
   * Add a cookie. This will be sent back to the client in the response.
   * @param cookie the cookie
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.axle.core.http.HttpServerResponse addCookie(io.vertx.axle.core.http.Cookie cookie) { 
    delegate.addCookie(cookie.getDelegate());
    return this;
  }

  /**
   * Expire a cookie, notifying a User Agent to remove it from its cookie jar.
   * @param name the name of the cookie
   * @return the cookie, if it existed, or null
   */
  public io.vertx.axle.core.http.Cookie removeCookie(String name) { 
    io.vertx.axle.core.http.Cookie ret = io.vertx.axle.core.http.Cookie.newInstance(delegate.removeCookie(name));
    return ret;
  }

  /**
   * Remove a cookie from the cookie set. If invalidate is true then it will expire a cookie, notifying a User Agent to
   * remove it from its cookie jar.
   * @param name the name of the cookie
   * @param invalidate 
   * @return the cookie, if it existed, or null
   */
  public io.vertx.axle.core.http.Cookie removeCookie(String name, boolean invalidate) { 
    io.vertx.axle.core.http.Cookie ret = io.vertx.axle.core.http.Cookie.newInstance(delegate.removeCookie(name, invalidate));
    return ret;
  }

  /**
   * Like {@link io.vertx.axle.core.http.HttpServerResponse#putHeader} but using CharSequence
   * @param name 
   * @param value 
   * @return 
   */
  public io.vertx.axle.core.http.HttpServerResponse putHeader(CharSequence name, CharSequence value) { 
    delegate.putHeader(name, value);
    return this;
  }

  /**
   * Like {@link io.vertx.axle.core.http.HttpServerResponse#putHeader} but providing multiple values via a String Iterable
   * @param name 
   * @param values 
   * @return 
   */
  public io.vertx.axle.core.http.HttpServerResponse putHeader(String name, Iterable<String> values) { 
    delegate.putHeader(name, values);
    return this;
  }

  /**
   * Like {@link io.vertx.axle.core.http.HttpServerResponse#putHeader} but with CharSequence Iterable
   * @param name 
   * @param values 
   * @return 
   */
  public io.vertx.axle.core.http.HttpServerResponse putHeader(CharSequence name, Iterable<CharSequence> values) { 
    delegate.putHeader(name, values);
    return this;
  }

  /**
   * Like {@link io.vertx.axle.core.http.HttpServerResponse#putTrailer} but using CharSequence
   * @param name 
   * @param value 
   * @return 
   */
  public io.vertx.axle.core.http.HttpServerResponse putTrailer(CharSequence name, CharSequence value) { 
    delegate.putTrailer(name, value);
    return this;
  }

  /**
   * Like {@link io.vertx.axle.core.http.HttpServerResponse#putTrailer} but providing multiple values via a String Iterable
   * @param name 
   * @param values 
   * @return 
   */
  public io.vertx.axle.core.http.HttpServerResponse putTrailer(String name, Iterable<String> values) { 
    delegate.putTrailer(name, values);
    return this;
  }

  /**
   * Like {@link io.vertx.axle.core.http.HttpServerResponse#putTrailer} but with CharSequence Iterable
   * @param name 
   * @param value 
   * @return 
   */
  public io.vertx.axle.core.http.HttpServerResponse putTrailer(CharSequence name, Iterable<CharSequence> value) { 
    delegate.putTrailer(name, value);
    return this;
  }

  private io.vertx.axle.core.MultiMap cached_0;
  private io.vertx.axle.core.MultiMap cached_1;

  public static  HttpServerResponse newInstance(io.vertx.core.http.HttpServerResponse arg) {
    return arg != null ? new HttpServerResponse(arg) : null;
  }
}
