/*
 * Copyright (c) 2011-2022 Contributors to the Eclipse Foundation
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0, or the Apache License, Version 2.0
 * which is available at https://www.apache.org/licenses/LICENSE-2.0.
 *
 * SPDX-License-Identifier: EPL-2.0 OR Apache-2.0
 */
package io.vertx.grpc.common.impl;

import io.grpc.Decompressor;
import io.grpc.KnownLength;
import io.grpc.MethodDescriptor;
import io.netty.buffer.ByteBufInputStream;
import io.vertx.core.buffer.Buffer;
import io.vertx.core.buffer.impl.BufferImpl;
import io.vertx.grpc.common.GrpcMessage;
import io.vertx.grpc.common.GrpcMessageDecoder;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;

public class BridgeMessageDecoder<T> implements GrpcMessageDecoder<T> {

  private MethodDescriptor.Marshaller<T> marshaller;
  private Decompressor decompressor;

  public BridgeMessageDecoder(MethodDescriptor.Marshaller<T> marshaller, Decompressor decompressor) {
    this.marshaller = marshaller;
    this.decompressor = decompressor;
  }

  private static class KnownLengthStream extends ByteBufInputStream implements KnownLength {
    public KnownLengthStream(Buffer buffer) {
      super(((BufferImpl)buffer).getByteBuf(), buffer.length());
    }

    @Override
    public void close() {
      try {
        super.close();
      } catch (IOException ignore) {
      }
    }
  }

  @Override
  public T decode(GrpcMessage msg) {
    try (KnownLengthStream kls = new KnownLengthStream(msg.payload())) {
      if (msg.encoding().equals("identity")) {
        return marshaller.parse(kls);
      } else {
        try (InputStream in = decompressor.decompress(kls)) {
          return marshaller.parse(in);
        } catch (IOException e) {
          throw new RuntimeException(e);
        }
      }
    }
  }
}
