/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.db;

import com.google.common.collect.Iterables;
import java.net.InetAddress;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.apache.cassandra.config.CFMetaData;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.config.ReadRepairDecision;
import org.apache.cassandra.db.Table;
import org.apache.cassandra.exceptions.InvalidRequestException;
import org.apache.cassandra.exceptions.UnavailableException;
import org.apache.cassandra.locator.AbstractReplicationStrategy;
import org.apache.cassandra.locator.NetworkTopologyStrategy;
import org.apache.cassandra.transport.ProtocolException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public enum ConsistencyLevel {
    ANY(0),
    ONE(1),
    TWO(2),
    THREE(3),
    QUORUM(4),
    ALL(5),
    LOCAL_QUORUM(6, true),
    EACH_QUORUM(7),
    LOCAL_ONE(10, true);

    private static final Logger logger;
    public final int code;
    private final boolean isDCLocal;
    private static final ConsistencyLevel[] codeIdx;

    private ConsistencyLevel(int code) {
        this(code, false);
    }

    private ConsistencyLevel(int code, boolean isDCLocal) {
        this.code = code;
        this.isDCLocal = isDCLocal;
    }

    public static ConsistencyLevel fromCode(int code) {
        if (code < 0 || code >= codeIdx.length) {
            throw new ProtocolException(String.format("Unknown code %d for a consistency level", code));
        }
        return codeIdx[code];
    }

    private int quorumFor(Table table) {
        return table.getReplicationStrategy().getReplicationFactor() / 2 + 1;
    }

    private int localQuorumFor(Table table, String dc) {
        return table.getReplicationStrategy() instanceof NetworkTopologyStrategy ? ((NetworkTopologyStrategy)table.getReplicationStrategy()).getReplicationFactor(dc) / 2 + 1 : this.quorumFor(table);
    }

    public int blockFor(Table table) {
        switch (this) {
            case ONE: 
            case LOCAL_ONE: {
                return 1;
            }
            case ANY: {
                return 1;
            }
            case TWO: {
                return 2;
            }
            case THREE: {
                return 3;
            }
            case QUORUM: {
                return this.quorumFor(table);
            }
            case ALL: {
                return table.getReplicationStrategy().getReplicationFactor();
            }
            case LOCAL_QUORUM: {
                return this.localQuorumFor(table, DatabaseDescriptor.getLocalDataCenter());
            }
            case EACH_QUORUM: {
                if (table.getReplicationStrategy() instanceof NetworkTopologyStrategy) {
                    NetworkTopologyStrategy strategy = (NetworkTopologyStrategy)table.getReplicationStrategy();
                    int n = 0;
                    for (String dc : strategy.getDatacenters()) {
                        n += this.localQuorumFor(table, dc);
                    }
                    return n;
                }
                return this.quorumFor(table);
            }
        }
        throw new UnsupportedOperationException("Invalid consistency level: " + this.toString());
    }

    public boolean isDatacenterLocal() {
        return this.isDCLocal;
    }

    private boolean isLocal(InetAddress endpoint) {
        return DatabaseDescriptor.getLocalDataCenter().equals(DatabaseDescriptor.getEndpointSnitch().getDatacenter(endpoint));
    }

    private int countLocalEndpoints(Iterable<InetAddress> liveEndpoints) {
        int count = 0;
        for (InetAddress endpoint : liveEndpoints) {
            if (!this.isLocal(endpoint)) continue;
            ++count;
        }
        return count;
    }

    private Map<String, Integer> countPerDCEndpoints(Table table, Iterable<InetAddress> liveEndpoints) {
        NetworkTopologyStrategy strategy = (NetworkTopologyStrategy)table.getReplicationStrategy();
        HashMap<String, Integer> dcEndpoints = new HashMap<String, Integer>();
        for (String dc : strategy.getDatacenters()) {
            dcEndpoints.put(dc, 0);
        }
        for (InetAddress endpoint : liveEndpoints) {
            String dc = DatabaseDescriptor.getEndpointSnitch().getDatacenter(endpoint);
            dcEndpoints.put(dc, (Integer)dcEndpoints.get(dc) + 1);
        }
        return dcEndpoints;
    }

    public List<InetAddress> filterForQuery(Table table, List<InetAddress> liveEndpoints) {
        return this.filterForQuery(table, liveEndpoints, ReadRepairDecision.NONE);
    }

    public List<InetAddress> filterForQuery(Table table, List<InetAddress> liveEndpoints, ReadRepairDecision readRepair) {
        if (this.isDCLocal) {
            Collections.sort(liveEndpoints, DatabaseDescriptor.getLocalComparator());
        }
        switch (readRepair) {
            case NONE: {
                return liveEndpoints.subList(0, Math.min(liveEndpoints.size(), this.blockFor(table)));
            }
            case GLOBAL: {
                return liveEndpoints;
            }
            case DC_LOCAL: {
                ArrayList<InetAddress> local = new ArrayList<InetAddress>();
                ArrayList<InetAddress> other = new ArrayList<InetAddress>();
                for (InetAddress add : liveEndpoints) {
                    if (this.isLocal(add)) {
                        local.add(add);
                        continue;
                    }
                    other.add(add);
                }
                int blockFor = this.blockFor(table);
                if (local.size() < blockFor) {
                    local.addAll(other.subList(0, Math.min(blockFor - local.size(), other.size())));
                }
                return local;
            }
        }
        throw new AssertionError();
    }

    public boolean isSufficientLiveNodes(Table table, Iterable<InetAddress> liveEndpoints) {
        switch (this) {
            case ANY: {
                return true;
            }
            case LOCAL_ONE: {
                return this.countLocalEndpoints(liveEndpoints) >= 1;
            }
            case LOCAL_QUORUM: {
                return this.countLocalEndpoints(liveEndpoints) >= this.blockFor(table);
            }
            case EACH_QUORUM: {
                if (!(table.getReplicationStrategy() instanceof NetworkTopologyStrategy)) break;
                for (Map.Entry<String, Integer> entry : this.countPerDCEndpoints(table, liveEndpoints).entrySet()) {
                    if (entry.getValue() >= this.localQuorumFor(table, entry.getKey())) continue;
                    return false;
                }
                return true;
            }
        }
        return Iterables.size(liveEndpoints) >= this.blockFor(table);
    }

    public void assureSufficientLiveNodes(Table table, Iterable<InetAddress> liveEndpoints) throws UnavailableException {
        int blockFor = this.blockFor(table);
        switch (this) {
            case ANY: {
                break;
            }
            case LOCAL_ONE: {
                if (this.countLocalEndpoints(liveEndpoints) != 0) break;
                throw new UnavailableException(this, 1, 0);
            }
            case LOCAL_QUORUM: {
                int localLive = this.countLocalEndpoints(liveEndpoints);
                if (localLive >= blockFor) break;
                if (logger.isDebugEnabled()) {
                    StringBuilder builder = new StringBuilder("Local replicas [");
                    for (InetAddress endpoint : liveEndpoints) {
                        if (!this.isLocal(endpoint)) continue;
                        builder.append(endpoint).append(",");
                    }
                    builder.append("] are insufficient to satisfy LOCAL_QUORUM requirement of ").append(blockFor).append(" live nodes in '").append(DatabaseDescriptor.getLocalDataCenter()).append("'");
                    logger.debug(builder.toString());
                }
                throw new UnavailableException(this, blockFor, localLive);
            }
            case EACH_QUORUM: {
                if (table.getReplicationStrategy() instanceof NetworkTopologyStrategy) {
                    for (Map.Entry<String, Integer> entry : this.countPerDCEndpoints(table, liveEndpoints).entrySet()) {
                        int dcBlockFor = this.localQuorumFor(table, entry.getKey());
                        int dcLive = entry.getValue();
                        if (dcLive >= dcBlockFor) continue;
                        throw new UnavailableException(this, dcBlockFor, dcLive);
                    }
                    break;
                }
            }
            default: {
                int live = Iterables.size(liveEndpoints);
                if (live >= blockFor) break;
                logger.debug("Live nodes {} do not satisfy ConsistencyLevel ({} required)", (Object)Iterables.toString(liveEndpoints), (Object)blockFor);
                throw new UnavailableException(this, blockFor, live);
            }
        }
    }

    public void validateForRead(String table) throws InvalidRequestException {
        switch (this) {
            case ANY: {
                throw new InvalidRequestException("ANY ConsistencyLevel is only supported for writes");
            }
            case EACH_QUORUM: {
                throw new InvalidRequestException("EACH_QUORUM ConsistencyLevel is only supported for writes");
            }
        }
    }

    public void validateForWrite(String table) throws InvalidRequestException {
    }

    public void validateCounterForWrite(CFMetaData metadata) throws InvalidRequestException {
        if (this == ANY) {
            throw new InvalidRequestException("Consistency level ANY is not yet supported for counter columnfamily " + metadata.cfName);
        }
        if (!metadata.getReplicateOnWrite() && this != ONE && this != LOCAL_ONE) {
            throw new InvalidRequestException("cannot achieve CL > CL.ONE without replicate_on_write on columnfamily " + metadata.cfName);
        }
    }

    private void requireNetworkTopologyStrategy(String table) throws InvalidRequestException {
        AbstractReplicationStrategy strategy = Table.open(table).getReplicationStrategy();
        if (!(strategy instanceof NetworkTopologyStrategy)) {
            throw new InvalidRequestException(String.format("consistency level %s not compatible with replication strategy (%s)", new Object[]{this, strategy.getClass().getName()}));
        }
    }

    static {
        logger = LoggerFactory.getLogger(ConsistencyLevel.class);
        int maxCode = -1;
        for (ConsistencyLevel cl : ConsistencyLevel.values()) {
            maxCode = Math.max(maxCode, cl.code);
        }
        codeIdx = new ConsistencyLevel[maxCode + 1];
        for (ConsistencyLevel cl : ConsistencyLevel.values()) {
            if (codeIdx[cl.code] != null) {
                throw new IllegalStateException("Duplicate code");
            }
            ConsistencyLevel.codeIdx[cl.code] = cl;
        }
    }
}

