/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.db;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Predicate;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Iterables;
import com.google.common.collect.Sets;
import com.google.common.primitives.Longs;
import com.google.common.util.concurrent.Uninterruptibles;
import java.io.File;
import java.io.FileFilter;
import java.io.IOError;
import java.io.IOException;
import java.nio.file.FileVisitResult;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import org.apache.cassandra.config.CFMetaData;
import org.apache.cassandra.config.Config;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.db.BlacklistedDirectories;
import org.apache.cassandra.io.FSError;
import org.apache.cassandra.io.FSWriteError;
import org.apache.cassandra.io.sstable.Component;
import org.apache.cassandra.io.sstable.Descriptor;
import org.apache.cassandra.io.sstable.SSTable;
import org.apache.cassandra.io.sstable.SSTableDeletingTask;
import org.apache.cassandra.io.util.FileUtils;
import org.apache.cassandra.service.StorageService;
import org.apache.cassandra.utils.ByteBufferUtil;
import org.apache.cassandra.utils.Pair;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Directories {
    private static final Logger logger = LoggerFactory.getLogger(Directories.class);
    public static final String BACKUPS_SUBDIR = "backups";
    public static final String SNAPSHOT_SUBDIR = "snapshots";
    public static final String SECONDARY_INDEX_NAME_SEPARATOR = ".";
    public static final DataDirectory[] dataDirectories;
    private final CFMetaData metadata;
    private final File[] dataPaths;

    public static boolean verifyFullPermissions(File dir, String dataDir) {
        if (!dir.isDirectory()) {
            logger.error("Not a directory {}", (Object)dataDir);
            return false;
        }
        if (!FileAction.hasPrivilege(dir, FileAction.X)) {
            logger.error("Doesn't have execute permissions for {} directory", (Object)dataDir);
            return false;
        }
        if (!FileAction.hasPrivilege(dir, FileAction.R)) {
            logger.error("Doesn't have read permissions for {} directory", (Object)dataDir);
            return false;
        }
        if (dir.exists() && !FileAction.hasPrivilege(dir, FileAction.W)) {
            logger.error("Doesn't have write permissions for {} directory", (Object)dataDir);
            return false;
        }
        return true;
    }

    public Directories(CFMetaData metadata) {
        this.metadata = metadata;
        String cfId = ByteBufferUtil.bytesToHex(ByteBufferUtil.bytes(metadata.cfId));
        int idx = metadata.cfName.indexOf(SECONDARY_INDEX_NAME_SEPARATOR);
        String directoryName = idx >= 0 ? metadata.cfName.substring(0, idx) + "-" + cfId + File.separator + metadata.cfName.substring(idx) : metadata.cfName + "-" + cfId;
        this.dataPaths = new File[dataDirectories.length];
        for (int i = 0; i < dataDirectories.length; ++i) {
            this.dataPaths[i] = new File(Directories.dataDirectories[i].location, Directories.join(metadata.ksName, this.metadata.cfName));
        }
        boolean olderDirectoryExists = Iterables.any(Arrays.asList(this.dataPaths), (Predicate)new Predicate<File>(){

            public boolean apply(File file) {
                return file.exists();
            }
        });
        if (!olderDirectoryExists) {
            for (int i = 0; i < dataDirectories.length; ++i) {
                this.dataPaths[i] = new File(Directories.dataDirectories[i].location, Directories.join(metadata.ksName, directoryName));
            }
        }
        for (File dir : this.dataPaths) {
            try {
                FileUtils.createDirectory(dir);
            }
            catch (FSError e) {
                logger.error("Failed to create {} directory", (Object)dir);
                FileUtils.handleFSError(e);
            }
        }
    }

    public File getLocationForDisk(DataDirectory dataDirectory) {
        for (File dir : this.dataPaths) {
            if (!dir.getAbsolutePath().startsWith(dataDirectory.location.getAbsolutePath())) continue;
            return dir;
        }
        return null;
    }

    public Descriptor find(String filename) {
        for (File dir : this.dataPaths) {
            if (!new File(dir, filename).exists()) continue;
            return (Descriptor)Descriptor.fromFilename((File)dir, (String)filename).left;
        }
        return null;
    }

    public File getDirectoryForNewSSTables() {
        File path = this.getWriteableLocationAsFile();
        if (!(path != null || DatabaseDescriptor.getDiskAccessMode() != Config.DiskAccessMode.mmap && DatabaseDescriptor.getIndexAccessMode() != Config.DiskAccessMode.mmap || FileUtils.isCleanerAvailable())) {
            logger.info("Forcing GC to free up disk space.  Upgrade to the Oracle JVM to avoid this");
            StorageService.instance.requestGC();
            SSTableDeletingTask.rescheduleFailedTasks();
            Uninterruptibles.sleepUninterruptibly((long)10L, (TimeUnit)TimeUnit.SECONDS);
            path = this.getWriteableLocationAsFile();
        }
        return path;
    }

    public File getWriteableLocationAsFile() {
        return this.getLocationForDisk(this.getWriteableLocation());
    }

    public DataDirectory getWriteableLocation() {
        ArrayList<DataDirectory> candidates = new ArrayList<DataDirectory>();
        for (DataDirectory dataDir : dataDirectories) {
            if (BlacklistedDirectories.isUnwritable(this.getLocationForDisk(dataDir))) continue;
            candidates.add(dataDir);
        }
        if (candidates.isEmpty()) {
            throw new IOError(new IOException("All configured data directories have been blacklisted as unwritable for erroring out"));
        }
        Collections.sort(candidates);
        Collections.sort(candidates, new Comparator<DataDirectory>(){

            @Override
            public int compare(DataDirectory a, DataDirectory b) {
                return a.currentTasks.get() - b.currentTasks.get();
            }
        });
        return (DataDirectory)candidates.get(0);
    }

    public static File getSnapshotDirectory(Descriptor desc, String snapshotName) {
        return Directories.getSnapshotDirectory(desc.directory, snapshotName);
    }

    public static File getSnapshotDirectory(File location, String snapshotName) {
        if (location.getName().startsWith(SECONDARY_INDEX_NAME_SEPARATOR)) {
            return Directories.getOrCreate(location.getParentFile(), SNAPSHOT_SUBDIR, snapshotName, location.getName());
        }
        return Directories.getOrCreate(location, SNAPSHOT_SUBDIR, snapshotName);
    }

    public File getSnapshotManifestFile(String snapshotName) {
        return new File(this.getDirectoryForNewSSTables(), Directories.join(SNAPSHOT_SUBDIR, snapshotName, "manifest.json"));
    }

    public static File getBackupsDirectory(Descriptor desc) {
        return Directories.getBackupsDirectory(desc.directory);
    }

    public static File getBackupsDirectory(File location) {
        if (location.getName().startsWith(SECONDARY_INDEX_NAME_SEPARATOR)) {
            return Directories.getOrCreate(location.getParentFile(), BACKUPS_SUBDIR, location.getName());
        }
        return Directories.getOrCreate(location, BACKUPS_SUBDIR);
    }

    public SSTableLister sstableLister() {
        return new SSTableLister();
    }

    public Map<String, Pair<Long, Long>> getSnapshotDetails() {
        HashMap<String, Pair<Long, Long>> snapshotSpaceMap = new HashMap<String, Pair<Long, Long>>();
        for (File dir : this.dataPaths) {
            File[] snapshots;
            File snapshotDir = new File(dir, SNAPSHOT_SUBDIR);
            if (!snapshotDir.exists() || !snapshotDir.isDirectory() || (snapshots = snapshotDir.listFiles()) == null) continue;
            for (File snapshot : snapshots) {
                if (!snapshot.isDirectory()) continue;
                long sizeOnDisk = FileUtils.folderSize(snapshot);
                long trueSize = this.getTrueAllocatedSizeIn(snapshot);
                Pair<Long, Long> spaceUsed = (Pair<Long, Long>)snapshotSpaceMap.get(snapshot.getName());
                spaceUsed = spaceUsed == null ? Pair.create(sizeOnDisk, trueSize) : Pair.create((Long)spaceUsed.left + sizeOnDisk, (Long)spaceUsed.right + trueSize);
                snapshotSpaceMap.put(snapshot.getName(), spaceUsed);
            }
        }
        return snapshotSpaceMap;
    }

    public boolean snapshotExists(String snapshotName) {
        for (File dir : this.dataPaths) {
            File snapshotDir = new File(dir, Directories.join(SNAPSHOT_SUBDIR, snapshotName));
            if (!snapshotDir.exists()) continue;
            return true;
        }
        return false;
    }

    public static void clearSnapshot(String snapshotName, List<File> snapshotDirectories) {
        String tag = snapshotName == null ? "" : snapshotName;
        for (File dir : snapshotDirectories) {
            File snapshotDir = new File(dir, Directories.join(SNAPSHOT_SUBDIR, tag));
            if (!snapshotDir.exists()) continue;
            if (logger.isDebugEnabled()) {
                logger.debug("Removing snapshot directory {}", (Object)snapshotDir);
            }
            FileUtils.deleteRecursive(snapshotDir);
        }
    }

    public long snapshotCreationTime(String snapshotName) {
        for (File dir : this.dataPaths) {
            File snapshotDir = new File(dir, Directories.join(SNAPSHOT_SUBDIR, snapshotName));
            if (!snapshotDir.exists()) continue;
            return snapshotDir.lastModified();
        }
        throw new RuntimeException("Snapshot " + snapshotName + " doesn't exist");
    }

    public long trueSnapshotsSize() {
        long result = 0L;
        for (File dir : this.dataPaths) {
            result += this.getTrueAllocatedSizeIn(new File(dir, Directories.join(SNAPSHOT_SUBDIR)));
        }
        return result;
    }

    public long getTrueAllocatedSizeIn(File input) {
        if (!input.isDirectory()) {
            return 0L;
        }
        TrueFilesSizeVisitor visitor = new TrueFilesSizeVisitor();
        try {
            Files.walkFileTree(input.toPath(), visitor);
        }
        catch (IOException e) {
            logger.error("Could not calculate the size of {}. {}", (Object)input, (Object)e);
        }
        return visitor.getAllocatedSize();
    }

    public static List<File> getKSChildDirectories(String ksName) {
        ArrayList<File> result = new ArrayList<File>();
        for (DataDirectory dataDirectory : dataDirectories) {
            File ksDir = new File(dataDirectory.location, ksName);
            File[] cfDirs = ksDir.listFiles();
            if (cfDirs == null) continue;
            for (File cfDir : cfDirs) {
                if (!cfDir.isDirectory()) continue;
                result.add(cfDir);
            }
        }
        return result;
    }

    public List<File> getCFDirectories() {
        ArrayList<File> result = new ArrayList<File>();
        for (File dataDirectory : this.dataPaths) {
            if (!dataDirectory.isDirectory()) continue;
            result.add(dataDirectory);
        }
        return result;
    }

    private static File getOrCreate(File base, String ... subdirs) {
        File dir;
        File file = dir = subdirs == null || subdirs.length == 0 ? base : new File(base, Directories.join(subdirs));
        if (dir.exists()) {
            if (!dir.isDirectory()) {
                throw new AssertionError((Object)String.format("Invalid directory path %s: path exists but is not a directory", dir));
            }
        } else if (!(dir.mkdirs() || dir.exists() && dir.isDirectory())) {
            throw new FSWriteError((Throwable)new IOException("Unable to create directory " + dir), dir);
        }
        return dir;
    }

    private static String join(String ... s) {
        return StringUtils.join((Object[])s, (String)File.separator);
    }

    @VisibleForTesting
    static void overrideDataDirectoriesForTest(String loc) {
        for (int i = 0; i < dataDirectories.length; ++i) {
            Directories.dataDirectories[i] = new DataDirectory(new File(loc));
        }
    }

    @VisibleForTesting
    static void resetDataDirectoriesAfterTest() {
        String[] locations = DatabaseDescriptor.getAllDataFileLocations();
        for (int i = 0; i < locations.length; ++i) {
            Directories.dataDirectories[i] = new DataDirectory(new File(locations[i]));
        }
    }

    static {
        String[] locations = DatabaseDescriptor.getAllDataFileLocations();
        dataDirectories = new DataDirectory[locations.length];
        for (int i = 0; i < locations.length; ++i) {
            Directories.dataDirectories[i] = new DataDirectory(new File(locations[i]));
        }
    }

    private class TrueFilesSizeVisitor
    extends SimpleFileVisitor<Path> {
        private final AtomicLong size = new AtomicLong(0L);
        private final Set<String> visited = Sets.newHashSet();
        private final Set<String> alive;

        public TrueFilesSizeVisitor() {
            ImmutableSet.Builder builder = ImmutableSet.builder();
            for (File file : Directories.this.sstableLister().listFiles()) {
                builder.add((Object)file.getName());
            }
            this.alive = builder.build();
        }

        private boolean isAcceptable(Path file) {
            String fileName = file.toFile().getName();
            Pair<Descriptor, Component> pair = SSTable.tryComponentFromFilename(file.getParent().toFile(), fileName);
            return pair != null && ((Descriptor)pair.left).ksname.equals(((Directories)Directories.this).metadata.ksName) && ((Descriptor)pair.left).cfname.equals(((Directories)Directories.this).metadata.cfName) && !this.visited.contains(fileName) && !this.alive.contains(fileName);
        }

        @Override
        public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
            if (this.isAcceptable(file)) {
                this.size.addAndGet(attrs.size());
                this.visited.add(file.toFile().getName());
            }
            return FileVisitResult.CONTINUE;
        }

        @Override
        public FileVisitResult visitFileFailed(Path file, IOException exc) throws IOException {
            return FileVisitResult.CONTINUE;
        }

        public long getAllocatedSize() {
            return this.size.get();
        }
    }

    public class SSTableLister {
        private boolean skipTemporary;
        private boolean includeBackups;
        private boolean onlyBackups;
        private int nbFiles;
        private final Map<Descriptor, Set<Component>> components = new HashMap<Descriptor, Set<Component>>();
        private boolean filtered;
        private String snapshotName;

        public SSTableLister skipTemporary(boolean b) {
            if (this.filtered) {
                throw new IllegalStateException("list() has already been called");
            }
            this.skipTemporary = b;
            return this;
        }

        public SSTableLister includeBackups(boolean b) {
            if (this.filtered) {
                throw new IllegalStateException("list() has already been called");
            }
            this.includeBackups = b;
            return this;
        }

        public SSTableLister onlyBackups(boolean b) {
            if (this.filtered) {
                throw new IllegalStateException("list() has already been called");
            }
            this.onlyBackups = b;
            this.includeBackups = b;
            return this;
        }

        public SSTableLister snapshots(String sn) {
            if (this.filtered) {
                throw new IllegalStateException("list() has already been called");
            }
            this.snapshotName = sn;
            return this;
        }

        public Map<Descriptor, Set<Component>> list() {
            this.filter();
            return ImmutableMap.copyOf(this.components);
        }

        public List<File> listFiles() {
            this.filter();
            ArrayList<File> l = new ArrayList<File>(this.nbFiles);
            for (Map.Entry<Descriptor, Set<Component>> entry : this.components.entrySet()) {
                for (Component c : entry.getValue()) {
                    l.add(new File(entry.getKey().filenameFor(c)));
                }
            }
            return l;
        }

        private void filter() {
            if (this.filtered) {
                return;
            }
            for (File location : Directories.this.dataPaths) {
                if (BlacklistedDirectories.isUnreadable(location)) continue;
                if (this.snapshotName != null) {
                    Directories.getSnapshotDirectory(location, this.snapshotName).listFiles(this.getFilter());
                    continue;
                }
                if (!this.onlyBackups) {
                    location.listFiles(this.getFilter());
                }
                if (!this.includeBackups) continue;
                Directories.getBackupsDirectory(location).listFiles(this.getFilter());
            }
            this.filtered = true;
        }

        private FileFilter getFilter() {
            return new FileFilter(){

                @Override
                public boolean accept(File file) {
                    if (file.isDirectory()) {
                        return false;
                    }
                    Pair<Descriptor, Component> pair = SSTable.tryComponentFromFilename(file.getParentFile(), file.getName());
                    if (pair == null) {
                        return false;
                    }
                    if (!((Descriptor)pair.left).ksname.equals(((Directories)Directories.this).metadata.ksName) || !((Descriptor)pair.left).cfname.equals(((Directories)Directories.this).metadata.cfName)) {
                        return false;
                    }
                    if (SSTableLister.this.skipTemporary && ((Descriptor)pair.left).type.isTemporary) {
                        return false;
                    }
                    HashSet previous = (HashSet)SSTableLister.this.components.get(pair.left);
                    if (previous == null) {
                        previous = new HashSet();
                        SSTableLister.this.components.put(pair.left, previous);
                    }
                    previous.add(pair.right);
                    SSTableLister.this.nbFiles++;
                    return false;
                }
            };
        }
    }

    public static class DataDirectory
    implements Comparable<DataDirectory> {
        public final File location;
        public final AtomicInteger currentTasks = new AtomicInteger();
        public final AtomicLong estimatedWorkingSize = new AtomicLong();

        public DataDirectory(File location) {
            this.location = location;
        }

        public long getEstimatedAvailableSpace() {
            return this.location.getUsableSpace() - this.estimatedWorkingSize.get();
        }

        @Override
        public int compareTo(DataDirectory o) {
            return -1 * Longs.compare((long)this.getEstimatedAvailableSpace(), (long)o.getEstimatedAvailableSpace());
        }
    }

    public static enum FileAction {
        X,
        W,
        XW,
        R,
        XR,
        RW,
        XRW;


        public static boolean hasPrivilege(File file, FileAction action) {
            boolean privilege = false;
            switch (action) {
                case X: {
                    privilege = file.canExecute();
                    break;
                }
                case W: {
                    privilege = file.canWrite();
                    break;
                }
                case XW: {
                    privilege = file.canExecute() && file.canWrite();
                    break;
                }
                case R: {
                    privilege = file.canRead();
                    break;
                }
                case XR: {
                    privilege = file.canExecute() && file.canRead();
                    break;
                }
                case RW: {
                    privilege = file.canRead() && file.canWrite();
                    break;
                }
                case XRW: {
                    privilege = file.canExecute() && file.canRead() && file.canWrite();
                }
            }
            return privilege;
        }
    }
}

