/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.db.commitlog;

import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.MappedByteBuffer;
import java.nio.channels.FileChannel;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.Iterator;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.atomic.AtomicInteger;
import org.apache.cassandra.config.CFMetaData;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.config.Schema;
import org.apache.cassandra.db.ColumnFamily;
import org.apache.cassandra.db.Mutation;
import org.apache.cassandra.db.commitlog.CommitLog;
import org.apache.cassandra.db.commitlog.CommitLogDescriptor;
import org.apache.cassandra.db.commitlog.ReplayPosition;
import org.apache.cassandra.io.FSWriteError;
import org.apache.cassandra.io.util.FileUtils;
import org.apache.cassandra.utils.CLibrary;
import org.apache.cassandra.utils.PureJavaCrc32;
import org.apache.cassandra.utils.concurrent.OpOrder;
import org.apache.cassandra.utils.concurrent.WaitQueue;
import org.cliffc.high_scale_lib.NonBlockingHashMap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CommitLogSegment {
    private static final Logger logger = LoggerFactory.getLogger(CommitLogSegment.class);
    private static final long idBase;
    private static final AtomicInteger nextId;
    public static final int ENTRY_OVERHEAD_SIZE = 12;
    static final int SYNC_MARKER_SIZE = 8;
    private final OpOrder appendOrder = new OpOrder();
    private final AtomicInteger allocatePosition = new AtomicInteger();
    private volatile int lastSyncedOffset;
    private int discardedTailFrom;
    private final WaitQueue syncComplete = new WaitQueue();
    private final NonBlockingHashMap<UUID, AtomicInteger> cfDirty = new NonBlockingHashMap(1024);
    private final ConcurrentHashMap<UUID, AtomicInteger> cfClean = new ConcurrentHashMap();
    public final long id = CommitLogSegment.getNextId();
    private final File logFile;
    private final RandomAccessFile logFileAccessor;
    private final int fd;
    private final MappedByteBuffer buffer;
    public final CommitLogDescriptor descriptor = new CommitLogDescriptor(this.id);

    static CommitLogSegment freshSegment() {
        return new CommitLogSegment(null);
    }

    static long getNextId() {
        return idBase + (long)nextId.getAndIncrement();
    }

    CommitLogSegment(String filePath) {
        this.logFile = new File(DatabaseDescriptor.getCommitLogLocation(), this.descriptor.fileName());
        boolean isCreating = true;
        try {
            File oldFile;
            if (filePath != null && (oldFile = new File(filePath)).exists()) {
                logger.debug("Re-using discarded CommitLog segment for {} from {}", (Object)this.id, (Object)filePath);
                if (!oldFile.renameTo(this.logFile)) {
                    throw new IOException("Rename from " + filePath + " to " + this.id + " failed");
                }
                isCreating = false;
            }
            this.logFileAccessor = new RandomAccessFile(this.logFile, "rw");
            if (isCreating) {
                logger.debug("Creating new commit log segment {}", (Object)this.logFile.getPath());
            }
            this.logFileAccessor.setLength(DatabaseDescriptor.getCommitLogSegmentSize());
            this.fd = CLibrary.getfd(this.logFileAccessor.getFD());
            this.buffer = this.logFileAccessor.getChannel().map(FileChannel.MapMode.READ_WRITE, 0L, DatabaseDescriptor.getCommitLogSegmentSize());
            CommitLogDescriptor.writeHeader(this.buffer, this.descriptor);
            this.buffer.putInt(16, 0);
            this.buffer.putLong(20, 0L);
            this.allocatePosition.set(24);
            this.lastSyncedOffset = 16;
        }
        catch (IOException e) {
            throw new FSWriteError((Throwable)e, this.logFile);
        }
    }

    Allocation allocate(Mutation mutation, int size) {
        OpOrder.Group opGroup = this.appendOrder.start();
        try {
            int position = this.allocate(size);
            if (position < 0) {
                opGroup.close();
                return null;
            }
            this.markDirty(mutation, position);
            return new Allocation(this, opGroup, position, (ByteBuffer)this.buffer.duplicate().position(position).limit(position + size));
        }
        catch (Throwable t) {
            opGroup.close();
            throw t;
        }
    }

    private int allocate(int size) {
        int next;
        int prev;
        do {
            if ((next = (prev = this.allocatePosition.get()) + size) < this.buffer.capacity()) continue;
            return -1;
        } while (!this.allocatePosition.compareAndSet(prev, next));
        return prev;
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    void discardUnusedTail() {
        try (OpOrder.Group group = this.appendOrder.start();){
            while (true) {
                int next;
                int prev;
                if ((prev = this.allocatePosition.get()) == (next = this.buffer.capacity() + 1)) {
                    return;
                }
                if (this.allocatePosition.compareAndSet(prev, next)) {
                    this.discardedTailFrom = prev;
                    return;
                }
                continue;
                break;
            }
        }
    }

    void waitForModifications() {
        this.appendOrder.awaitNewBarrier();
    }

    synchronized void sync() {
        try {
            if (this.allocatePosition.get() <= this.lastSyncedOffset + 8) {
                return;
            }
            int nextMarker = this.allocate(8);
            boolean close = false;
            if (nextMarker < 0) {
                this.discardUnusedTail();
                close = true;
                this.waitForModifications();
                nextMarker = this.discardedTailFrom < this.buffer.capacity() - 8 ? this.discardedTailFrom : this.buffer.capacity();
            } else {
                this.waitForModifications();
            }
            assert (nextMarker > this.lastSyncedOffset);
            int offset = this.lastSyncedOffset;
            PureJavaCrc32 crc = new PureJavaCrc32();
            crc.updateInt((int)(this.id & 0xFFFFFFFFL));
            crc.updateInt((int)(this.id >>> 32));
            crc.updateInt(offset);
            this.buffer.putInt(offset, nextMarker);
            this.buffer.putInt(offset + 4, crc.getCrc());
            if (nextMarker < this.buffer.capacity()) {
                this.buffer.putInt(nextMarker, 0);
                this.buffer.putInt(nextMarker + 4, 0);
            }
            this.buffer.force();
            if (close) {
                nextMarker = this.buffer.capacity();
            }
            this.lastSyncedOffset = nextMarker;
            this.syncComplete.signalAll();
            CLibrary.trySkipCache(this.fd, (long)offset, nextMarker);
            if (close) {
                this.close();
            }
        }
        catch (Exception e) {
            throw new FSWriteError((Throwable)e, this.getPath());
        }
    }

    public boolean isStillAllocating() {
        return this.allocatePosition.get() < this.buffer.capacity();
    }

    void delete() {
        FileUtils.deleteWithConfirm(this.logFile);
    }

    CommitLogSegment recycle() {
        try {
            this.sync();
        }
        catch (FSWriteError e) {
            logger.error("I/O error flushing {} {}", (Object)this, (Object)e.getMessage());
            throw e;
        }
        this.close();
        return new CommitLogSegment(this.getPath());
    }

    public ReplayPosition getContext() {
        return new ReplayPosition(this.id, this.allocatePosition.get());
    }

    public String getPath() {
        return this.logFile.getPath();
    }

    public String getName() {
        return this.logFile.getName();
    }

    void waitForFinalSync() {
        WaitQueue.Signal signal;
        while (true) {
            signal = this.syncComplete.register();
            if (this.lastSyncedOffset >= this.buffer.capacity()) break;
            signal.awaitUninterruptibly();
        }
        signal.cancel();
    }

    void close() {
        try {
            if (FileUtils.isCleanerAvailable()) {
                FileUtils.clean(this.buffer);
            }
            this.logFileAccessor.close();
        }
        catch (IOException e) {
            throw new FSWriteError((Throwable)e, this.getPath());
        }
    }

    void markDirty(Mutation mutation, int allocatedPosition) {
        for (ColumnFamily columnFamily : mutation.getColumnFamilies()) {
            CFMetaData cfm = columnFamily.metadata();
            if (cfm.isPurged()) {
                logger.error("Attempted to write commit log entry for unrecognized table: {}", (Object)columnFamily.id());
                continue;
            }
            CommitLogSegment.ensureAtleast(this.cfDirty, cfm.cfId, allocatedPosition);
        }
    }

    public synchronized void markClean(UUID cfId, ReplayPosition context) {
        if (!this.cfDirty.containsKey((Object)cfId)) {
            return;
        }
        if (context.segment == this.id) {
            this.markClean(cfId, context.position);
        } else if (context.segment > this.id) {
            this.markClean(cfId, Integer.MAX_VALUE);
        }
    }

    private void markClean(UUID cfId, int position) {
        CommitLogSegment.ensureAtleast(this.cfClean, cfId, position);
        this.removeCleanFromDirty();
    }

    private static void ensureAtleast(ConcurrentMap<UUID, AtomicInteger> map, UUID cfId, int value) {
        int cur;
        AtomicInteger i2;
        AtomicInteger i = (AtomicInteger)map.get(cfId);
        if (i == null && (i2 = map.putIfAbsent(cfId, i = new AtomicInteger())) != null) {
            i = i2;
        }
        while ((cur = i.get()) <= value && !i.compareAndSet(cur, value)) {
        }
    }

    private void removeCleanFromDirty() {
        if (this.isStillAllocating()) {
            return;
        }
        Iterator<Map.Entry<UUID, AtomicInteger>> iter = this.cfClean.entrySet().iterator();
        while (iter.hasNext()) {
            Map.Entry<UUID, AtomicInteger> clean = iter.next();
            UUID cfId = clean.getKey();
            AtomicInteger cleanPos = clean.getValue();
            AtomicInteger dirtyPos = (AtomicInteger)this.cfDirty.get((Object)cfId);
            if (dirtyPos == null || dirtyPos.intValue() > cleanPos.intValue()) continue;
            this.cfDirty.remove((Object)cfId);
            iter.remove();
        }
    }

    public synchronized Collection<UUID> getDirtyCFIDs() {
        if (this.cfClean.isEmpty() || this.cfDirty.isEmpty()) {
            return this.cfDirty.keySet();
        }
        ArrayList<UUID> r = new ArrayList<UUID>(this.cfDirty.size());
        for (Map.Entry dirty : this.cfDirty.entrySet()) {
            UUID cfId = (UUID)dirty.getKey();
            AtomicInteger dirtyPos = (AtomicInteger)dirty.getValue();
            AtomicInteger cleanPos = this.cfClean.get(cfId);
            if (cleanPos != null && cleanPos.intValue() >= dirtyPos.intValue()) continue;
            r.add((UUID)dirty.getKey());
        }
        return r;
    }

    public synchronized boolean isUnused() {
        if (this.isStillAllocating()) {
            return false;
        }
        this.removeCleanFromDirty();
        return this.cfDirty.isEmpty();
    }

    public boolean contains(ReplayPosition context) {
        return context.segment == this.id;
    }

    public String dirtyString() {
        StringBuilder sb = new StringBuilder();
        for (UUID cfId : this.getDirtyCFIDs()) {
            CFMetaData m = Schema.instance.getCFMetaData(cfId);
            sb.append(m == null ? "<deleted>" : m.cfName).append(" (").append(cfId).append("), ");
        }
        return sb.toString();
    }

    public String toString() {
        return "CommitLogSegment(" + this.getPath() + ')';
    }

    static {
        nextId = new AtomicInteger(1);
        long maxId = Long.MIN_VALUE;
        for (File file : new File(DatabaseDescriptor.getCommitLogLocation()).listFiles()) {
            if (!CommitLogDescriptor.isValid(file.getName())) continue;
            maxId = Math.max(CommitLogDescriptor.fromFileName((String)file.getName()).id, maxId);
        }
        idBase = Math.max(System.currentTimeMillis(), maxId + 1L);
    }

    static class Allocation {
        private final CommitLogSegment segment;
        private final OpOrder.Group appendOp;
        private final int position;
        private final ByteBuffer buffer;

        Allocation(CommitLogSegment segment, OpOrder.Group appendOp, int position, ByteBuffer buffer) {
            this.segment = segment;
            this.appendOp = appendOp;
            this.position = position;
            this.buffer = buffer;
        }

        CommitLogSegment getSegment() {
            return this.segment;
        }

        ByteBuffer getBuffer() {
            return this.buffer;
        }

        void markWritten() {
            this.appendOp.close();
        }

        void awaitDiskSync() {
            while (this.segment.lastSyncedOffset < this.position) {
                WaitQueue.Signal signal = this.segment.syncComplete.register(CommitLog.instance.metrics.waitingOnCommit.time());
                if (this.segment.lastSyncedOffset < this.position) {
                    signal.awaitUninterruptibly();
                    continue;
                }
                signal.cancel();
            }
        }

        public ReplayPosition getReplayPosition() {
            return new ReplayPosition(this.segment.id, this.buffer.limit());
        }
    }

    public static class CommitLogSegmentFileComparator
    implements Comparator<File> {
        @Override
        public int compare(File f, File f2) {
            CommitLogDescriptor desc = CommitLogDescriptor.fromFileName(f.getName());
            CommitLogDescriptor desc2 = CommitLogDescriptor.fromFileName(f2.getName());
            return Long.compare(desc.id, desc2.id);
        }
    }
}

