/*
 * Decompiled with CFR 0.152.
 */
package org.apache.directory.server.kerberos.kdc.authentication;

import java.net.InetAddress;
import java.nio.ByteBuffer;
import java.util.Date;
import java.util.List;
import java.util.Set;
import javax.security.auth.kerberos.KerberosKey;
import javax.security.auth.kerberos.KerberosPrincipal;
import org.apache.directory.server.i18n.I18n;
import org.apache.directory.server.kerberos.kdc.KdcContext;
import org.apache.directory.server.kerberos.kdc.KdcServer;
import org.apache.directory.server.kerberos.kdc.authentication.AuthenticationContext;
import org.apache.directory.server.kerberos.protocol.KerberosDecoder;
import org.apache.directory.server.kerberos.sam.SamException;
import org.apache.directory.server.kerberos.sam.SamSubsystem;
import org.apache.directory.server.kerberos.shared.crypto.encryption.CipherTextHandler;
import org.apache.directory.server.kerberos.shared.crypto.encryption.KeyUsage;
import org.apache.directory.server.kerberos.shared.crypto.encryption.RandomKeyFactory;
import org.apache.directory.server.kerberos.shared.store.PrincipalStore;
import org.apache.directory.server.kerberos.shared.store.PrincipalStoreEntry;
import org.apache.directory.shared.asn1.AbstractAsn1Object;
import org.apache.directory.shared.asn1.EncoderException;
import org.apache.directory.shared.kerberos.KerberosTime;
import org.apache.directory.shared.kerberos.KerberosUtils;
import org.apache.directory.shared.kerberos.codec.types.EncryptionType;
import org.apache.directory.shared.kerberos.codec.types.LastReqType;
import org.apache.directory.shared.kerberos.codec.types.PaDataType;
import org.apache.directory.shared.kerberos.components.ETypeInfo;
import org.apache.directory.shared.kerberos.components.ETypeInfoEntry;
import org.apache.directory.shared.kerberos.components.EncKdcRepPart;
import org.apache.directory.shared.kerberos.components.EncTicketPart;
import org.apache.directory.shared.kerberos.components.EncryptedData;
import org.apache.directory.shared.kerberos.components.EncryptionKey;
import org.apache.directory.shared.kerberos.components.KdcReq;
import org.apache.directory.shared.kerberos.components.LastReq;
import org.apache.directory.shared.kerberos.components.LastReqEntry;
import org.apache.directory.shared.kerberos.components.MethodData;
import org.apache.directory.shared.kerberos.components.PaData;
import org.apache.directory.shared.kerberos.components.PaEncTsEnc;
import org.apache.directory.shared.kerberos.components.PrincipalName;
import org.apache.directory.shared.kerberos.components.TransitedEncoding;
import org.apache.directory.shared.kerberos.exceptions.ErrorType;
import org.apache.directory.shared.kerberos.exceptions.InvalidTicketException;
import org.apache.directory.shared.kerberos.exceptions.KerberosException;
import org.apache.directory.shared.kerberos.flags.KerberosFlag;
import org.apache.directory.shared.kerberos.flags.TicketFlag;
import org.apache.directory.shared.kerberos.flags.TicketFlags;
import org.apache.directory.shared.kerberos.messages.AsRep;
import org.apache.directory.shared.kerberos.messages.EncAsRepPart;
import org.apache.directory.shared.kerberos.messages.KerberosMessage;
import org.apache.directory.shared.kerberos.messages.Ticket;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/*
 * This class specifies class file version 49.0 but uses Java 6 signatures.  Assumed Java 6.
 */
public class AuthenticationService {
    private static final Logger LOG = LoggerFactory.getLogger(AuthenticationService.class);
    private static final CipherTextHandler cipherTextHandler = new CipherTextHandler();
    private static final String SERVICE_NAME = "Authentication Service (AS)";

    public static void execute(AuthenticationContext authContext) throws Exception {
        if (LOG.isDebugEnabled()) {
            AuthenticationService.monitorRequest(authContext);
        }
        authContext.setCipherTextHandler(cipherTextHandler);
        if (authContext.getRequest().getProtocolVersionNumber() != 5) {
            throw new KerberosException(ErrorType.KDC_ERR_BAD_PVNO);
        }
        AuthenticationService.selectEncryptionType(authContext);
        AuthenticationService.getClientEntry(authContext);
        AuthenticationService.verifyPolicy(authContext);
        AuthenticationService.verifySam(authContext);
        AuthenticationService.verifyEncryptedTimestamp(authContext);
        if (authContext.getClientKey() == null) {
            AuthenticationService.verifyEncryptedTimestamp(authContext);
        }
        AuthenticationService.getServerEntry(authContext);
        AuthenticationService.generateTicket(authContext);
        AuthenticationService.buildReply(authContext);
    }

    private static void selectEncryptionType(AuthenticationContext authContext) throws KerberosException, InvalidTicketException {
        AuthenticationContext kdcContext = authContext;
        KdcServer config = kdcContext.getConfig();
        Set requestedTypes = kdcContext.getRequest().getKdcReqBody().getEType();
        EncryptionType bestType = KerberosUtils.getBestEncryptionType((Set)requestedTypes, config.getEncryptionTypes());
        LOG.debug("Session will use encryption type {}.", (Object)bestType);
        if (bestType == null) {
            throw new KerberosException(ErrorType.KDC_ERR_ETYPE_NOSUPP);
        }
        kdcContext.setEncryptionType(bestType);
    }

    private static void getClientEntry(AuthenticationContext authContext) throws KerberosException, InvalidTicketException {
        KerberosPrincipal principal = KerberosUtils.getKerberosPrincipal((PrincipalName)authContext.getRequest().getKdcReqBody().getCName(), (String)authContext.getRequest().getKdcReqBody().getRealm());
        PrincipalStore store = authContext.getStore();
        PrincipalStoreEntry storeEntry = AuthenticationService.getEntry(principal, store, ErrorType.KDC_ERR_C_PRINCIPAL_UNKNOWN);
        authContext.setClientEntry(storeEntry);
    }

    private static void verifyPolicy(AuthenticationContext authContext) throws KerberosException, InvalidTicketException {
        PrincipalStoreEntry entry = authContext.getClientEntry();
        if (entry.isDisabled()) {
            throw new KerberosException(ErrorType.KDC_ERR_CLIENT_REVOKED);
        }
        if (entry.isLockedOut()) {
            throw new KerberosException(ErrorType.KDC_ERR_CLIENT_REVOKED);
        }
        if (entry.getExpiration().getTime() < new Date().getTime()) {
            throw new KerberosException(ErrorType.KDC_ERR_CLIENT_REVOKED);
        }
    }

    private static void verifySam(AuthenticationContext authContext) throws KerberosException, InvalidTicketException {
        LOG.debug("Verifying using SAM subsystem.");
        KdcReq request = authContext.getRequest();
        KdcServer config = authContext.getConfig();
        PrincipalStoreEntry clientEntry = authContext.getClientEntry();
        String clientName = clientEntry.getPrincipal().getName();
        EncryptionKey clientKey = null;
        if (clientEntry.getSamType() != null) {
            List preAuthData;
            if (LOG.isDebugEnabled()) {
                LOG.debug("Entry for client principal {} has a valid SAM type.  Invoking SAM subsystem for pre-authentication.", (Object)clientName);
            }
            if ((preAuthData = request.getPaData()) == null || preAuthData.size() == 0) {
                throw new KerberosException(ErrorType.KDC_ERR_PREAUTH_REQUIRED, AuthenticationService.preparePreAuthenticationError(config.getEncryptionTypes()));
            }
            try {
                for (PaData paData : preAuthData) {
                    if (!paData.getPaDataType().equals((Object)PaDataType.PA_ENC_TIMESTAMP)) continue;
                    KerberosKey samKey = SamSubsystem.getInstance().verify(clientEntry, paData.getPaDataValue());
                    clientKey = new EncryptionKey(EncryptionType.getTypeByValue((int)samKey.getKeyType()), samKey.getEncoded());
                }
            }
            catch (SamException se) {
                throw new KerberosException(ErrorType.KRB_ERR_GENERIC, (Throwable)se);
            }
            authContext.setClientKey(clientKey);
            authContext.setPreAuthenticated(true);
            if (LOG.isDebugEnabled()) {
                LOG.debug("Pre-authentication using SAM subsystem successful for {}.", (Object)clientName);
            }
        }
    }

    private static void verifyEncryptedTimestamp(AuthenticationContext authContext) throws KerberosException, InvalidTicketException {
        LOG.debug("Verifying using encrypted timestamp.");
        KdcServer config = authContext.getConfig();
        KdcReq request = authContext.getRequest();
        CipherTextHandler cipherTextHandler = authContext.getCipherTextHandler();
        PrincipalStoreEntry clientEntry = authContext.getClientEntry();
        String clientName = clientEntry.getPrincipal().getName();
        EncryptionKey clientKey = null;
        if (clientEntry.getSamType() == null) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Entry for client principal {} has no SAM type.  Proceeding with standard pre-authentication.", (Object)clientName);
            }
            EncryptionType encryptionType = authContext.getEncryptionType();
            clientKey = (EncryptionKey)clientEntry.getKeyMap().get(encryptionType);
            if (clientKey == null) {
                throw new KerberosException(ErrorType.KDC_ERR_NULL_KEY);
            }
            if (config.isPaEncTimestampRequired()) {
                List preAuthData = request.getPaData();
                if (preAuthData == null) {
                    throw new KerberosException(ErrorType.KDC_ERR_PREAUTH_REQUIRED, AuthenticationService.preparePreAuthenticationError(config.getEncryptionTypes()));
                }
                PaEncTsEnc timestamp = null;
                for (PaData paData : preAuthData) {
                    if (!paData.getPaDataType().equals((Object)PaDataType.PA_ENC_TIMESTAMP)) continue;
                    EncryptedData dataValue = KerberosDecoder.decodeEncryptedData((byte[])paData.getPaDataValue());
                    byte[] decryptedData = cipherTextHandler.decrypt(clientKey, dataValue, KeyUsage.AS_REQ_PA_ENC_TIMESTAMP_WITH_CKEY);
                    timestamp = KerberosDecoder.decodePaEncTsEnc((byte[])decryptedData);
                }
                if (preAuthData.size() > 0 && timestamp == null) {
                    throw new KerberosException(ErrorType.KDC_ERR_PADATA_TYPE_NOSUPP);
                }
                if (timestamp == null) {
                    throw new KerberosException(ErrorType.KDC_ERR_PREAUTH_REQUIRED, AuthenticationService.preparePreAuthenticationError(config.getEncryptionTypes()));
                }
                if (!timestamp.getPaTimestamp().isInClockSkew(config.getAllowableClockSkew())) {
                    throw new KerberosException(ErrorType.KDC_ERR_PREAUTH_FAILED);
                }
            }
        }
        authContext.setClientKey(clientKey);
        authContext.setPreAuthenticated(true);
        if (LOG.isDebugEnabled()) {
            LOG.debug("Pre-authentication by encrypted timestamp successful for {}.", (Object)clientName);
        }
    }

    private static void getServerEntry(AuthenticationContext authContext) throws KerberosException, InvalidTicketException {
        PrincipalName principal = authContext.getRequest().getKdcReqBody().getSName();
        PrincipalStore store = authContext.getStore();
        KerberosPrincipal principalWithRealm = new KerberosPrincipal(principal.getNameString() + "@" + authContext.getRequest().getKdcReqBody().getRealm());
        authContext.setServerEntry(AuthenticationService.getEntry(principalWithRealm, store, ErrorType.KDC_ERR_S_PRINCIPAL_UNKNOWN));
    }

    private static void generateTicket(AuthenticationContext authContext) throws KerberosException, InvalidTicketException {
        KdcReq request = authContext.getRequest();
        CipherTextHandler cipherTextHandler = authContext.getCipherTextHandler();
        PrincipalName serverPrincipal = request.getKdcReqBody().getSName();
        EncryptionType encryptionType = authContext.getEncryptionType();
        EncryptionKey serverKey = (EncryptionKey)authContext.getServerEntry().getKeyMap().get(encryptionType);
        PrincipalName ticketPrincipal = request.getKdcReqBody().getSName();
        EncTicketPart encTicketPart = new EncTicketPart();
        KdcServer config = authContext.getConfig();
        TicketFlags ticketFlags = new TicketFlags();
        encTicketPart.setFlags(ticketFlags);
        ticketFlags.setFlag((KerberosFlag)TicketFlag.INITIAL);
        if (authContext.isPreAuthenticated()) {
            ticketFlags.setFlag((KerberosFlag)TicketFlag.PRE_AUTHENT);
        }
        if (request.getKdcReqBody().getKdcOptions().get(1)) {
            if (!config.isForwardableAllowed()) {
                throw new KerberosException(ErrorType.KDC_ERR_POLICY);
            }
            ticketFlags.setFlag((KerberosFlag)TicketFlag.FORWARDABLE);
        }
        if (request.getKdcReqBody().getKdcOptions().get(3)) {
            if (!config.isProxiableAllowed()) {
                throw new KerberosException(ErrorType.KDC_ERR_POLICY);
            }
            ticketFlags.setFlag((KerberosFlag)TicketFlag.PROXIABLE);
        }
        if (request.getKdcReqBody().getKdcOptions().get(5)) {
            if (!config.isPostdatedAllowed()) {
                throw new KerberosException(ErrorType.KDC_ERR_POLICY);
            }
            ticketFlags.setFlag((KerberosFlag)TicketFlag.MAY_POSTDATE);
        }
        if (request.getKdcReqBody().getKdcOptions().get(30) || request.getKdcReqBody().getKdcOptions().get(31) || request.getKdcReqBody().getKdcOptions().get(4) || request.getKdcReqBody().getKdcOptions().get(2) || request.getKdcReqBody().getKdcOptions().get(28)) {
            throw new KerberosException(ErrorType.KDC_ERR_BADOPTION);
        }
        EncryptionKey sessionKey = RandomKeyFactory.getRandomKey((EncryptionType)authContext.getEncryptionType());
        encTicketPart.setKey(sessionKey);
        encTicketPart.setCName(request.getKdcReqBody().getCName());
        encTicketPart.setCRealm(request.getKdcReqBody().getRealm());
        encTicketPart.setTransited(new TransitedEncoding());
        String serverRealm = request.getKdcReqBody().getRealm();
        KerberosTime now = new KerberosTime();
        encTicketPart.setAuthTime(now);
        KerberosTime startTime = request.getKdcReqBody().getFrom();
        if (startTime == null || startTime.lessThan(now) || startTime.isInClockSkew(config.getAllowableClockSkew()) && !request.getKdcReqBody().getKdcOptions().get(6)) {
            startTime = now;
        }
        if (startTime != null && startTime.greaterThan(now) && !startTime.isInClockSkew(config.getAllowableClockSkew()) && !request.getKdcReqBody().getKdcOptions().get(6)) {
            throw new KerberosException(ErrorType.KDC_ERR_CANNOT_POSTDATE);
        }
        if (request.getKdcReqBody().getKdcOptions().get(6)) {
            if (!config.isPostdatedAllowed()) {
                throw new KerberosException(ErrorType.KDC_ERR_POLICY);
            }
            ticketFlags.setFlag((KerberosFlag)TicketFlag.POSTDATED);
            ticketFlags.setFlag((KerberosFlag)TicketFlag.INVALID);
            encTicketPart.setStartTime(startTime);
        }
        long till = 0L;
        till = request.getKdcReqBody().getTill().getTime() == 0L ? Long.MAX_VALUE : request.getKdcReqBody().getTill().getTime();
        long endTime = Math.min(till, startTime.getTime() + config.getMaximumTicketLifetime());
        KerberosTime kerberosEndTime = new KerberosTime(endTime);
        encTicketPart.setEndTime(kerberosEndTime);
        if (kerberosEndTime.lessThan(startTime)) {
            throw new KerberosException(ErrorType.KDC_ERR_NEVER_VALID);
        }
        long ticketLifeTime = Math.abs(startTime.getTime() - kerberosEndTime.getTime());
        if (ticketLifeTime < config.getAllowableClockSkew()) {
            throw new KerberosException(ErrorType.KDC_ERR_NEVER_VALID);
        }
        KerberosTime tempRtime = request.getKdcReqBody().getRTime();
        if (request.getKdcReqBody().getKdcOptions().get(27) && request.getKdcReqBody().getTill().greaterThan(kerberosEndTime)) {
            if (!config.isRenewableAllowed()) {
                throw new KerberosException(ErrorType.KDC_ERR_POLICY);
            }
            request.getKdcReqBody().getKdcOptions().set(8);
            tempRtime = request.getKdcReqBody().getTill();
        }
        if (request.getKdcReqBody().getKdcOptions().get(8)) {
            if (!config.isRenewableAllowed()) {
                throw new KerberosException(ErrorType.KDC_ERR_POLICY);
            }
            ticketFlags.setFlag((KerberosFlag)TicketFlag.RENEWABLE);
            if (tempRtime == null || tempRtime.isZero()) {
                tempRtime = KerberosTime.INFINITY;
            }
            long renewTill = Math.min(tempRtime.getTime(), startTime.getTime() + config.getMaximumRenewableLifetime());
            encTicketPart.setRenewTill(new KerberosTime(renewTill));
        }
        if (request.getKdcReqBody().getAddresses() != null && request.getKdcReqBody().getAddresses().getAddresses() != null && request.getKdcReqBody().getAddresses().getAddresses().length > 0) {
            encTicketPart.setClientAddresses(request.getKdcReqBody().getAddresses());
        } else if (!config.isEmptyAddressesAllowed()) {
            throw new KerberosException(ErrorType.KDC_ERR_POLICY);
        }
        EncryptedData encryptedData = cipherTextHandler.seal(serverKey, (AbstractAsn1Object)encTicketPart, KeyUsage.AS_OR_TGS_REP_TICKET_WITH_SRVKEY);
        Ticket newTicket = new Ticket(ticketPrincipal, encryptedData);
        newTicket.setRealm(serverRealm);
        newTicket.setEncTicketPart(encTicketPart);
        if (LOG.isDebugEnabled()) {
            LOG.debug("Ticket will be issued for access to {}.", (Object)serverPrincipal.toString());
        }
        authContext.setTicket(newTicket);
    }

    private static void buildReply(AuthenticationContext authContext) throws KerberosException, InvalidTicketException {
        KdcReq request = authContext.getRequest();
        Ticket ticket = authContext.getTicket();
        AsRep reply = new AsRep();
        reply.setCName(request.getKdcReqBody().getCName());
        reply.setCRealm(request.getKdcReqBody().getRealm());
        reply.setTicket(ticket);
        EncKdcRepPart encKdcRepPart = new EncKdcRepPart();
        encKdcRepPart.setKey(ticket.getEncTicketPart().getKey());
        LastReq lastReq = new LastReq();
        lastReq.addEntry(new LastReqEntry(LastReqType.TIME_OF_INITIAL_REQ, new KerberosTime()));
        encKdcRepPart.setLastReq(lastReq);
        encKdcRepPart.setNonce(request.getKdcReqBody().getNonce());
        encKdcRepPart.setFlags(ticket.getEncTicketPart().getFlags());
        encKdcRepPart.setAuthTime(ticket.getEncTicketPart().getAuthTime());
        encKdcRepPart.setStartTime(ticket.getEncTicketPart().getStartTime());
        encKdcRepPart.setEndTime(ticket.getEncTicketPart().getEndTime());
        if (ticket.getEncTicketPart().getFlags().isRenewable()) {
            encKdcRepPart.setRenewTill(ticket.getEncTicketPart().getRenewTill());
        }
        encKdcRepPart.setSName(ticket.getSName());
        encKdcRepPart.setSRealm(ticket.getRealm());
        encKdcRepPart.setClientAddresses(ticket.getEncTicketPart().getClientAddresses());
        EncAsRepPart encAsRepPart = new EncAsRepPart();
        encAsRepPart.setEncKdcRepPart(encKdcRepPart);
        if (LOG.isDebugEnabled()) {
            AuthenticationService.monitorContext(authContext);
            AuthenticationService.monitorReply(reply, encKdcRepPart);
        }
        EncryptionKey clientKey = authContext.getClientKey();
        EncryptedData encryptedData = cipherTextHandler.seal(clientKey, (AbstractAsn1Object)encAsRepPart, KeyUsage.AS_REP_ENC_PART_WITH_CKEY);
        reply.setEncPart(encryptedData);
        reply.setEncKdcRepPart(encKdcRepPart);
        authContext.setReply((KerberosMessage)reply);
    }

    private static void monitorRequest(KdcContext kdcContext) {
        KdcReq request = kdcContext.getRequest();
        if (LOG.isDebugEnabled()) {
            try {
                String clientAddress = kdcContext.getClientAddress().getHostAddress();
                StringBuffer sb = new StringBuffer();
                sb.append("Received Authentication Service (AS) request:");
                sb.append("\n\tmessageType:           " + request.getMessageType());
                sb.append("\n\tprotocolVersionNumber: " + request.getProtocolVersionNumber());
                sb.append("\n\tclientAddress:         " + clientAddress);
                sb.append("\n\tnonce:                 " + request.getKdcReqBody().getNonce());
                sb.append("\n\tkdcOptions:            " + request.getKdcReqBody().getKdcOptions());
                sb.append("\n\tclientPrincipal:       " + request.getKdcReqBody().getCName());
                sb.append("\n\tserverPrincipal:       " + request.getKdcReqBody().getSName());
                sb.append("\n\tencryptionType:        " + KerberosUtils.getEncryptionTypesString((Set)request.getKdcReqBody().getEType()));
                sb.append("\n\trealm:                 " + request.getKdcReqBody().getRealm());
                sb.append("\n\tfrom time:             " + request.getKdcReqBody().getFrom());
                sb.append("\n\ttill time:             " + request.getKdcReqBody().getTill());
                sb.append("\n\trenew-till time:       " + request.getKdcReqBody().getRTime());
                sb.append("\n\thostAddresses:         " + request.getKdcReqBody().getAddresses());
                LOG.debug(sb.toString());
            }
            catch (Exception e) {
                LOG.error(I18n.err((I18n)I18n.ERR_153, (Object[])new Object[0]), (Throwable)e);
            }
        }
    }

    private static void monitorContext(AuthenticationContext authContext) {
        try {
            long clockSkew = authContext.getConfig().getAllowableClockSkew();
            InetAddress clientAddress = authContext.getClientAddress();
            StringBuilder sb = new StringBuilder();
            sb.append("Monitoring Authentication Service (AS) context:");
            sb.append("\n\tclockSkew              " + clockSkew);
            sb.append("\n\tclientAddress          " + clientAddress);
            KerberosPrincipal clientPrincipal = authContext.getClientEntry().getPrincipal();
            PrincipalStoreEntry clientEntry = authContext.getClientEntry();
            sb.append("\n\tprincipal              " + clientPrincipal);
            sb.append("\n\tcn                     " + clientEntry.getCommonName());
            sb.append("\n\trealm                  " + clientEntry.getRealmName());
            sb.append("\n\tprincipal              " + clientEntry.getPrincipal());
            sb.append("\n\tSAM type               " + clientEntry.getSamType());
            PrincipalName serverPrincipal = authContext.getRequest().getKdcReqBody().getSName();
            PrincipalStoreEntry serverEntry = authContext.getServerEntry();
            sb.append("\n\tprincipal              " + serverPrincipal);
            sb.append("\n\tcn                     " + serverEntry.getCommonName());
            sb.append("\n\trealm                  " + serverEntry.getRealmName());
            sb.append("\n\tprincipal              " + serverEntry.getPrincipal());
            sb.append("\n\tSAM type               " + serverEntry.getSamType());
            EncryptionType encryptionType = authContext.getEncryptionType();
            int clientKeyVersion = ((EncryptionKey)clientEntry.getKeyMap().get(encryptionType)).getKeyVersion();
            int serverKeyVersion = ((EncryptionKey)serverEntry.getKeyMap().get(encryptionType)).getKeyVersion();
            sb.append("\n\tRequest key type       " + encryptionType);
            sb.append("\n\tClient key version     " + clientKeyVersion);
            sb.append("\n\tServer key version     " + serverKeyVersion);
            LOG.debug(sb.toString());
        }
        catch (Exception e) {
            LOG.error(I18n.err((I18n)I18n.ERR_154, (Object[])new Object[0]), (Throwable)e);
        }
    }

    private static void monitorReply(AsRep reply, EncKdcRepPart part) {
        if (LOG.isDebugEnabled()) {
            try {
                StringBuffer sb = new StringBuffer();
                sb.append("Responding with Authentication Service (AS) reply:");
                sb.append("\n\tmessageType:           " + reply.getMessageType());
                sb.append("\n\tprotocolVersionNumber: " + reply.getProtocolVersionNumber());
                sb.append("\n\tnonce:                 " + part.getNonce());
                sb.append("\n\tclientPrincipal:       " + reply.getCName());
                sb.append("\n\tclient realm:          " + reply.getCRealm());
                sb.append("\n\tserverPrincipal:       " + part.getSName());
                sb.append("\n\tserver realm:          " + part.getSRealm());
                sb.append("\n\tauth time:             " + part.getAuthTime());
                sb.append("\n\tstart time:            " + part.getStartTime());
                sb.append("\n\tend time:              " + part.getEndTime());
                sb.append("\n\trenew-till time:       " + part.getRenewTill());
                sb.append("\n\thostAddresses:         " + part.getClientAddresses());
                LOG.debug(sb.toString());
            }
            catch (Exception e) {
                LOG.error(I18n.err((I18n)I18n.ERR_155, (Object[])new Object[0]), (Throwable)e);
            }
        }
    }

    private static PrincipalStoreEntry getEntry(KerberosPrincipal principal, PrincipalStore store, ErrorType errorType) throws KerberosException {
        PrincipalStoreEntry entry = null;
        try {
            entry = store.getPrincipal(principal);
        }
        catch (Exception e) {
            throw new KerberosException(errorType, (Throwable)e);
        }
        if (entry == null) {
            throw new KerberosException(errorType);
        }
        if (entry.getKeyMap() == null || entry.getKeyMap().isEmpty()) {
            throw new KerberosException(ErrorType.KDC_ERR_NULL_KEY);
        }
        return entry;
    }

    private static byte[] preparePreAuthenticationError(Set<EncryptionType> encryptionTypes) {
        PaData[] paDataSequence = new PaData[2];
        PaData paData = new PaData();
        paData.setPaDataType(PaDataType.PA_ENC_TIMESTAMP);
        paData.setPaDataValue(new byte[0]);
        paDataSequence[0] = paData;
        ETypeInfo eTypeInfo = new ETypeInfo();
        for (EncryptionType encryptionType : encryptionTypes) {
            ETypeInfoEntry etypeInfoEntry = new ETypeInfoEntry(encryptionType, null);
            eTypeInfo.addETypeInfoEntry(etypeInfoEntry);
        }
        byte[] encTypeInfo = null;
        try {
            ByteBuffer buffer = ByteBuffer.allocate(eTypeInfo.computeLength());
            encTypeInfo = eTypeInfo.encode(buffer).array();
        }
        catch (EncoderException ioe) {
            return null;
        }
        PaData responsePaData = new PaData(PaDataType.PA_ENCTYPE_INFO, encTypeInfo);
        MethodData methodData = new MethodData();
        methodData.addPaData(responsePaData);
        try {
            ByteBuffer buffer = ByteBuffer.allocate(methodData.computeLength());
            return methodData.encode(buffer).array();
        }
        catch (EncoderException ee) {
            return null;
        }
    }
}

