/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs.s3a;

import com.amazonaws.AbortedException;
import com.amazonaws.AmazonClientException;
import com.amazonaws.AmazonServiceException;
import com.amazonaws.ClientConfiguration;
import com.amazonaws.Protocol;
import com.amazonaws.SdkBaseException;
import com.amazonaws.auth.AWSCredentialsProvider;
import com.amazonaws.auth.EnvironmentVariableCredentialsProvider;
import com.amazonaws.retry.RetryUtils;
import com.amazonaws.services.dynamodbv2.model.AmazonDynamoDBException;
import com.amazonaws.services.dynamodbv2.model.LimitExceededException;
import com.amazonaws.services.dynamodbv2.model.ProvisionedThroughputExceededException;
import com.amazonaws.services.dynamodbv2.model.ResourceNotFoundException;
import com.amazonaws.services.s3.model.AmazonS3Exception;
import com.amazonaws.services.s3.model.MultiObjectDeleteException;
import com.amazonaws.services.s3.model.S3ObjectSummary;
import java.io.Closeable;
import java.io.EOFException;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InterruptedIOException;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.net.SocketTimeoutException;
import java.net.URI;
import java.nio.file.AccessDeniedException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import javax.annotation.Nullable;
import org.apache.commons.lang3.StringUtils;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.classification.InterfaceStability;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.LocatedFileStatus;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.PathFilter;
import org.apache.hadoop.fs.RemoteIterator;
import org.apache.hadoop.fs.s3a.AWSBadRequestException;
import org.apache.hadoop.fs.s3a.AWSClientIOException;
import org.apache.hadoop.fs.s3a.AWSCredentialProviderList;
import org.apache.hadoop.fs.s3a.AWSNoResponseException;
import org.apache.hadoop.fs.s3a.AWSRedirectException;
import org.apache.hadoop.fs.s3a.AWSS3IOException;
import org.apache.hadoop.fs.s3a.AWSServiceIOException;
import org.apache.hadoop.fs.s3a.AWSServiceThrottledException;
import org.apache.hadoop.fs.s3a.AWSStatus500Exception;
import org.apache.hadoop.fs.s3a.Constants;
import org.apache.hadoop.fs.s3a.CredentialInitializationException;
import org.apache.hadoop.fs.s3a.S3AEncryptionMethods;
import org.apache.hadoop.fs.s3a.S3AFileStatus;
import org.apache.hadoop.fs.s3a.S3AFileSystem;
import org.apache.hadoop.fs.s3a.SimpleAWSCredentialsProvider;
import org.apache.hadoop.fs.s3a.TemporaryAWSCredentialsProvider;
import org.apache.hadoop.fs.s3a.Tristate;
import org.apache.hadoop.fs.s3a.UnknownStoreException;
import org.apache.hadoop.fs.s3a.auth.IAMInstanceCredentialsProvider;
import org.apache.hadoop.fs.s3a.impl.ErrorTranslation;
import org.apache.hadoop.fs.s3a.impl.MultiObjectDeleteSupport;
import org.apache.hadoop.fs.s3a.impl.NetworkBinding;
import org.apache.hadoop.fs.s3native.S3xLoginHelper;
import org.apache.hadoop.io.IOUtils;
import org.apache.hadoop.net.ConnectTimeoutException;
import org.apache.hadoop.security.ProviderUtils;
import org.apache.hadoop.thirdparty.com.google.common.annotations.VisibleForTesting;
import org.apache.hadoop.thirdparty.com.google.common.base.Preconditions;
import org.apache.hadoop.thirdparty.com.google.common.collect.Lists;
import org.apache.hadoop.util.VersionInfo;
import org.apache.hadoop.util.functional.RemoteIterators;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
@InterfaceStability.Evolving
public final class S3AUtils {
    private static final Logger LOG = LoggerFactory.getLogger(S3AUtils.class);
    static final String CONSTRUCTOR_EXCEPTION = "constructor exception";
    static final String INSTANTIATION_EXCEPTION = "instantiation exception";
    static final String NOT_AWS_PROVIDER = "does not implement AWSCredentialsProvider";
    static final String ABSTRACT_PROVIDER = "is abstract and therefore cannot be created";
    static final String ENDPOINT_KEY = "Endpoint";
    static final String E_FS_CLOSED = "FileSystem is closed!";
    static final String CREDENTIAL_PROVIDER_PATH = "hadoop.security.credential.provider.path";
    public static final String SSE_C_NO_KEY_ERROR = S3AEncryptionMethods.SSE_C.getMethod() + " is enabled but no encryption key was declared in " + "fs.s3a.server-side-encryption.key";
    public static final String SSE_S3_WITH_KEY_ERROR = S3AEncryptionMethods.SSE_S3.getMethod() + " is enabled but an encryption key was set in " + "fs.s3a.server-side-encryption.key";
    private static final String EOF_MESSAGE_IN_XML_PARSER = "Failed to sanitize XML document destined for handler class";
    private static final String BUCKET_PATTERN = "fs.s3a.bucket.%s.%s";
    @VisibleForTesting
    public static final String E_FORBIDDEN_AWS_PROVIDER = "AWS provider class cannot be used";
    public static final List<Class<?>> STANDARD_AWS_PROVIDERS = Collections.unmodifiableList(Arrays.asList(TemporaryAWSCredentialsProvider.class, SimpleAWSCredentialsProvider.class, EnvironmentVariableCredentialsProvider.class, IAMInstanceCredentialsProvider.class));
    public static final PathFilter HIDDEN_FILE_FILTER = new PathFilter(){

        public boolean accept(Path path) {
            String name = path.getName();
            return !name.startsWith("_") && !name.startsWith(".");
        }

        public String toString() {
            return "HIDDEN_FILE_FILTER";
        }
    };
    public static final PathFilter ACCEPT_ALL = new PathFilter(){

        public boolean accept(Path file) {
            return true;
        }

        public String toString() {
            return "ACCEPT_ALL";
        }
    };

    private S3AUtils() {
    }

    public static IOException translateException(String operation, Path path, AmazonClientException exception) {
        return S3AUtils.translateException(operation, path.toString(), (SdkBaseException)((Object)exception));
    }

    public static IOException translateException(@Nullable String operation, String path, SdkBaseException exception) {
        IOException ioe;
        String message = String.format("%s%s: %s", new Object[]{operation, StringUtils.isNotEmpty((CharSequence)path) ? " on " + path : "", exception});
        if (!(exception instanceof AmazonServiceException)) {
            Exception innerCause = S3AUtils.containsInterruptedException(exception);
            if (innerCause != null) {
                return S3AUtils.translateInterruptedException(exception, innerCause, message);
            }
            if (S3AUtils.signifiesConnectionBroken(exception)) {
                return (EOFException)new EOFException(message).initCause(exception);
            }
            if (exception instanceof CredentialInitializationException) {
                return (AccessDeniedException)new AccessDeniedException(path, null, exception.toString()).initCause(exception);
            }
            return new AWSClientIOException(message, exception);
        }
        if (exception instanceof AmazonDynamoDBException) {
            return S3AUtils.translateDynamoDBException(path, message, (AmazonDynamoDBException)((Object)exception));
        }
        AmazonServiceException ase = (AmazonServiceException)exception;
        AmazonS3Exception s3Exception = ase instanceof AmazonS3Exception ? (AmazonS3Exception)ase : null;
        int status = ase.getStatusCode();
        message = message + ":" + ase.getErrorCode();
        switch (status) {
            case 301: 
            case 307: {
                if (s3Exception != null) {
                    if (s3Exception.getAdditionalDetails() != null && s3Exception.getAdditionalDetails().containsKey(ENDPOINT_KEY)) {
                        message = String.format("Received permanent redirect response to endpoint %s.  This likely indicates that the S3 endpoint configured in %s does not match the AWS region containing the bucket.", s3Exception.getAdditionalDetails().get(ENDPOINT_KEY), "fs.s3a.endpoint");
                    }
                    ioe = new AWSRedirectException(message, (AmazonServiceException)s3Exception);
                    break;
                }
                ioe = new AWSRedirectException(message, ase);
                break;
            }
            case 400: {
                ioe = new AWSBadRequestException(message, ase);
                break;
            }
            case 401: 
            case 403: {
                ioe = new AccessDeniedException(path, null, message);
                ioe.initCause((Throwable)ase);
                break;
            }
            case 404: {
                if (ErrorTranslation.isUnknownBucket(ase)) {
                    ioe = new UnknownStoreException(path, (Throwable)ase);
                    break;
                }
                ioe = new FileNotFoundException(message);
                ioe.initCause((Throwable)ase);
                break;
            }
            case 410: {
                ioe = new FileNotFoundException(message);
                ioe.initCause((Throwable)ase);
                break;
            }
            case 405: {
                ioe = new AWSBadRequestException(message, (AmazonServiceException)s3Exception);
                break;
            }
            case 416: {
                ioe = new EOFException(message);
                ioe.initCause((Throwable)ase);
                break;
            }
            case 443: 
            case 444: {
                ioe = new AWSNoResponseException(message, ase);
                break;
            }
            case 503: {
                ioe = new AWSServiceThrottledException(message, ase);
                break;
            }
            case 500: {
                ioe = new AWSStatus500Exception(message, ase);
                break;
            }
            case 200: {
                if (exception instanceof MultiObjectDeleteException) {
                    return MultiObjectDeleteSupport.translateDeleteException(message, (MultiObjectDeleteException)exception);
                }
            }
            default: {
                ioe = s3Exception != null ? new AWSS3IOException(message, s3Exception) : new AWSServiceIOException(message, ase);
            }
        }
        return ioe;
    }

    public static IOException extractException(String operation, String path, ExecutionException ee) {
        Throwable cause = ee.getCause();
        IOException ioe = cause instanceof AmazonClientException ? S3AUtils.translateException(operation, path, (SdkBaseException)((Object)((AmazonClientException)cause))) : (cause instanceof IOException ? (IOException)cause : new IOException(operation + " failed: " + cause, cause));
        return ioe;
    }

    static Exception containsInterruptedException(Throwable thrown) {
        if (thrown == null) {
            return null;
        }
        if (thrown instanceof InterruptedException || thrown instanceof InterruptedIOException || thrown instanceof AbortedException) {
            return (Exception)thrown;
        }
        return S3AUtils.containsInterruptedException(thrown.getCause());
    }

    private static InterruptedIOException translateInterruptedException(SdkBaseException exception, Exception innerCause, String message) {
        String name;
        Object ioe = innerCause instanceof SocketTimeoutException ? new SocketTimeoutException(message) : ((name = innerCause.getClass().getName()).endsWith(".ConnectTimeoutException") || name.endsWith("$ConnectTimeoutException") ? new ConnectTimeoutException(message) : new InterruptedIOException(message));
        ((Throwable)ioe).initCause(exception);
        return ioe;
    }

    public static boolean isThrottleException(Exception ex) {
        return ex instanceof AWSServiceThrottledException || ex instanceof ProvisionedThroughputExceededException || ex instanceof LimitExceededException || ex instanceof AmazonServiceException && 503 == ((AmazonServiceException)ex).getStatusCode() || ex instanceof SdkBaseException && RetryUtils.isThrottlingException((SdkBaseException)((SdkBaseException)((Object)ex)));
    }

    public static boolean signifiesConnectionBroken(SdkBaseException ex) {
        return ex.toString().contains(EOF_MESSAGE_IN_XML_PARSER);
    }

    public static IOException translateDynamoDBException(String path, String message, AmazonDynamoDBException ddbException) {
        if (S3AUtils.isThrottleException((Exception)((Object)ddbException))) {
            return new AWSServiceThrottledException(message, (AmazonServiceException)ddbException);
        }
        if (ddbException instanceof ResourceNotFoundException) {
            return (FileNotFoundException)new FileNotFoundException(message).initCause(ddbException);
        }
        int statusCode = ddbException.getStatusCode();
        String errorCode = ddbException.getErrorCode();
        IOException result = null;
        if (statusCode == 400) {
            switch (errorCode) {
                case "AccessDeniedException": {
                    result = (IOException)new AccessDeniedException(path, null, ddbException.toString()).initCause(ddbException);
                    break;
                }
                default: {
                    result = new AWSBadRequestException(message, (AmazonServiceException)ddbException);
                }
            }
        }
        if (result == null) {
            result = new AWSServiceIOException(message, (AmazonServiceException)ddbException);
        }
        return result;
    }

    public static String stringify(AmazonServiceException e) {
        StringBuilder builder = new StringBuilder(String.format("%s: %s error %d: %s; %s%s%n", e.getErrorType(), e.getServiceName(), e.getStatusCode(), e.getErrorCode(), e.getErrorMessage(), e.isRetryable() ? " (retryable)" : ""));
        String rawResponseContent = e.getRawResponseContent();
        if (rawResponseContent != null) {
            builder.append(rawResponseContent);
        }
        return builder.toString();
    }

    public static String stringify(AmazonS3Exception e) {
        StringBuilder builder = new StringBuilder(S3AUtils.stringify((AmazonServiceException)e));
        Map details = e.getAdditionalDetails();
        if (details != null) {
            builder.append('\n');
            for (Map.Entry d : details.entrySet()) {
                builder.append((String)d.getKey()).append('=').append((String)d.getValue()).append('\n');
            }
        }
        return builder.toString();
    }

    public static S3AFileStatus createFileStatus(Path keyPath, S3ObjectSummary summary, long blockSize, String owner, String eTag, String versionId) {
        long size = summary.getSize();
        return S3AUtils.createFileStatus(keyPath, S3AUtils.objectRepresentsDirectory(summary.getKey(), size), size, summary.getLastModified(), blockSize, owner, eTag, versionId);
    }

    public static S3AFileStatus createUploadFileStatus(Path keyPath, boolean isDir, long size, long blockSize, String owner, String eTag, String versionId) {
        Date date = isDir ? null : new Date();
        return S3AUtils.createFileStatus(keyPath, isDir, size, date, blockSize, owner, eTag, versionId);
    }

    private static S3AFileStatus createFileStatus(Path keyPath, boolean isDir, long size, Date modified, long blockSize, String owner, String eTag, String versionId) {
        if (isDir) {
            return new S3AFileStatus(Tristate.UNKNOWN, keyPath, owner);
        }
        return new S3AFileStatus(size, S3AUtils.dateToLong(modified), keyPath, blockSize, owner, eTag, versionId);
    }

    public static boolean objectRepresentsDirectory(String name, long size) {
        return !name.isEmpty() && name.charAt(name.length() - 1) == '/' && size == 0L;
    }

    public static long dateToLong(Date date) {
        if (date == null) {
            return 0L;
        }
        return date.getTime();
    }

    public static AWSCredentialProviderList createAWSCredentialProviderSet(@Nullable URI binding, Configuration conf) throws IOException {
        S3xLoginHelper.rejectSecretsInURIs(binding);
        AWSCredentialProviderList credentials = S3AUtils.buildAWSProviderList(binding, conf, "fs.s3a.aws.credentials.provider", STANDARD_AWS_PROVIDERS, new HashSet());
        LOG.debug("For URI {}, using credentials {}", (Object)binding, (Object)credentials);
        return credentials;
    }

    public static List<Class<?>> loadAWSProviderClasses(Configuration conf, String key, Class<?> ... defaultValue) throws IOException {
        try {
            return Arrays.asList(conf.getClasses(key, (Class[])defaultValue));
        }
        catch (RuntimeException e) {
            Throwable c = e.getCause() != null ? e.getCause() : e;
            throw new IOException("From option " + key + ' ' + c, c);
        }
    }

    public static AWSCredentialProviderList buildAWSProviderList(@Nullable URI binding, Configuration conf, String key, List<Class<?>> defaultValues, Set<Class<?>> forbidden) throws IOException {
        List<Class<?>> awsClasses = S3AUtils.loadAWSProviderClasses(conf, key, defaultValues.toArray(new Class[defaultValues.size()]));
        if (awsClasses.isEmpty()) {
            awsClasses = defaultValues;
        }
        AWSCredentialProviderList providers = new AWSCredentialProviderList();
        for (Class<?> aClass : awsClasses) {
            if (forbidden.contains(aClass)) {
                throw new IOException("AWS provider class cannot be used in option " + key + ": " + aClass);
            }
            providers.add(S3AUtils.createAWSCredentialProvider(conf, aClass, binding));
        }
        return providers;
    }

    private static AWSCredentialsProvider createAWSCredentialProvider(Configuration conf, Class<?> credClass, @Nullable URI uri) throws IOException {
        AWSCredentialsProvider credentials = null;
        String className = credClass.getName();
        if (!AWSCredentialsProvider.class.isAssignableFrom(credClass)) {
            throw new IOException("Class " + credClass + " " + NOT_AWS_PROVIDER);
        }
        if (Modifier.isAbstract(credClass.getModifiers())) {
            throw new IOException("Class " + credClass + " " + ABSTRACT_PROVIDER);
        }
        LOG.debug("Credential provider class is {}", (Object)className);
        try {
            Constructor<?> cons = S3AUtils.getConstructor(credClass, URI.class, Configuration.class);
            if (cons != null) {
                credentials = (AWSCredentialsProvider)cons.newInstance(uri, conf);
                return credentials;
            }
            cons = S3AUtils.getConstructor(credClass, Configuration.class);
            if (cons != null) {
                credentials = (AWSCredentialsProvider)cons.newInstance(conf);
                return credentials;
            }
            Method factory = S3AUtils.getFactoryMethod(credClass, AWSCredentialsProvider.class, "getInstance");
            if (factory != null) {
                credentials = (AWSCredentialsProvider)factory.invoke(null, new Object[0]);
                return credentials;
            }
            cons = S3AUtils.getConstructor(credClass, new Class[0]);
            if (cons != null) {
                credentials = (AWSCredentialsProvider)cons.newInstance(new Object[0]);
                return credentials;
            }
            throw new IOException(String.format("%s constructor exception.  A class specified in %s must provide a public constructor of a supported signature, or a public factory method named getInstance that accepts no arguments.", className, "fs.s3a.aws.credentials.provider"));
        }
        catch (InvocationTargetException e) {
            Throwable targetException = e.getTargetException();
            if (targetException == null) {
                targetException = e;
            }
            if (targetException instanceof IOException) {
                throw (IOException)targetException;
            }
            if (targetException instanceof SdkBaseException) {
                throw S3AUtils.translateException("Instantiate " + className, "", (SdkBaseException)targetException);
            }
            throw new IOException(className + " " + INSTANTIATION_EXCEPTION + ": " + targetException, targetException);
        }
        catch (IllegalArgumentException | ReflectiveOperationException e) {
            throw new IOException(className + " " + INSTANTIATION_EXCEPTION + ": " + e, e);
        }
    }

    public static boolean setIfDefined(Configuration config, String key, String val, String origin) {
        if (StringUtils.isNotEmpty((CharSequence)val)) {
            config.set(key, val, origin);
            return true;
        }
        return false;
    }

    public static S3xLoginHelper.Login getAWSAccessKeys(URI name, Configuration conf) throws IOException {
        S3xLoginHelper.rejectSecretsInURIs(name);
        Configuration c = ProviderUtils.excludeIncompatibleCredentialProviders((Configuration)conf, S3AFileSystem.class);
        String bucket = name != null ? name.getHost() : "";
        String accessKey = S3AUtils.lookupPassword(bucket, c, "fs.s3a.access.key");
        String secretKey = S3AUtils.lookupPassword(bucket, c, "fs.s3a.secret.key");
        return new S3xLoginHelper.Login(accessKey, secretKey);
    }

    @Deprecated
    public static String lookupPassword(String bucket, Configuration conf, String baseKey, String overrideVal) throws IOException {
        return S3AUtils.lookupPassword(bucket, conf, baseKey, overrideVal, "");
    }

    public static String lookupPassword(String bucket, Configuration conf, String baseKey) throws IOException {
        return S3AUtils.lookupPassword(bucket, conf, baseKey, null, "");
    }

    public static String lookupPassword(String bucket, Configuration conf, String baseKey, String overrideVal, String defVal) throws IOException {
        String initialVal;
        Preconditions.checkArgument((boolean)baseKey.startsWith("fs.s3a."), (String)"%s does not start with $%s", (Object)baseKey, (Object)"fs.s3a.");
        if (StringUtils.isNotEmpty((CharSequence)bucket)) {
            String subkey = baseKey.substring("fs.s3a.".length());
            String shortBucketKey = String.format(BUCKET_PATTERN, bucket, subkey);
            String longBucketKey = String.format(BUCKET_PATTERN, bucket, baseKey);
            initialVal = S3AUtils.getPassword(conf, longBucketKey, overrideVal);
            initialVal = S3AUtils.getPassword(conf, shortBucketKey, initialVal);
        } else {
            initialVal = overrideVal;
        }
        return S3AUtils.getPassword(conf, baseKey, initialVal, defVal);
    }

    private static String getPassword(Configuration conf, String key, String val) throws IOException {
        return S3AUtils.getPassword(conf, key, val, "");
    }

    private static String getPassword(Configuration conf, String key, String val, String defVal) throws IOException {
        return StringUtils.isEmpty((CharSequence)val) ? S3AUtils.lookupPassword(conf, key, defVal) : val;
    }

    static String lookupPassword(Configuration conf, String key, String defVal) throws IOException {
        try {
            char[] pass = conf.getPassword(key);
            return pass != null ? new String(pass).trim() : defVal;
        }
        catch (IOException ioe) {
            throw new IOException("Cannot find password option " + key, ioe);
        }
    }

    public static String stringify(S3ObjectSummary summary) {
        StringBuilder builder = new StringBuilder(summary.getKey().length() + 100);
        builder.append(summary.getKey()).append(' ');
        builder.append("size=").append(summary.getSize());
        return builder.toString();
    }

    public static int intOption(Configuration conf, String key, int defVal, int min) {
        int v = conf.getInt(key, defVal);
        Preconditions.checkArgument((v >= min ? 1 : 0) != 0, (Object)String.format("Value of %s: %d is below the minimum value %d", key, v, min));
        LOG.debug("Value of {} is {}", (Object)key, (Object)v);
        return v;
    }

    public static long longOption(Configuration conf, String key, long defVal, long min) {
        long v = conf.getLong(key, defVal);
        Preconditions.checkArgument((v >= min ? 1 : 0) != 0, (Object)String.format("Value of %s: %d is below the minimum value %d", key, v, min));
        LOG.debug("Value of {} is {}", (Object)key, (Object)v);
        return v;
    }

    public static long longBytesOption(Configuration conf, String key, long defVal, long min) {
        long v = conf.getLongBytes(key, defVal);
        Preconditions.checkArgument((v >= min ? 1 : 0) != 0, (Object)String.format("Value of %s: %d is below the minimum value %d", key, v, min));
        LOG.debug("Value of {} is {}", (Object)key, (Object)v);
        return v;
    }

    public static long getMultipartSizeProperty(Configuration conf, String property, long defVal) {
        long partSize = conf.getLongBytes(property, defVal);
        if (partSize < 0x500000L) {
            LOG.warn("{} must be at least 5 MB; configured value is {}", (Object)property, (Object)partSize);
            partSize = 0x500000L;
        }
        return partSize;
    }

    public static int ensureOutputParameterInRange(String name, long size) {
        if (size > Integer.MAX_VALUE) {
            LOG.warn("s3a: {} capped to ~2.14GB (maximum allowed size with current output mechanism)", (Object)name);
            return Integer.MAX_VALUE;
        }
        return (int)size;
    }

    private static Constructor<?> getConstructor(Class<?> cl, Class<?> ... args) {
        try {
            Constructor<?> cons = cl.getDeclaredConstructor(args);
            return Modifier.isPublic(cons.getModifiers()) ? cons : null;
        }
        catch (NoSuchMethodException | SecurityException e) {
            return null;
        }
    }

    private static Method getFactoryMethod(Class<?> cl, Class<?> returnType, String methodName) {
        try {
            Method m = cl.getDeclaredMethod(methodName, new Class[0]);
            if (Modifier.isPublic(m.getModifiers()) && Modifier.isStatic(m.getModifiers()) && returnType.isAssignableFrom(m.getReturnType())) {
                return m;
            }
            return null;
        }
        catch (NoSuchMethodException | SecurityException e) {
            return null;
        }
    }

    public static Configuration propagateBucketOptions(Configuration source, String bucket) {
        Preconditions.checkArgument((boolean)StringUtils.isNotEmpty((CharSequence)bucket), (Object)"bucket");
        String bucketPrefix = "fs.s3a.bucket." + bucket + '.';
        LOG.debug("Propagating entries under {}", (Object)bucketPrefix);
        Configuration dest = new Configuration(source);
        for (Map.Entry entry : source) {
            String key = (String)entry.getKey();
            String value = (String)entry.getValue();
            if (!key.startsWith(bucketPrefix) || bucketPrefix.equals(key)) continue;
            String stripped = key.substring(bucketPrefix.length());
            if (stripped.startsWith("bucket.") || "impl".equals(stripped)) {
                LOG.debug("Ignoring bucket option {}", (Object)key);
                continue;
            }
            String origin = "[" + StringUtils.join((Object[])source.getPropertySources(key), (String)", ") + "]";
            String generic = "fs.s3a." + stripped;
            LOG.debug("Updating {} from {}", (Object)generic, (Object)origin);
            dest.set(generic, value, key + " via " + origin);
        }
        return dest;
    }

    public static void deleteQuietly(FileSystem fs, Path path, boolean recursive) {
        try {
            fs.delete(path, recursive);
        }
        catch (IOException e) {
            LOG.debug("Failed to delete {}", (Object)path, (Object)e);
        }
    }

    public static void deleteWithWarning(FileSystem fs, Path path, boolean recursive) {
        try {
            fs.delete(path, recursive);
        }
        catch (IOException e) {
            LOG.warn("Failed to delete {}", (Object)path, (Object)e);
        }
    }

    @Deprecated
    public static ClientConfiguration createAwsConf(Configuration conf, String bucket) throws IOException {
        return S3AUtils.createAwsConf(conf, bucket, null);
    }

    public static ClientConfiguration createAwsConf(Configuration conf, String bucket, String awsServiceIdentifier) throws IOException {
        ClientConfiguration awsConf = new ClientConfiguration();
        S3AUtils.initConnectionSettings(conf, awsConf);
        S3AUtils.initProxySupport(conf, bucket, awsConf);
        S3AUtils.initUserAgent(conf, awsConf);
        if (StringUtils.isNotEmpty((CharSequence)awsServiceIdentifier)) {
            String signerOverride;
            String configKey = null;
            switch (awsServiceIdentifier) {
                case "S3": {
                    configKey = Constants.SIGNING_ALGORITHM_S3;
                    break;
                }
                case "DDB": {
                    configKey = Constants.SIGNING_ALGORITHM_DDB;
                    break;
                }
                case "STS": {
                    configKey = Constants.SIGNING_ALGORITHM_STS;
                    break;
                }
            }
            if (configKey != null && !(signerOverride = conf.getTrimmed(configKey, "")).isEmpty()) {
                LOG.debug("Signer override for {}} = {}", (Object)awsServiceIdentifier, (Object)signerOverride);
                awsConf.setSignerOverride(signerOverride);
            }
        }
        return awsConf;
    }

    public static void initConnectionSettings(Configuration conf, ClientConfiguration awsConf) throws IOException {
        awsConf.setMaxConnections(S3AUtils.intOption(conf, "fs.s3a.connection.maximum", 48, 1));
        S3AUtils.initProtocolSettings(conf, awsConf);
        awsConf.setMaxErrorRetry(S3AUtils.intOption(conf, "fs.s3a.attempts.maximum", 10, 0));
        awsConf.setConnectionTimeout(S3AUtils.intOption(conf, "fs.s3a.connection.establish.timeout", 50000, 0));
        awsConf.setSocketTimeout(S3AUtils.intOption(conf, "fs.s3a.connection.timeout", 200000, 0));
        int sockSendBuffer = S3AUtils.intOption(conf, "fs.s3a.socket.send.buffer", 8192, 2048);
        int sockRecvBuffer = S3AUtils.intOption(conf, "fs.s3a.socket.recv.buffer", 8192, 2048);
        long requestTimeoutMillis = conf.getTimeDuration("fs.s3a.connection.request.timeout", 0L, TimeUnit.SECONDS, TimeUnit.MILLISECONDS);
        if (requestTimeoutMillis > Integer.MAX_VALUE) {
            LOG.debug("Request timeout is too high({} ms). Setting to {} ms instead", (Object)requestTimeoutMillis, (Object)Integer.MAX_VALUE);
            requestTimeoutMillis = Integer.MAX_VALUE;
        }
        awsConf.setRequestTimeout((int)requestTimeoutMillis);
        awsConf.setSocketBufferSizeHints(sockSendBuffer, sockRecvBuffer);
        String signerOverride = conf.getTrimmed("fs.s3a.signing-algorithm", "");
        if (!signerOverride.isEmpty()) {
            LOG.debug("Signer override = {}", (Object)signerOverride);
            awsConf.setSignerOverride(signerOverride);
        }
    }

    private static void initProtocolSettings(Configuration conf, ClientConfiguration awsConf) throws IOException {
        boolean secureConnections = conf.getBoolean("fs.s3a.connection.ssl.enabled", true);
        awsConf.setProtocol(secureConnections ? Protocol.HTTPS : Protocol.HTTP);
        if (secureConnections) {
            NetworkBinding.bindSSLChannelMode(conf, awsConf);
        }
    }

    public static void initProxySupport(Configuration conf, String bucket, ClientConfiguration awsConf) throws IllegalArgumentException, IOException {
        String proxyHost = conf.getTrimmed("fs.s3a.proxy.host", "");
        int proxyPort = conf.getInt("fs.s3a.proxy.port", -1);
        if (!proxyHost.isEmpty()) {
            awsConf.setProxyHost(proxyHost);
            if (proxyPort >= 0) {
                awsConf.setProxyPort(proxyPort);
            } else if (conf.getBoolean("fs.s3a.connection.ssl.enabled", true)) {
                LOG.warn("Proxy host set without port. Using HTTPS default 443");
                awsConf.setProxyPort(443);
            } else {
                LOG.warn("Proxy host set without port. Using HTTP default 80");
                awsConf.setProxyPort(80);
            }
            String proxyUsername = S3AUtils.lookupPassword(bucket, conf, "fs.s3a.proxy.username", null, null);
            String proxyPassword = S3AUtils.lookupPassword(bucket, conf, "fs.s3a.proxy.password", null, null);
            if (proxyUsername == null != (proxyPassword == null)) {
                String msg = "Proxy error: fs.s3a.proxy.username or fs.s3a.proxy.password set without the other.";
                LOG.error(msg);
                throw new IllegalArgumentException(msg);
            }
            awsConf.setProxyUsername(proxyUsername);
            awsConf.setProxyPassword(proxyPassword);
            awsConf.setProxyDomain(conf.getTrimmed("fs.s3a.proxy.domain"));
            awsConf.setProxyWorkstation(conf.getTrimmed("fs.s3a.proxy.workstation"));
            if (LOG.isDebugEnabled()) {
                LOG.debug("Using proxy server {}:{} as user {} with password {} on domain {} as workstation {}", new Object[]{awsConf.getProxyHost(), awsConf.getProxyPort(), String.valueOf(awsConf.getProxyUsername()), awsConf.getProxyPassword(), awsConf.getProxyDomain(), awsConf.getProxyWorkstation()});
            }
        } else if (proxyPort >= 0) {
            String msg = "Proxy error: fs.s3a.proxy.port set without fs.s3a.proxy.host";
            LOG.error(msg);
            throw new IllegalArgumentException(msg);
        }
    }

    private static void initUserAgent(Configuration conf, ClientConfiguration awsConf) {
        String userAgent = "Hadoop " + VersionInfo.getVersion();
        String userAgentPrefix = conf.getTrimmed("fs.s3a.user.agent.prefix", "");
        if (!userAgentPrefix.isEmpty()) {
            userAgent = userAgentPrefix + ", " + userAgent;
        }
        LOG.debug("Using User-Agent: {}", (Object)userAgent);
        awsConf.setUserAgentPrefix(userAgent);
    }

    public static S3AFileStatus[] iteratorToStatuses(RemoteIterator<S3AFileStatus> iterator, Set<Path> tombstones) throws IOException {
        ArrayList<S3AFileStatus> statuses = new ArrayList<S3AFileStatus>();
        while (iterator.hasNext()) {
            S3AFileStatus status = (S3AFileStatus)((Object)iterator.next());
            if (tombstones.contains(status.getPath())) continue;
            statuses.add(status);
        }
        return statuses.toArray(new S3AFileStatus[0]);
    }

    public static long applyLocatedFiles(RemoteIterator<? extends LocatedFileStatus> iterator, CallOnLocatedFileStatus eval) throws IOException {
        return RemoteIterators.foreach(iterator, eval::call);
    }

    public static <T> List<T> mapLocatedFiles(RemoteIterator<? extends LocatedFileStatus> iterator, LocatedFileStatusMap<T> eval) throws IOException {
        ArrayList results = new ArrayList();
        S3AUtils.applyLocatedFiles(iterator, s -> results.add(eval.call(s)));
        return results;
    }

    public static <T> List<T> flatmapLocatedFiles(RemoteIterator<LocatedFileStatus> iterator, LocatedFileStatusMap<Optional<T>> eval) throws IOException {
        ArrayList results = new ArrayList();
        S3AUtils.applyLocatedFiles(iterator, s -> ((Optional)eval.call(s)).map(r -> results.add(r)));
        return results;
    }

    public static List<LocatedFileStatus> listAndFilter(FileSystem fileSystem, Path path, boolean recursive, PathFilter filter) throws IOException {
        return S3AUtils.flatmapLocatedFiles((RemoteIterator<LocatedFileStatus>)fileSystem.listFiles(path, recursive), status -> S3AUtils.maybe(filter.accept(status.getPath()), status));
    }

    public static <T> Optional<T> maybe(boolean include, T value) {
        return include ? Optional.of(value) : Optional.empty();
    }

    static void patchSecurityCredentialProviders(Configuration conf) {
        Collection customCredentials = conf.getStringCollection("fs.s3a.security.credential.provider.path");
        Collection hadoopCredentials = conf.getStringCollection(CREDENTIAL_PROVIDER_PATH);
        if (!customCredentials.isEmpty()) {
            ArrayList all = Lists.newArrayList((Iterable)customCredentials);
            all.addAll(hadoopCredentials);
            String joined = StringUtils.join((Iterable)all, (char)',');
            LOG.debug("Setting {} to {}", (Object)CREDENTIAL_PROVIDER_PATH, (Object)joined);
            conf.set(CREDENTIAL_PROVIDER_PATH, joined, "patch of fs.s3a.security.credential.provider.path");
        }
    }

    public static String getServerSideEncryptionKey(String bucket, Configuration conf) {
        try {
            return S3AUtils.lookupPassword(bucket, conf, "fs.s3a.server-side-encryption.key");
        }
        catch (IOException e) {
            LOG.error("Cannot retrieve fs.s3a.server-side-encryption.key", (Throwable)e);
            return "";
        }
    }

    public static S3AEncryptionMethods getEncryptionAlgorithm(String bucket, Configuration conf) throws IOException {
        S3AEncryptionMethods sse = S3AEncryptionMethods.getMethod(S3AUtils.lookupPassword(bucket, conf, "fs.s3a.server-side-encryption-algorithm"));
        String sseKey = S3AUtils.getServerSideEncryptionKey(bucket, conf);
        int sseKeyLen = StringUtils.isBlank((CharSequence)sseKey) ? 0 : sseKey.length();
        String diagnostics = S3AUtils.passwordDiagnostics(sseKey, "key");
        switch (sse) {
            case SSE_C: {
                LOG.debug("Using SSE-C with {}", (Object)diagnostics);
                if (sseKeyLen != 0) break;
                throw new IOException(SSE_C_NO_KEY_ERROR);
            }
            case SSE_S3: {
                if (sseKeyLen == 0) break;
                throw new IOException(SSE_S3_WITH_KEY_ERROR + " (" + diagnostics + ")");
            }
            case SSE_KMS: {
                LOG.debug("Using SSE-KMS with {}", (Object)diagnostics);
                break;
            }
            default: {
                LOG.debug("Data is unencrypted");
            }
        }
        return sse;
    }

    private static String passwordDiagnostics(String pass, String description) {
        if (pass == null) {
            return "null " + description;
        }
        int len = pass.length();
        switch (len) {
            case 0: {
                return "empty " + description;
            }
            case 1: {
                return description + " of length 1";
            }
        }
        return description + " of length " + len + " ending with " + pass.charAt(len - 1);
    }

    @Deprecated
    public static void closeAll(Logger log, Closeable ... closeables) {
        IOUtils.cleanupWithLogger((Logger)log, (Closeable[])closeables);
    }

    public static void closeAutocloseables(Logger log, AutoCloseable ... closeables) {
        if (log == null) {
            log = LOG;
        }
        for (AutoCloseable c : closeables) {
            if (c == null) continue;
            try {
                log.debug("Closing {}", (Object)c);
                c.close();
            }
            catch (Exception e) {
                log.debug("Exception in closing {}", (Object)c, (Object)e);
            }
        }
    }

    public static void setBucketOption(Configuration conf, String bucket, String genericKey, String value) {
        String baseKey = genericKey.startsWith("fs.s3a.") ? genericKey.substring("fs.s3a.".length()) : genericKey;
        conf.set("fs.s3a.bucket." + bucket + '.' + baseKey, value, "S3AUtils");
    }

    public static void clearBucketOption(Configuration conf, String bucket, String genericKey) {
        String baseKey = genericKey.startsWith("fs.s3a.") ? genericKey.substring("fs.s3a.".length()) : genericKey;
        String k = "fs.s3a.bucket." + bucket + '.' + baseKey;
        LOG.debug("Unset {}", (Object)k);
        conf.unset(k);
    }

    public static String getBucketOption(Configuration conf, String bucket, String genericKey) {
        String baseKey = genericKey.startsWith("fs.s3a.") ? genericKey.substring("fs.s3a.".length()) : genericKey;
        return conf.get("fs.s3a.bucket." + bucket + '.' + baseKey);
    }

    public static String maybeAddTrailingSlash(String key) {
        if (!key.isEmpty() && !key.endsWith("/")) {
            return key + '/';
        }
        return key;
    }

    @FunctionalInterface
    public static interface LocatedFileStatusMap<T> {
        public T call(LocatedFileStatus var1) throws IOException;
    }

    @FunctionalInterface
    public static interface CallOnLocatedFileStatus {
        public void call(LocatedFileStatus var1) throws IOException;
    }
}

