/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.security;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.DataInput;
import java.io.DataInputStream;
import java.io.DataOutput;
import java.io.DataOutputStream;
import java.io.IOException;
import java.security.GeneralSecurityException;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import javax.crypto.KeyGenerator;
import javax.crypto.Mac;
import javax.crypto.spec.SecretKeySpec;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.hdfs.security.BlockAccessKey;
import org.apache.hadoop.hdfs.security.BlockAccessToken;
import org.apache.hadoop.hdfs.security.ExportedAccessKeys;
import org.apache.hadoop.io.Text;
import org.apache.hadoop.io.WritableUtils;
import org.apache.hadoop.security.UserGroupInformation;

@InterfaceAudience.Private
public class AccessTokenHandler {
    private static final Log LOG = LogFactory.getLog(AccessTokenHandler.class);
    private final boolean isMaster;
    private final long keyUpdateInterval;
    private long tokenLifetime;
    private long serialNo = new SecureRandom().nextLong();
    private KeyGenerator keyGen;
    private BlockAccessKey currentKey;
    private BlockAccessKey nextKey;
    private Map<Long, BlockAccessKey> allKeys;

    public AccessTokenHandler(boolean isMaster, long keyUpdateInterval, long tokenLifetime) throws IOException {
        this.isMaster = isMaster;
        this.keyUpdateInterval = keyUpdateInterval;
        this.tokenLifetime = tokenLifetime;
        this.allKeys = new HashMap<Long, BlockAccessKey>();
        if (isMaster) {
            try {
                this.generateKeys();
                this.initMac(this.currentKey);
            }
            catch (GeneralSecurityException e) {
                throw (IOException)new IOException("Failed to create AccessTokenHandler").initCause(e);
            }
        }
    }

    private synchronized void generateKeys() throws NoSuchAlgorithmException {
        this.keyGen = KeyGenerator.getInstance("HmacSHA1");
        ++this.serialNo;
        this.currentKey = new BlockAccessKey(this.serialNo, new Text(this.keyGen.generateKey().getEncoded()), System.currentTimeMillis() + 2L * this.keyUpdateInterval + this.tokenLifetime);
        ++this.serialNo;
        this.nextKey = new BlockAccessKey(this.serialNo, new Text(this.keyGen.generateKey().getEncoded()), System.currentTimeMillis() + 3L * this.keyUpdateInterval + this.tokenLifetime);
        this.allKeys.put(this.currentKey.getKeyID(), this.currentKey);
        this.allKeys.put(this.nextKey.getKeyID(), this.nextKey);
    }

    private synchronized void initMac(BlockAccessKey key) throws IOException {
        try {
            Mac mac = Mac.getInstance("HmacSHA1");
            mac.init(new SecretKeySpec(key.getKey().getBytes(), "HmacSHA1"));
            key.setMac(mac);
        }
        catch (GeneralSecurityException e) {
            throw (IOException)new IOException("Failed to initialize Mac for access key, keyID=" + key.getKeyID()).initCause(e);
        }
    }

    public synchronized ExportedAccessKeys exportKeys() {
        if (!this.isMaster) {
            return null;
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug((Object)"Exporting access keys");
        }
        return new ExportedAccessKeys(true, this.keyUpdateInterval, this.tokenLifetime, this.currentKey, this.allKeys.values().toArray(new BlockAccessKey[0]));
    }

    private synchronized void removeExpiredKeys() {
        long now = System.currentTimeMillis();
        Iterator<Map.Entry<Long, BlockAccessKey>> it = this.allKeys.entrySet().iterator();
        while (it.hasNext()) {
            Map.Entry<Long, BlockAccessKey> e = it.next();
            if (e.getValue().getExpiryDate() >= now) continue;
            it.remove();
        }
    }

    public synchronized void setKeys(ExportedAccessKeys exportedKeys) throws IOException {
        if (this.isMaster || exportedKeys == null) {
            return;
        }
        LOG.info((Object)"Setting access keys");
        this.removeExpiredKeys();
        this.currentKey = exportedKeys.getCurrentKey();
        this.initMac(this.currentKey);
        BlockAccessKey[] receivedKeys = exportedKeys.getAllKeys();
        for (int i = 0; i < receivedKeys.length; ++i) {
            if (receivedKeys[i] == null) continue;
            this.allKeys.put(receivedKeys[i].getKeyID(), receivedKeys[i]);
        }
    }

    public synchronized void updateKeys() throws IOException {
        if (!this.isMaster) {
            return;
        }
        LOG.info((Object)"Updating access keys");
        this.removeExpiredKeys();
        this.allKeys.put(this.currentKey.getKeyID(), new BlockAccessKey(this.currentKey.getKeyID(), this.currentKey.getKey(), System.currentTimeMillis() + this.keyUpdateInterval + this.tokenLifetime));
        this.currentKey = new BlockAccessKey(this.nextKey.getKeyID(), this.nextKey.getKey(), System.currentTimeMillis() + 2L * this.keyUpdateInterval + this.tokenLifetime);
        this.initMac(this.currentKey);
        this.allKeys.put(this.currentKey.getKeyID(), this.currentKey);
        ++this.serialNo;
        this.nextKey = new BlockAccessKey(this.serialNo, new Text(this.keyGen.generateKey().getEncoded()), System.currentTimeMillis() + 3L * this.keyUpdateInterval + this.tokenLifetime);
        this.allKeys.put(this.nextKey.getKeyID(), this.nextKey);
    }

    private synchronized boolean verifyToken(long keyID, BlockAccessToken token) throws IOException {
        BlockAccessKey key = this.allKeys.get(keyID);
        if (key == null) {
            LOG.warn((Object)("Access key for keyID=" + keyID + " doesn't exist."));
            return false;
        }
        if (key.getMac() == null) {
            this.initMac(key);
        }
        Text tokenID = token.getTokenID();
        Text authenticator = new Text(key.getMac().doFinal(tokenID.getBytes()));
        return authenticator.equals((Object)token.getTokenAuthenticator());
    }

    public BlockAccessToken generateToken(long blockID, EnumSet<AccessMode> modes) throws IOException {
        UserGroupInformation ugi = UserGroupInformation.getCurrentUser();
        String userID = ugi == null ? null : ugi.getShortUserName();
        return this.generateToken(userID, blockID, modes);
    }

    public synchronized BlockAccessToken generateToken(String userID, long blockID, EnumSet<AccessMode> modes) throws IOException {
        if (LOG.isDebugEnabled()) {
            LOG.debug((Object)("Generating access token for user=" + userID + ", blockID=" + blockID + ", access modes=" + modes + ", keyID=" + this.currentKey.getKeyID()));
        }
        if (modes == null || modes.isEmpty()) {
            throw new IOException("access modes can't be null or empty");
        }
        ByteArrayOutputStream buf = new ByteArrayOutputStream(4096);
        DataOutputStream out = new DataOutputStream(buf);
        WritableUtils.writeVLong((DataOutput)out, (long)(System.currentTimeMillis() + this.tokenLifetime));
        WritableUtils.writeVLong((DataOutput)out, (long)this.currentKey.getKeyID());
        WritableUtils.writeString((DataOutput)out, (String)userID);
        WritableUtils.writeVLong((DataOutput)out, (long)blockID);
        WritableUtils.writeVInt((DataOutput)out, (int)modes.size());
        for (AccessMode aMode : modes) {
            WritableUtils.writeEnum((DataOutput)out, (Enum)aMode);
        }
        Text tokenID = new Text(buf.toByteArray());
        return new BlockAccessToken(tokenID, new Text(this.currentKey.getMac().doFinal(tokenID.getBytes())));
    }

    public boolean checkAccess(BlockAccessToken token, String userID, long blockID, AccessMode mode) throws IOException {
        long oExpiry = 0L;
        long oKeyID = 0L;
        String oUserID = null;
        long oBlockID = 0L;
        EnumSet<AccessMode> oModes = EnumSet.noneOf(AccessMode.class);
        try {
            ByteArrayInputStream buf = new ByteArrayInputStream(token.getTokenID().getBytes());
            DataInputStream in = new DataInputStream(buf);
            oExpiry = WritableUtils.readVLong((DataInput)in);
            oKeyID = WritableUtils.readVLong((DataInput)in);
            oUserID = WritableUtils.readString((DataInput)in);
            oBlockID = WritableUtils.readVLong((DataInput)in);
            int length = WritableUtils.readVInt((DataInput)in);
            for (int i = 0; i < length; ++i) {
                oModes.add((AccessMode)WritableUtils.readEnum((DataInput)in, AccessMode.class));
            }
        }
        catch (IOException e) {
            throw (IOException)new IOException("Unable to parse access token for user=" + userID + ", blockID=" + blockID + ", access mode=" + (Object)((Object)mode)).initCause(e);
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug((Object)("Verifying access token for user=" + userID + ", blockID=" + blockID + ", access mode=" + (Object)((Object)mode) + ", keyID=" + oKeyID));
        }
        return (userID == null || userID.equals(oUserID)) && oBlockID == blockID && !AccessTokenHandler.isExpired(oExpiry) && oModes.contains((Object)mode) && this.verifyToken(oKeyID, token);
    }

    private static boolean isExpired(long expiryDate) {
        return System.currentTimeMillis() > expiryDate;
    }

    static boolean isTokenExpired(BlockAccessToken token) throws IOException {
        ByteArrayInputStream buf = new ByteArrayInputStream(token.getTokenID().getBytes());
        DataInputStream in = new DataInputStream(buf);
        long expiryDate = WritableUtils.readVLong((DataInput)in);
        return AccessTokenHandler.isExpired(expiryDate);
    }

    synchronized void setTokenLifetime(long tokenLifetime) {
        this.tokenLifetime = tokenLifetime;
    }

    public static enum AccessMode {
        READ,
        WRITE,
        COPY,
        REPLACE;

    }
}

