/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.server.namenode;

import com.google.common.base.Joiner;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Lists;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.SortedMap;
import java.util.TreeMap;
import java.util.TreeSet;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.fs.FileUtil;
import org.apache.hadoop.hdfs.server.common.Storage;
import org.apache.hadoop.hdfs.server.namenode.FSImageStorageInspector;
import org.apache.hadoop.hdfs.server.namenode.FileJournalManager;
import org.apache.hadoop.hdfs.server.namenode.NNStorage;

class FSImageTransactionalStorageInspector
extends FSImageStorageInspector {
    public static final Log LOG = LogFactory.getLog(FSImageTransactionalStorageInspector.class);
    private boolean needToSave = false;
    private boolean isUpgradeFinalized = true;
    List<FSImageStorageInspector.FSImageFile> foundImages = new ArrayList<FSImageStorageInspector.FSImageFile>();
    List<FileJournalManager.EditLogFile> foundEditLogs = new ArrayList<FileJournalManager.EditLogFile>();
    SortedMap<Long, LogGroup> logGroups = new TreeMap<Long, LogGroup>();
    long maxSeenTxId = 0L;
    private static final Pattern IMAGE_REGEX = Pattern.compile(NNStorage.NameNodeFile.IMAGE.getName() + "_(\\d+)");

    FSImageTransactionalStorageInspector() {
    }

    @Override
    public void inspectDirectory(Storage.StorageDirectory sd) throws IOException {
        File[] filesInStorage;
        if (!sd.getVersionFile().exists()) {
            LOG.info((Object)("No version file in " + sd.getRoot()));
            this.needToSave |= true;
            return;
        }
        File currentDir = sd.getCurrentDir();
        try {
            filesInStorage = FileUtil.listFiles((File)currentDir);
        }
        catch (IOException ioe) {
            LOG.warn((Object)("Unable to inspect storage directory " + currentDir), (Throwable)ioe);
            return;
        }
        for (File f : filesInStorage) {
            LOG.debug((Object)("Checking file " + f));
            String name = f.getName();
            Matcher imageMatch = IMAGE_REGEX.matcher(name);
            if (!imageMatch.matches()) continue;
            if (sd.getStorageDirType().isOfType(NNStorage.NameNodeDirType.IMAGE)) {
                try {
                    long txid = Long.valueOf(imageMatch.group(1));
                    this.foundImages.add(new FSImageStorageInspector.FSImageFile(sd, f, txid));
                }
                catch (NumberFormatException nfe) {
                    LOG.error((Object)("Image file " + f + " has improperly formatted " + "transaction ID"));
                }
                continue;
            }
            LOG.warn((Object)("Found image file at " + f + " but storage directory is " + "not configured to contain images."));
        }
        try {
            this.maxSeenTxId = Math.max(this.maxSeenTxId, NNStorage.readTransactionIdFile(sd));
        }
        catch (IOException ioe) {
            LOG.warn((Object)("Unable to determine the max transaction ID seen by " + sd), (Throwable)ioe);
        }
        List<FileJournalManager.EditLogFile> editLogs = FileJournalManager.matchEditLogs(filesInStorage);
        if (sd.getStorageDirType().isOfType(NNStorage.NameNodeDirType.EDITS)) {
            for (FileJournalManager.EditLogFile log : editLogs) {
                this.addEditLog(log);
            }
        } else if (!editLogs.isEmpty()) {
            LOG.warn((Object)("Found the following edit log file(s) in " + sd + " even though it was not configured to store edits:\n" + "  " + Joiner.on((String)"\n  ").join(editLogs)));
        }
        this.isUpgradeFinalized = this.isUpgradeFinalized && !sd.getPreviousDir().exists();
    }

    private void addEditLog(FileJournalManager.EditLogFile foundEditLog) {
        this.foundEditLogs.add(foundEditLog);
        LogGroup group = (LogGroup)this.logGroups.get(foundEditLog.getFirstTxId());
        if (group == null) {
            group = new LogGroup(foundEditLog.getFirstTxId());
            this.logGroups.put(foundEditLog.getFirstTxId(), group);
        }
        group.add(foundEditLog);
    }

    @Override
    public boolean isUpgradeFinalized() {
        return this.isUpgradeFinalized;
    }

    FSImageStorageInspector.FSImageFile getLatestImage() {
        FSImageStorageInspector.FSImageFile ret = null;
        for (FSImageStorageInspector.FSImageFile img : this.foundImages) {
            if (ret != null && img.txId <= ret.txId) continue;
            ret = img;
        }
        return ret;
    }

    public List<FSImageStorageInspector.FSImageFile> getFoundImages() {
        return ImmutableList.copyOf(this.foundImages);
    }

    public List<FileJournalManager.EditLogFile> getEditLogFiles() {
        return ImmutableList.copyOf(this.foundEditLogs);
    }

    @Override
    public FSImageStorageInspector.LoadPlan createLoadPlan() throws IOException {
        if (this.foundImages.isEmpty()) {
            throw new FileNotFoundException("No valid image files found");
        }
        FSImageStorageInspector.FSImageFile recoveryImage = this.getLatestImage();
        LogLoadPlan logPlan = this.createLogLoadPlan(recoveryImage.txId, Long.MAX_VALUE);
        return new TransactionalLoadPlan(recoveryImage, logPlan);
    }

    LogLoadPlan createLogLoadPlan(long sinceTxId, long maxStartTxId) throws IOException {
        long lastLogGroupStartTxId;
        SortedMap<Object, Object> usefulGroups;
        long expectedTxId = sinceTxId + 1L;
        ArrayList<FileJournalManager.EditLogFile> recoveryLogs = new ArrayList<FileJournalManager.EditLogFile>();
        SortedMap<Long, LogGroup> tailGroups = this.logGroups.tailMap(expectedTxId);
        if (this.logGroups.size() > tailGroups.size()) {
            LOG.debug((Object)("Excluded " + (this.logGroups.size() - tailGroups.size()) + " groups of logs because they start with a txid less than image " + "txid " + sinceTxId));
        }
        if ((usefulGroups = maxStartTxId > sinceTxId ? tailGroups.headMap(maxStartTxId) : new TreeMap()).size() > tailGroups.size()) {
            LOG.debug((Object)("Excluded " + (tailGroups.size() - usefulGroups.size()) + " groups of logs because they start with a txid higher than max " + "txid " + sinceTxId));
        }
        for (Map.Entry<Object, Object> entry : usefulGroups.entrySet()) {
            long logStartTxId = (Long)entry.getKey();
            LogGroup logGroup = (LogGroup)entry.getValue();
            logGroup.planRecovery();
            if (expectedTxId != -12345L && logStartTxId != expectedTxId) {
                throw new IOException("Expected next log group would start at txid " + expectedTxId + " but starts at txid " + logStartTxId);
            }
            recoveryLogs.add(logGroup.getBestNonCorruptLog());
            if (logGroup.hasKnownLastTxId()) {
                expectedTxId = logGroup.getLastTxId() + 1L;
                continue;
            }
            expectedTxId = -12345L;
        }
        long l = lastLogGroupStartTxId = usefulGroups.isEmpty() ? 0L : (Long)usefulGroups.lastKey();
        if (this.maxSeenTxId > sinceTxId && this.maxSeenTxId > lastLogGroupStartTxId) {
            String msg = "At least one storage directory indicated it has seen a log segment starting at txid " + this.maxSeenTxId;
            msg = usefulGroups.isEmpty() ? msg + " but there are no logs to load." : msg + " but the most recent log file found starts with txid " + lastLogGroupStartTxId;
            throw new IOException(msg);
        }
        return new LogLoadPlan(recoveryLogs, Lists.newArrayList(usefulGroups.values()));
    }

    @Override
    public boolean needToSave() {
        return this.needToSave;
    }

    static class LogLoadPlan {
        final List<FileJournalManager.EditLogFile> editLogs;
        final List<LogGroup> logGroupsToRecover;

        LogLoadPlan(List<FileJournalManager.EditLogFile> editLogs, List<LogGroup> logGroupsToRecover) {
            this.editLogs = editLogs;
            this.logGroupsToRecover = logGroupsToRecover;
        }

        public void doRecovery() throws IOException {
            for (LogGroup g : this.logGroupsToRecover) {
                g.recover();
            }
        }

        public List<File> getEditsFiles() {
            ArrayList<File> ret = new ArrayList<File>();
            for (FileJournalManager.EditLogFile log : this.editLogs) {
                ret.add(log.getFile());
            }
            return ret;
        }
    }

    static class TransactionalLoadPlan
    extends FSImageStorageInspector.LoadPlan {
        final FSImageStorageInspector.FSImageFile image;
        final LogLoadPlan logPlan;

        public TransactionalLoadPlan(FSImageStorageInspector.FSImageFile image, LogLoadPlan logPlan) {
            this.image = image;
            this.logPlan = logPlan;
        }

        @Override
        boolean doRecovery() throws IOException {
            this.logPlan.doRecovery();
            return false;
        }

        @Override
        File getImageFile() {
            return this.image.getFile();
        }

        @Override
        List<File> getEditsFiles() {
            return this.logPlan.getEditsFiles();
        }

        @Override
        Storage.StorageDirectory getStorageDirectoryForProperties() {
            return this.image.sd;
        }
    }

    static class LogGroup {
        long startTxId;
        List<FileJournalManager.EditLogFile> logs = new ArrayList<FileJournalManager.EditLogFile>();
        private Set<Long> endTxIds = new TreeSet<Long>();
        private boolean hasInProgress = false;
        private boolean hasFinalized = false;

        LogGroup(long startTxId) {
            this.startTxId = startTxId;
        }

        FileJournalManager.EditLogFile getBestNonCorruptLog() {
            for (FileJournalManager.EditLogFile log : this.logs) {
                if (log.isCorrupt() || log.isInProgress()) continue;
                return log;
            }
            for (FileJournalManager.EditLogFile log : this.logs) {
                if (log.isCorrupt()) continue;
                return log;
            }
            throw new IllegalStateException("No non-corrupt logs for txid " + this.startTxId);
        }

        boolean hasKnownLastTxId() {
            for (FileJournalManager.EditLogFile log : this.logs) {
                if (log.isInProgress()) continue;
                return true;
            }
            return false;
        }

        long getLastTxId() {
            for (FileJournalManager.EditLogFile log : this.logs) {
                if (log.isInProgress()) continue;
                return log.getLastTxId();
            }
            throw new IllegalStateException("LogGroup only has in-progress logs");
        }

        void add(FileJournalManager.EditLogFile log) {
            assert (log.getFirstTxId() == this.startTxId);
            this.logs.add(log);
            if (log.isInProgress()) {
                this.hasInProgress = true;
            } else {
                this.hasFinalized = true;
                this.endTxIds.add(log.getLastTxId());
            }
        }

        void planRecovery() throws IOException {
            assert (this.hasInProgress || this.hasFinalized);
            this.checkConsistentEndTxIds();
            if (this.hasFinalized && this.hasInProgress) {
                this.planMixedLogRecovery();
            } else if (!this.hasFinalized && this.hasInProgress) {
                this.planAllInProgressRecovery();
            } else if (this.hasFinalized && !this.hasInProgress) {
                LOG.debug((Object)("No recovery necessary for logs starting at txid " + this.startTxId));
            }
        }

        private void planMixedLogRecovery() throws IOException {
            for (FileJournalManager.EditLogFile log : this.logs) {
                if (!log.isInProgress()) continue;
                LOG.warn((Object)("Log at " + log.getFile() + " is in progress, but " + "other logs starting at the same txid " + this.startTxId + " are finalized. Moving aside."));
                log.markCorrupt();
            }
        }

        private void planAllInProgressRecovery() throws IOException {
            LOG.warn((Object)("Logs beginning at txid " + this.startTxId + " were are all " + "in-progress (probably truncated due to a previous NameNode " + "crash)"));
            if (this.logs.size() == 1) {
                FileJournalManager.EditLogFile log = this.logs.get(0);
                if (log.validateLog().numTransactions == 0L) {
                    LOG.warn((Object)("Marking log at " + log.getFile() + " as corrupt since " + "it has no transactions in it."));
                    log.markCorrupt();
                }
                return;
            }
            long maxValidTxnCount = Long.MIN_VALUE;
            for (FileJournalManager.EditLogFile log : this.logs) {
                long validTxnCount = log.validateLog().numTransactions;
                LOG.warn((Object)("  Log " + log.getFile() + " valid txns=" + validTxnCount + " valid len=" + log.validateLog().validLength));
                maxValidTxnCount = Math.max(maxValidTxnCount, validTxnCount);
            }
            for (FileJournalManager.EditLogFile log : this.logs) {
                long txns = log.validateLog().numTransactions;
                if (txns < maxValidTxnCount) {
                    LOG.warn((Object)("Marking log at " + log.getFile() + " as corrupt since " + "it is has only " + txns + " valid txns whereas another " + "log has " + maxValidTxnCount));
                    log.markCorrupt();
                    continue;
                }
                if (txns != 0L) continue;
                LOG.warn((Object)("Marking log at " + log.getFile() + " as corrupt since " + "it has no transactions in it."));
                log.markCorrupt();
            }
        }

        private void checkConsistentEndTxIds() throws IOException {
            if (this.hasFinalized && this.endTxIds.size() > 1) {
                throw new IOException("More than one ending txid was found for logs starting at txid " + this.startTxId + ". " + "Found: " + StringUtils.join(this.endTxIds, (char)','));
            }
        }

        void recover() throws IOException {
            for (FileJournalManager.EditLogFile log : this.logs) {
                if (log.isCorrupt()) {
                    log.moveAsideCorruptFile();
                    continue;
                }
                if (!log.isInProgress()) continue;
                log.finalizeLog();
            }
        }
    }
}

