/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.mapreduce.security.token;

import java.io.IOException;
import java.net.InetAddress;
import java.net.URI;
import java.security.PrivilegedExceptionAction;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.LinkedBlockingQueue;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.classification.InterfaceStability;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.hdfs.DistributedFileSystem;
import org.apache.hadoop.hdfs.security.token.delegation.DelegationTokenIdentifier;
import org.apache.hadoop.hdfs.tools.DelegationTokenFetcher;
import org.apache.hadoop.io.Text;
import org.apache.hadoop.mapreduce.JobID;
import org.apache.hadoop.security.AccessControlException;
import org.apache.hadoop.security.Credentials;
import org.apache.hadoop.security.UserGroupInformation;
import org.apache.hadoop.security.token.SecretManager;
import org.apache.hadoop.security.token.Token;
import org.apache.hadoop.util.StringUtils;

@InterfaceAudience.Private
@InterfaceStability.Unstable
public class DelegationTokenRenewal {
    private static final Log LOG = LogFactory.getLog(DelegationTokenRenewal.class);
    public static final String SCHEME = "hdfs";
    private static Timer renewalTimer = new Timer(true);
    private static DelegationTokenCancelThread dtCancelThread = new DelegationTokenCancelThread();
    private static Set<DelegationTokenToRenew> delegationTokens;
    private static final Text kindHdfs;

    private static void addTokenToList(DelegationTokenToRenew t) {
        delegationTokens.add(t);
    }

    public static synchronized void registerDelegationTokensForRenewal(JobID jobId, Credentials ts, Configuration conf) {
        if (ts == null) {
            return;
        }
        Collection tokens = ts.getAllTokens();
        long now = System.currentTimeMillis();
        for (Token t : tokens) {
            if (!t.getKind().equals((Object)kindHdfs)) continue;
            Token dt = t;
            DelegationTokenToRenew dtr = new DelegationTokenToRenew(jobId, (Token<DelegationTokenIdentifier>)dt, conf, now);
            DelegationTokenRenewal.addTokenToList(dtr);
            DelegationTokenRenewal.setTimerForTokenRenewal(dtr, true);
            LOG.info((Object)("registering token for renewal for service =" + dt.getService() + " and jobID = " + jobId));
        }
    }

    private static String getHttpAddressForToken(Token<DelegationTokenIdentifier> token, Configuration conf) throws IOException {
        String[] ipaddr = token.getService().toString().split(":");
        InetAddress iaddr = InetAddress.getByName(ipaddr[0]);
        String dnsName = iaddr.getCanonicalHostName();
        String httpsPort = conf.get("dfs.hftp.https.port");
        if (httpsPort == null) {
            httpsPort = conf.get("dfs.https.port", "50470");
        }
        return "https://" + dnsName + ":" + httpsPort;
    }

    protected static long renewDelegationTokenOverHttps(final Token<DelegationTokenIdentifier> token, Configuration conf) throws InterruptedException, IOException {
        final String httpAddress = DelegationTokenRenewal.getHttpAddressForToken(token, conf);
        LOG.info((Object)("address to renew=" + httpAddress + "; tok=" + token.getService()));
        Long expDate = (Long)UserGroupInformation.getLoginUser().doAs((PrivilegedExceptionAction)new PrivilegedExceptionAction<Long>(){

            @Override
            public Long run() throws IOException {
                return DelegationTokenFetcher.renewDelegationToken((String)httpAddress, (Token)token);
            }
        });
        LOG.info((Object)("Renew over HTTP done. addr=" + httpAddress + ";res=" + expDate));
        return expDate;
    }

    private static long renewDelegationToken(DelegationTokenToRenew dttr) throws Exception {
        long newExpirationDate = System.currentTimeMillis() + 3600000L;
        Token<DelegationTokenIdentifier> token = dttr.token;
        Configuration conf = dttr.conf;
        if (token.getKind().equals((Object)kindHdfs)) {
            DistributedFileSystem dfs = null;
            try {
                dfs = DelegationTokenRenewal.getDFSForToken(token, conf);
            }
            catch (IOException e) {
                LOG.info((Object)"couldn't get DFS to renew. Will retry over HTTPS");
                dfs = null;
            }
            try {
                if (dfs != null) {
                    newExpirationDate = dfs.renewDelegationToken(token);
                }
                newExpirationDate = DelegationTokenRenewal.renewDelegationTokenOverHttps(token, conf);
            }
            catch (SecretManager.InvalidToken ite) {
                LOG.warn((Object)"invalid token - not scheduling for renew");
                DelegationTokenRenewal.removeFailedDelegationToken(dttr);
                throw new IOException("failed to renew token", ite);
            }
            catch (AccessControlException ioe) {
                LOG.warn((Object)("failed to renew token:" + token), (Throwable)ioe);
                DelegationTokenRenewal.removeFailedDelegationToken(dttr);
                throw new IOException("failed to renew token", ioe);
            }
            catch (Exception e) {
                LOG.warn((Object)("failed to renew token:" + token), (Throwable)e);
            }
        } else {
            throw new Exception("unknown token type to renew:" + token.getKind());
        }
        return newExpirationDate;
    }

    private static DistributedFileSystem getDFSForToken(Token<DelegationTokenIdentifier> token, final Configuration conf) throws Exception {
        DistributedFileSystem dfs = null;
        try {
            final URI uri = new URI("hdfs://" + token.getService().toString());
            dfs = (DistributedFileSystem)UserGroupInformation.getLoginUser().doAs((PrivilegedExceptionAction)new PrivilegedExceptionAction<DistributedFileSystem>(){

                @Override
                public DistributedFileSystem run() throws IOException {
                    return (DistributedFileSystem)FileSystem.get((URI)uri, (Configuration)conf);
                }
            });
        }
        catch (Exception e) {
            LOG.warn((Object)("Failed to create a dfs to renew/cancel for:" + token.getService()), (Throwable)e);
            throw e;
        }
        return dfs;
    }

    private static void setTimerForTokenRenewal(DelegationTokenToRenew token, boolean firstTime) {
        long renewIn;
        long now = System.currentTimeMillis();
        if (firstTime) {
            renewIn = now;
        } else {
            long expiresIn = token.expirationDate - now;
            renewIn = now + expiresIn - expiresIn / 10L;
        }
        try {
            RenewalTimerTask tTask = new RenewalTimerTask(token);
            token.setTimerTask(tTask);
            renewalTimer.schedule(token.timerTask, new Date(renewIn));
        }
        catch (Exception e) {
            LOG.warn((Object)"failed to schedule a task, token will not renew more", (Throwable)e);
        }
    }

    public static void close() {
        renewalTimer.cancel();
        delegationTokens.clear();
    }

    protected static void cancelDelegationTokenOverHttps(final Token<DelegationTokenIdentifier> token, Configuration conf) throws InterruptedException, IOException {
        final String httpAddress = DelegationTokenRenewal.getHttpAddressForToken(token, conf);
        LOG.info((Object)("address to cancel=" + httpAddress + "; tok=" + token.getService()));
        UserGroupInformation.getLoginUser().doAs((PrivilegedExceptionAction)new PrivilegedExceptionAction<Void>(){

            @Override
            public Void run() throws IOException {
                DelegationTokenFetcher.cancelDelegationToken((String)httpAddress, (Token)token);
                return null;
            }
        });
        LOG.info((Object)("Cancel over HTTP done. addr=" + httpAddress));
    }

    private static void cancelToken(DelegationTokenToRenew t) {
        Token<DelegationTokenIdentifier> token = t.token;
        Configuration conf = t.conf;
        if (token.getKind().equals((Object)kindHdfs)) {
            dtCancelThread.cancelToken(token, conf);
        }
    }

    private static void removeFailedDelegationToken(DelegationTokenToRenew t) {
        JobID jobId = t.jobId;
        if (LOG.isDebugEnabled()) {
            LOG.debug((Object)("removing failed delegation token for jobid=" + jobId + ";t=" + t.token.getService()));
        }
        delegationTokens.remove(t);
        if (t.timerTask != null) {
            t.timerTask.cancel();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static void removeDelegationTokenRenewalForJob(JobID jobId) {
        Set<DelegationTokenToRenew> set = delegationTokens;
        synchronized (set) {
            Iterator<DelegationTokenToRenew> it = delegationTokens.iterator();
            while (it.hasNext()) {
                DelegationTokenToRenew dttr = it.next();
                if (!dttr.jobId.equals(jobId)) continue;
                if (LOG.isDebugEnabled()) {
                    LOG.debug((Object)("removing delegation token for jobid=" + jobId + ";t=" + dttr.token.getService()));
                }
                if (dttr.timerTask != null) {
                    dttr.timerTask.cancel();
                }
                DelegationTokenRenewal.cancelToken(dttr);
                it.remove();
            }
        }
    }

    static {
        dtCancelThread.start();
        delegationTokens = Collections.synchronizedSet(new HashSet());
        kindHdfs = DelegationTokenIdentifier.HDFS_DELEGATION_KIND;
    }

    private static class RenewalTimerTask
    extends TimerTask {
        private DelegationTokenToRenew dttr;

        RenewalTimerTask(DelegationTokenToRenew t) {
            this.dttr = t;
        }

        @Override
        public void run() {
            Token<DelegationTokenIdentifier> token = this.dttr.token;
            long newExpirationDate = 0L;
            try {
                newExpirationDate = DelegationTokenRenewal.renewDelegationToken(this.dttr);
            }
            catch (Exception e) {
                return;
            }
            if (LOG.isDebugEnabled()) {
                LOG.debug((Object)("renewing for:" + token.getService() + ";newED=" + newExpirationDate));
            }
            this.dttr.expirationDate = newExpirationDate;
            DelegationTokenRenewal.setTimerForTokenRenewal(this.dttr, false);
        }
    }

    private static class DelegationTokenCancelThread
    extends Thread {
        private LinkedBlockingQueue<TokenWithConf> queue = new LinkedBlockingQueue();

        public DelegationTokenCancelThread() {
            super("Delegation Token Canceler");
            this.setDaemon(true);
        }

        public void cancelToken(Token<DelegationTokenIdentifier> token, Configuration conf) {
            TokenWithConf tokenWithConf = new TokenWithConf(token, conf);
            while (!this.queue.offer(tokenWithConf)) {
                LOG.warn((Object)("Unable to add token " + token + " for cancellation. " + "Will retry.."));
                try {
                    Thread.sleep(100L);
                }
                catch (InterruptedException e) {
                    throw new RuntimeException(e);
                }
            }
        }

        @Override
        public void run() {
            while (true) {
                TokenWithConf tokenWithConf = null;
                try {
                    tokenWithConf = this.queue.take();
                    DistributedFileSystem dfs = null;
                    try {
                        dfs = DelegationTokenRenewal.getDFSForToken((Token<DelegationTokenIdentifier>)tokenWithConf.token, tokenWithConf.conf);
                    }
                    catch (Exception e) {
                        LOG.info((Object)"couldn't get DFS to cancel. Will retry over HTTPS");
                        dfs = null;
                    }
                    if (dfs != null) {
                        dfs.cancelDelegationToken(tokenWithConf.token);
                    } else {
                        DelegationTokenRenewal.cancelDelegationTokenOverHttps(tokenWithConf.token, tokenWithConf.conf);
                    }
                    if (!LOG.isDebugEnabled()) continue;
                    LOG.debug((Object)("Canceling token " + tokenWithConf.token.getService() + " for dfs=" + dfs));
                    continue;
                }
                catch (IOException e) {
                    LOG.warn((Object)("Failed to cancel token " + tokenWithConf.token + " " + StringUtils.stringifyException((Throwable)e)));
                    continue;
                }
                catch (InterruptedException ie) {
                    return;
                }
                catch (Throwable t) {
                    LOG.warn((Object)("Got exception " + StringUtils.stringifyException((Throwable)t) + ". Exiting.."));
                    System.exit(-1);
                    continue;
                }
                break;
            }
        }

        private static class TokenWithConf {
            Token<DelegationTokenIdentifier> token;
            Configuration conf;

            TokenWithConf(Token<DelegationTokenIdentifier> token, Configuration conf) {
                this.token = token;
                this.conf = conf;
            }
        }
    }

    private static class DelegationTokenToRenew {
        public final Token<DelegationTokenIdentifier> token;
        public final JobID jobId;
        public final Configuration conf;
        public long expirationDate;
        public TimerTask timerTask;

        public DelegationTokenToRenew(JobID jId, Token<DelegationTokenIdentifier> t, Configuration newConf, long newExpirationDate) {
            this.token = t;
            this.jobId = jId;
            this.conf = newConf;
            this.expirationDate = newExpirationDate;
            this.timerTask = null;
            if (this.token == null || this.jobId == null || this.conf == null) {
                throw new IllegalArgumentException("invalid params for Renew Token;t=" + this.token + ";j=" + this.jobId + ";c=" + this.conf);
            }
        }

        public void setTimerTask(TimerTask tTask) {
            this.timerTask = tTask;
        }

        public String toString() {
            return this.token + ";exp=" + this.expirationDate;
        }

        public boolean equals(Object obj) {
            if (obj == this) {
                return true;
            }
            if (obj == null || this.getClass() != obj.getClass()) {
                return false;
            }
            return this.token.equals(((DelegationTokenToRenew)obj).token);
        }

        public int hashCode() {
            return this.token.hashCode();
        }
    }
}

