/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.io.hfile;

import com.google.common.annotations.VisibleForTesting;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Random;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.atomic.AtomicInteger;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hbase.MultithreadedTestUtil;
import org.apache.hadoop.hbase.io.HeapSize;
import org.apache.hadoop.hbase.io.compress.Compression;
import org.apache.hadoop.hbase.io.hfile.BlockCache;
import org.apache.hadoop.hbase.io.hfile.BlockCacheKey;
import org.apache.hadoop.hbase.io.hfile.BlockType;
import org.apache.hadoop.hbase.io.hfile.Cacheable;
import org.apache.hadoop.hbase.io.hfile.CacheableDeserializer;
import org.apache.hadoop.hbase.io.hfile.CacheableDeserializerIdManager;
import org.apache.hadoop.hbase.io.hfile.HFileBlock;
import org.apache.hadoop.hbase.io.hfile.HFileContext;
import org.apache.hadoop.hbase.io.hfile.HFileContextBuilder;
import org.apache.hadoop.hbase.io.hfile.bucket.BucketCache;
import org.apache.hadoop.hbase.util.ChecksumType;
import org.junit.Assert;

public class CacheTestUtils {
    private static final boolean includesMemstoreTS = true;

    public static void testHeapSizeChanges(BlockCache toBeTested, int blockSize) {
        HFileBlockPair[] blocks = CacheTestUtils.generateHFileBlocks(blockSize, 1);
        long heapSize = ((HeapSize)toBeTested).heapSize();
        toBeTested.cacheBlock(blocks[0].blockName, (Cacheable)blocks[0].block);
        Assert.assertTrue((heapSize < ((HeapSize)toBeTested).heapSize() ? 1 : 0) != 0);
        toBeTested.evictBlock(blocks[0].blockName);
        Assert.assertEquals((long)heapSize, (long)((HeapSize)toBeTested).heapSize());
    }

    public static void testCacheMultiThreaded(final BlockCache toBeTested, int blockSize, int numThreads, int numQueries, double passingScore) throws Exception {
        Configuration conf = new Configuration();
        MultithreadedTestUtil.TestContext ctx = new MultithreadedTestUtil.TestContext(conf);
        final AtomicInteger totalQueries = new AtomicInteger();
        final ConcurrentLinkedQueue<HFileBlockPair> blocksToTest = new ConcurrentLinkedQueue<HFileBlockPair>();
        final AtomicInteger hits = new AtomicInteger();
        final AtomicInteger miss = new AtomicInteger();
        HFileBlockPair[] blocks = CacheTestUtils.generateHFileBlocks(numQueries, blockSize);
        blocksToTest.addAll(Arrays.asList(blocks));
        for (int i = 0; i < numThreads; ++i) {
            MultithreadedTestUtil.RepeatingTestThread t = new MultithreadedTestUtil.RepeatingTestThread(ctx){

                @Override
                public void doAnAction() throws Exception {
                    if (!blocksToTest.isEmpty()) {
                        HFileBlockPair ourBlock = (HFileBlockPair)blocksToTest.poll();
                        if (ourBlock == null) {
                            this.ctx.setStopFlag(true);
                            return;
                        }
                        toBeTested.cacheBlock(ourBlock.blockName, (Cacheable)ourBlock.block);
                        Cacheable retrievedBlock = toBeTested.getBlock(ourBlock.blockName, false, false, true);
                        if (retrievedBlock != null) {
                            Assert.assertEquals((Object)ourBlock.block, (Object)retrievedBlock);
                            toBeTested.evictBlock(ourBlock.blockName);
                            hits.incrementAndGet();
                            Assert.assertNull((Object)toBeTested.getBlock(ourBlock.blockName, false, false, true));
                        } else {
                            miss.incrementAndGet();
                        }
                        totalQueries.incrementAndGet();
                    }
                }
            };
            t.setDaemon(true);
            ctx.addThread(t);
        }
        ctx.startThreads();
        while (!blocksToTest.isEmpty() && ctx.shouldRun()) {
            Thread.sleep(10L);
        }
        ctx.stop();
        if ((double)hits.get() / ((double)hits.get() + (double)miss.get()) < passingScore) {
            Assert.fail((String)("Too many nulls returned. Hits: " + hits.get() + " Misses: " + miss.get()));
        }
    }

    public static void testCacheSimple(BlockCache toBeTested, int blockSize, int numBlocks) throws Exception {
        HFileBlockPair[] blocks;
        for (HFileBlockPair block : blocks = CacheTestUtils.generateHFileBlocks(numBlocks, blockSize)) {
            Assert.assertNull((Object)toBeTested.getBlock(block.blockName, true, false, true));
        }
        for (HFileBlockPair block : blocks) {
            toBeTested.cacheBlock(block.blockName, (Cacheable)block.block);
        }
        for (HFileBlockPair block : blocks) {
            HFileBlock buf = (HFileBlock)toBeTested.getBlock(block.blockName, true, false, true);
            if (buf == null) continue;
            Assert.assertEquals((Object)block.block, (Object)buf);
        }
        for (HFileBlockPair block : blocks) {
            try {
                if (toBeTested.getBlock(block.blockName, true, false, true) == null) continue;
                toBeTested.cacheBlock(block.blockName, (Cacheable)block.block);
                if (toBeTested instanceof BucketCache) continue;
                Assert.fail((String)"Cache should not allow re-caching a block");
            }
            catch (RuntimeException runtimeException) {
                // empty catch block
            }
        }
    }

    public static void hammerSingleKey(final BlockCache toBeTested, int BlockSize, int numThreads, int numQueries) throws Exception {
        final BlockCacheKey key = new BlockCacheKey("key", 0L);
        final byte[] buf = new byte[5120];
        Arrays.fill(buf, (byte)5);
        ByteArrayCacheable bac = new ByteArrayCacheable(buf);
        Configuration conf = new Configuration();
        MultithreadedTestUtil.TestContext ctx = new MultithreadedTestUtil.TestContext(conf);
        final AtomicInteger totalQueries = new AtomicInteger();
        toBeTested.cacheBlock(key, (Cacheable)bac);
        for (int i = 0; i < numThreads; ++i) {
            MultithreadedTestUtil.RepeatingTestThread t = new MultithreadedTestUtil.RepeatingTestThread(ctx){

                @Override
                public void doAnAction() throws Exception {
                    ByteArrayCacheable returned = (ByteArrayCacheable)toBeTested.getBlock(key, false, false, true);
                    Assert.assertArrayEquals((byte[])buf, (byte[])returned.buf);
                    totalQueries.incrementAndGet();
                }
            };
            t.setDaemon(true);
            ctx.addThread(t);
        }
        ctx.startThreads();
        while (totalQueries.get() < numQueries && ctx.shouldRun()) {
            Thread.sleep(10L);
        }
        ctx.stop();
    }

    public static void hammerEviction(final BlockCache toBeTested, int BlockSize, int numThreads, int numQueries) throws Exception {
        Configuration conf = new Configuration();
        MultithreadedTestUtil.TestContext ctx = new MultithreadedTestUtil.TestContext(conf);
        final AtomicInteger totalQueries = new AtomicInteger();
        int i = 0;
        while (i < numThreads) {
            final int finalI = i++;
            final byte[] buf = new byte[5120];
            MultithreadedTestUtil.RepeatingTestThread t = new MultithreadedTestUtil.RepeatingTestThread(ctx){

                @Override
                public void doAnAction() throws Exception {
                    for (int j = 0; j < 100; ++j) {
                        BlockCacheKey key = new BlockCacheKey("key_" + finalI + "_" + j, 0L);
                        Arrays.fill(buf, (byte)(finalI * j));
                        ByteArrayCacheable bac = new ByteArrayCacheable(buf);
                        ByteArrayCacheable gotBack = (ByteArrayCacheable)toBeTested.getBlock(key, true, false, true);
                        if (gotBack != null) {
                            Assert.assertArrayEquals((byte[])gotBack.buf, (byte[])bac.buf);
                            continue;
                        }
                        toBeTested.cacheBlock(key, (Cacheable)bac);
                    }
                    totalQueries.incrementAndGet();
                }
            };
            t.setDaemon(true);
            ctx.addThread(t);
        }
        ctx.startThreads();
        while (totalQueries.get() < numQueries && ctx.shouldRun()) {
            Thread.sleep(10L);
        }
        ctx.stop();
        Assert.assertTrue((toBeTested.getStats().getEvictedCount() > 0L ? 1 : 0) != 0);
    }

    public static HFileBlockPair[] generateHFileBlocks(int blockSize, int numBlocks) {
        HFileBlockPair[] returnedBlocks = new HFileBlockPair[numBlocks];
        Random rand = new Random();
        HashSet<String> usedStrings = new HashSet<String>();
        for (int i = 0; i < numBlocks; ++i) {
            ByteBuffer cachedBuffer = ByteBuffer.allocate(blockSize - 13);
            rand.nextBytes(cachedBuffer.array());
            cachedBuffer.rewind();
            int onDiskSizeWithoutHeader = blockSize - 13;
            int uncompressedSizeWithoutHeader = blockSize - 13;
            long prevBlockOffset = rand.nextLong();
            BlockType.DATA.write(cachedBuffer);
            cachedBuffer.putInt(onDiskSizeWithoutHeader);
            cachedBuffer.putInt(uncompressedSizeWithoutHeader);
            cachedBuffer.putLong(prevBlockOffset);
            cachedBuffer.rewind();
            HFileContext meta = new HFileContextBuilder().withHBaseCheckSum(false).withIncludesMvcc(true).withIncludesTags(false).withCompression(Compression.Algorithm.NONE).withBytesPerCheckSum(0).withChecksumType(ChecksumType.NULL).build();
            HFileBlock generated = new HFileBlock(BlockType.DATA, onDiskSizeWithoutHeader, uncompressedSizeWithoutHeader, prevBlockOffset, cachedBuffer, false, (long)blockSize, onDiskSizeWithoutHeader + 33, meta);
            String strKey = new Long(rand.nextLong()).toString();
            while (!usedStrings.add(strKey)) {
                strKey = new Long(rand.nextLong()).toString();
            }
            returnedBlocks[i] = new HFileBlockPair();
            returnedBlocks[i].blockName = new BlockCacheKey(strKey, 0L);
            returnedBlocks[i].block = generated;
        }
        return returnedBlocks;
    }

    @VisibleForTesting
    public static class HFileBlockPair {
        BlockCacheKey blockName;
        HFileBlock block;

        public BlockCacheKey getBlockName() {
            return this.blockName;
        }

        public HFileBlock getBlock() {
            return this.block;
        }
    }

    public static class ByteArrayCacheable
    implements Cacheable {
        static final CacheableDeserializer<Cacheable> blockDeserializer = new CacheableDeserializer<Cacheable>(){

            public Cacheable deserialize(ByteBuffer b) throws IOException {
                int len = b.getInt();
                Thread.yield();
                byte[] buf = new byte[len];
                b.get(buf);
                return new ByteArrayCacheable(buf);
            }

            public int getDeserialiserIdentifier() {
                return deserializerIdentifier;
            }

            public Cacheable deserialize(ByteBuffer b, boolean reuse) throws IOException {
                return this.deserialize(b);
            }
        };
        final byte[] buf;
        private static final int deserializerIdentifier = CacheableDeserializerIdManager.registerDeserializer(blockDeserializer);

        public ByteArrayCacheable(byte[] buf) {
            this.buf = buf;
        }

        public long heapSize() {
            return 4 + this.buf.length;
        }

        public int getSerializedLength() {
            return 4 + this.buf.length;
        }

        public void serialize(ByteBuffer destination) {
            destination.putInt(this.buf.length);
            Thread.yield();
            destination.put(this.buf);
            destination.rewind();
        }

        public CacheableDeserializer<Cacheable> getDeserializer() {
            return blockDeserializer;
        }

        public BlockType getBlockType() {
            return BlockType.DATA;
        }
    }
}

