/*
 * Decompiled with CFR 0.152.
 */
package org.apache.calcite.adapter.druid;

import com.google.common.base.Function;
import com.google.common.collect.BoundType;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Lists;
import com.google.common.collect.Range;
import com.google.common.collect.TreeRangeSet;
import java.sql.Date;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Calendar;
import java.util.List;
import java.util.regex.Pattern;
import org.apache.calcite.adapter.druid.DruidTable;
import org.apache.calcite.avatica.util.DateTimeUtils;
import org.apache.calcite.avatica.util.TimeUnitRange;
import org.apache.calcite.rel.type.RelDataType;
import org.apache.calcite.rex.RexCall;
import org.apache.calcite.rex.RexInputRef;
import org.apache.calcite.rex.RexLiteral;
import org.apache.calcite.rex.RexNode;
import org.apache.calcite.sql.SqlKind;
import org.apache.calcite.util.trace.CalciteTrace;
import org.joda.time.Interval;
import org.joda.time.chrono.ISOChronology;
import org.slf4j.Logger;

public class DruidDateTimeUtils {
    protected static final Logger LOGGER = CalciteTrace.getPlannerTracer();
    private static final Pattern TIMESTAMP_PATTERN = Pattern.compile("[0-9][0-9][0-9][0-9]-[0-9][0-9]-[0-9][0-9] [0-9][0-9]:[0-9][0-9]:[0-9][0-9]");

    private DruidDateTimeUtils() {
    }

    public static List<Interval> createInterval(RelDataType type, RexNode e) {
        List<Range> ranges = DruidDateTimeUtils.extractRanges(type, e, false);
        if (ranges == null) {
            return null;
        }
        TreeRangeSet condensedRanges = TreeRangeSet.create();
        for (Range r : ranges) {
            condensedRanges.add(r);
        }
        if (LOGGER.isDebugEnabled()) {
            LOGGER.debug("Inferred ranges on interval : " + condensedRanges);
        }
        return DruidDateTimeUtils.toInterval(ImmutableList.copyOf(condensedRanges.asRanges()));
    }

    protected static List<Interval> toInterval(List<Range> ranges) {
        List<Interval> intervals = Lists.transform(ranges, new Function<Range, Interval>(){

            @Override
            public Interval apply(Range range) {
                long end;
                if (!range.hasLowerBound() && !range.hasUpperBound()) {
                    return DruidTable.DEFAULT_INTERVAL;
                }
                long start = range.hasLowerBound() ? DruidDateTimeUtils.toLong(range.lowerEndpoint()).longValue() : DruidTable.DEFAULT_INTERVAL.getStartMillis();
                long l = end = range.hasUpperBound() ? DruidDateTimeUtils.toLong(range.upperEndpoint()).longValue() : DruidTable.DEFAULT_INTERVAL.getEndMillis();
                if (range.hasLowerBound() && range.lowerBoundType() == BoundType.OPEN) {
                    ++start;
                }
                if (range.hasUpperBound() && range.upperBoundType() == BoundType.CLOSED) {
                    ++end;
                }
                return new Interval(start, end, ISOChronology.getInstanceUTC());
            }
        });
        if (LOGGER.isInfoEnabled()) {
            LOGGER.info("Converted time ranges " + ranges + " to interval " + intervals);
        }
        return intervals;
    }

    protected static List<Range> extractRanges(RelDataType type, RexNode node, boolean withNot) {
        switch (node.getKind()) {
            case EQUALS: 
            case LESS_THAN: 
            case LESS_THAN_OR_EQUAL: 
            case GREATER_THAN: 
            case GREATER_THAN_OR_EQUAL: 
            case BETWEEN: 
            case IN: {
                return DruidDateTimeUtils.leafToRanges(type, (RexCall)node, withNot);
            }
            case NOT: {
                return DruidDateTimeUtils.extractRanges(type, ((RexCall)node).getOperands().get(0), !withNot);
            }
            case OR: {
                RexCall call = (RexCall)node;
                ArrayList<Range> intervals = Lists.newArrayList();
                for (RexNode child : call.getOperands()) {
                    List<Range> extracted = DruidDateTimeUtils.extractRanges(type, child, withNot);
                    if (extracted == null) continue;
                    intervals.addAll(extracted);
                }
                return intervals;
            }
            case AND: {
                RexCall call = (RexCall)node;
                ArrayList ranges = new ArrayList<Range>();
                for (RexNode child : call.getOperands()) {
                    List<Range> extractedRanges = DruidDateTimeUtils.extractRanges(type, child, false);
                    if (extractedRanges == null || extractedRanges.isEmpty()) {
                        return null;
                    }
                    if (ranges.isEmpty()) {
                        ranges.addAll(extractedRanges);
                        continue;
                    }
                    ArrayList overlapped = Lists.newArrayList();
                    for (Range current : ranges) {
                        for (Range interval : extractedRanges) {
                            if (!current.isConnected(interval)) continue;
                            overlapped.add(current.intersection(interval));
                        }
                    }
                    ranges = overlapped;
                }
                return ranges;
            }
        }
        return null;
    }

    protected static List<Range> leafToRanges(RelDataType type, RexCall call, boolean withNot) {
        switch (call.getKind()) {
            case EQUALS: 
            case LESS_THAN: 
            case LESS_THAN_OR_EQUAL: 
            case GREATER_THAN: 
            case GREATER_THAN_OR_EQUAL: {
                RexLiteral literal = null;
                if (call.getOperands().get(0) instanceof RexInputRef && call.getOperands().get(1) instanceof RexLiteral) {
                    literal = DruidDateTimeUtils.extractLiteral(call.getOperands().get(1));
                } else if (call.getOperands().get(0) instanceof RexInputRef && call.getOperands().get(1).getKind() == SqlKind.CAST) {
                    literal = DruidDateTimeUtils.extractLiteral(call.getOperands().get(1));
                } else if (call.getOperands().get(1) instanceof RexInputRef && call.getOperands().get(0) instanceof RexLiteral) {
                    literal = DruidDateTimeUtils.extractLiteral(call.getOperands().get(0));
                } else if (call.getOperands().get(1) instanceof RexInputRef && call.getOperands().get(0).getKind() == SqlKind.CAST) {
                    literal = DruidDateTimeUtils.extractLiteral(call.getOperands().get(0));
                }
                if (literal == null) {
                    return null;
                }
                Comparable value = DruidDateTimeUtils.literalToType(literal, type);
                if (value == null) {
                    return null;
                }
                switch (call.getKind()) {
                    case LESS_THAN: {
                        return Arrays.asList(withNot ? Range.atLeast(value) : Range.lessThan(value));
                    }
                    case LESS_THAN_OR_EQUAL: {
                        return Arrays.asList(withNot ? Range.greaterThan(value) : Range.atMost(value));
                    }
                    case GREATER_THAN: {
                        return Arrays.asList(withNot ? Range.atMost(value) : Range.greaterThan(value));
                    }
                    case GREATER_THAN_OR_EQUAL: {
                        return Arrays.asList(withNot ? Range.lessThan(value) : Range.atLeast(value));
                    }
                }
                if (!withNot) {
                    return Arrays.asList(Range.closed(value, value));
                }
                return Arrays.asList(Range.lessThan(value), Range.greaterThan(value));
            }
            case BETWEEN: {
                boolean inverted;
                RexLiteral literal1 = DruidDateTimeUtils.extractLiteral(call.getOperands().get(2));
                if (literal1 == null) {
                    return null;
                }
                RexLiteral literal2 = DruidDateTimeUtils.extractLiteral(call.getOperands().get(3));
                if (literal2 == null) {
                    return null;
                }
                Comparable value1 = DruidDateTimeUtils.literalToType(literal1, type);
                Comparable value2 = DruidDateTimeUtils.literalToType(literal2, type);
                if (value1 == null || value2 == null) {
                    return null;
                }
                boolean bl = inverted = value1.compareTo(value2) > 0;
                if (!withNot) {
                    return Arrays.asList(inverted ? Range.closed(value2, value1) : Range.closed(value1, value2));
                }
                return Arrays.asList(Range.lessThan(inverted ? value2 : value1), Range.greaterThan(inverted ? value1 : value2));
            }
            case IN: {
                ArrayList<Range> ranges = Lists.newArrayList();
                for (int i = 1; i < call.getOperands().size(); ++i) {
                    RexLiteral literal = DruidDateTimeUtils.extractLiteral(call.getOperands().get(i));
                    if (literal == null) {
                        return null;
                    }
                    Comparable element = DruidDateTimeUtils.literalToType(literal, type);
                    if (element == null) {
                        return null;
                    }
                    if (withNot) {
                        ranges.addAll(Arrays.asList(Range.lessThan(element), Range.greaterThan(element)));
                        continue;
                    }
                    ranges.add(Range.closed(element, element));
                }
                return ranges;
            }
        }
        return null;
    }

    protected static Comparable literalToType(RexLiteral literal, RelDataType type) {
        Object value = null;
        switch (literal.getType().getSqlTypeName()) {
            case DATE: 
            case TIME: 
            case TIMESTAMP: 
            case INTERVAL_YEAR_MONTH: 
            case INTERVAL_DAY: {
                value = literal.getValue();
                break;
            }
            case TINYINT: 
            case SMALLINT: 
            case INTEGER: 
            case BIGINT: 
            case DOUBLE: 
            case DECIMAL: 
            case FLOAT: 
            case REAL: 
            case VARCHAR: 
            case CHAR: 
            case BOOLEAN: {
                value = literal.getValue3();
            }
        }
        if (value == null) {
            return null;
        }
        switch (type.getSqlTypeName()) {
            case BIGINT: {
                return DruidDateTimeUtils.toLong(value);
            }
            case INTEGER: {
                return DruidDateTimeUtils.toInt(value);
            }
            case FLOAT: {
                return DruidDateTimeUtils.toFloat(value);
            }
            case DOUBLE: {
                return DruidDateTimeUtils.toDouble(value);
            }
            case VARCHAR: 
            case CHAR: {
                return String.valueOf(value);
            }
            case TIMESTAMP: {
                return DruidDateTimeUtils.toTimestamp(value);
            }
        }
        return null;
    }

    private static RexLiteral extractLiteral(RexNode node) {
        RexNode target = node;
        if (node.getKind() == SqlKind.CAST) {
            target = ((RexCall)node).getOperands().get(0);
        }
        if (!(target instanceof RexLiteral)) {
            return null;
        }
        return (RexLiteral)target;
    }

    private static Comparable toTimestamp(Object literal) {
        if (literal instanceof Timestamp) {
            return (Timestamp)literal;
        }
        Long v = DruidDateTimeUtils.toLong(literal);
        if (v != null) {
            return new Timestamp(v);
        }
        return null;
    }

    private static Long toLong(Object literal) {
        block7: {
            if (literal instanceof Number) {
                return ((Number)literal).longValue();
            }
            if (literal instanceof Date) {
                return ((Date)literal).getTime();
            }
            if (literal instanceof Timestamp) {
                return ((Timestamp)literal).getTime();
            }
            if (literal instanceof Calendar) {
                return ((Calendar)literal).getTime().getTime();
            }
            if (literal instanceof String) {
                String s = (String)literal;
                try {
                    return Long.valueOf(s);
                }
                catch (NumberFormatException numberFormatException) {
                    if (!TIMESTAMP_PATTERN.matcher(s).matches()) break block7;
                    return DateTimeUtils.timestampStringToUnixDate(s);
                }
            }
        }
        return null;
    }

    private static Integer toInt(Object literal) {
        if (literal instanceof Number) {
            return ((Number)literal).intValue();
        }
        if (literal instanceof String) {
            try {
                return Integer.valueOf((String)literal);
            }
            catch (NumberFormatException numberFormatException) {
                // empty catch block
            }
        }
        return null;
    }

    private static Float toFloat(Object literal) {
        if (literal instanceof Number) {
            return Float.valueOf(((Number)literal).floatValue());
        }
        if (literal instanceof String) {
            try {
                return Float.valueOf((String)literal);
            }
            catch (NumberFormatException numberFormatException) {
                // empty catch block
            }
        }
        return null;
    }

    private static Double toDouble(Object literal) {
        if (literal instanceof Number) {
            return ((Number)literal).doubleValue();
        }
        if (literal instanceof String) {
            try {
                return Double.valueOf((String)literal);
            }
            catch (NumberFormatException numberFormatException) {
                // empty catch block
            }
        }
        return null;
    }

    public static long extractTotalTime(List<Interval> intervals) {
        long totalTime = 0L;
        for (Interval interval : intervals) {
            totalTime += interval.getEndMillis() - interval.getStartMillis();
        }
        return totalTime;
    }

    public static String extractGranularity(RexCall call) {
        if (call.getKind() != SqlKind.FLOOR || call.getOperands().size() != 2) {
            return null;
        }
        RexLiteral flag = (RexLiteral)call.operands.get(1);
        TimeUnitRange timeUnit = (TimeUnitRange)((Object)flag.getValue());
        if (timeUnit == null) {
            return null;
        }
        switch (timeUnit) {
            case YEAR: 
            case QUARTER: 
            case MONTH: 
            case WEEK: 
            case DAY: 
            case HOUR: 
            case MINUTE: 
            case SECOND: {
                return timeUnit.name();
            }
        }
        return null;
    }
}

