/*
 * Decompiled with CFR 0.152.
 */
package org.apache.jackrabbit.aws.ext.ds;

import com.amazonaws.AmazonClientException;
import com.amazonaws.AmazonServiceException;
import com.amazonaws.event.ProgressEvent;
import com.amazonaws.event.ProgressListener;
import com.amazonaws.services.s3.AmazonS3;
import com.amazonaws.services.s3.AmazonS3Client;
import com.amazonaws.services.s3.model.CopyObjectRequest;
import com.amazonaws.services.s3.model.DeleteObjectsRequest;
import com.amazonaws.services.s3.model.DeleteObjectsResult;
import com.amazonaws.services.s3.model.ObjectListing;
import com.amazonaws.services.s3.model.ObjectMetadata;
import com.amazonaws.services.s3.model.PutObjectRequest;
import com.amazonaws.services.s3.model.Region;
import com.amazonaws.services.s3.model.S3Object;
import com.amazonaws.services.s3.model.S3ObjectInputStream;
import com.amazonaws.services.s3.model.S3ObjectSummary;
import com.amazonaws.services.s3.transfer.TransferManager;
import com.amazonaws.services.s3.transfer.Upload;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Properties;
import java.util.Set;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import org.apache.jackrabbit.aws.ext.Utils;
import org.apache.jackrabbit.core.data.AsyncUploadCallback;
import org.apache.jackrabbit.core.data.AsyncUploadResult;
import org.apache.jackrabbit.core.data.Backend;
import org.apache.jackrabbit.core.data.CachingDataStore;
import org.apache.jackrabbit.core.data.DataIdentifier;
import org.apache.jackrabbit.core.data.DataStoreException;
import org.apache.jackrabbit.core.data.util.NamedThreadFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class S3Backend
implements Backend {
    private static final Logger LOG = LoggerFactory.getLogger(S3Backend.class);
    private static final String KEY_PREFIX = "dataStore_";
    private static final String DEFAULT_AWS_BUCKET_REGION = "us-standard";
    private static final String AWSDOTCOM = "amazonaws.com";
    private static final String S3 = "s3";
    private static final String DOT = ".";
    private static final String DASH = "-";
    private AmazonS3Client s3service;
    private String bucket;
    private TransferManager tmx;
    private CachingDataStore store;
    private Properties properties;
    private Date startTime;
    private ThreadPoolExecutor asyncWriteExecuter;

    public void init(CachingDataStore store, String homeDir, String config) throws DataStoreException {
        Properties initProps = null;
        if (this.properties != null) {
            initProps = this.properties;
        } else {
            if (config == null) {
                config = "aws.properties";
            }
            try {
                initProps = Utils.readConfig(config);
            }
            catch (IOException e) {
                throw new DataStoreException("Could not initialize S3 from " + config, (Throwable)e);
            }
            this.properties = initProps;
        }
        this.init(store, homeDir, initProps);
    }

    public void init(CachingDataStore store, String homeDir, Properties prop) throws DataStoreException {
        ClassLoader contextClassLoader = Thread.currentThread().getContextClassLoader();
        try {
            boolean renameKeyBool;
            this.startTime = new Date();
            Thread.currentThread().setContextClassLoader(this.getClass().getClassLoader());
            LOG.debug("init");
            this.store = store;
            this.s3service = Utils.openService(prop);
            if (this.bucket == null || "".equals(this.bucket.trim())) {
                this.bucket = prop.getProperty("s3Bucket");
            }
            String region = prop.getProperty("s3Region");
            Region s3Region = null;
            String endpoint = null;
            if (DEFAULT_AWS_BUCKET_REGION.equals(region)) {
                s3Region = Region.US_Standard;
                endpoint = "s3.amazonaws.com";
            } else if (Region.EU_Ireland.toString().equals(region)) {
                s3Region = Region.EU_Ireland;
                endpoint = "s3-eu-west-1.amazonaws.com";
            } else {
                s3Region = Region.fromValue((String)region);
                endpoint = "s3-" + region + DOT + AWSDOTCOM;
            }
            String propEndPoint = prop.getProperty("s3EndPoint");
            if (propEndPoint != null & !"".equals(propEndPoint)) {
                endpoint = propEndPoint;
            }
            this.s3service.setEndpoint(endpoint);
            LOG.info("S3 service endpoint [{}] ", (Object)endpoint);
            if (!this.s3service.doesBucketExist(this.bucket)) {
                this.s3service.createBucket(this.bucket, s3Region);
                LOG.info("Created bucket [{}] in [{}] ", (Object)this.bucket, (Object)region);
            }
            int writeThreads = 10;
            String writeThreadsStr = prop.getProperty("writeThreads");
            if (writeThreadsStr != null) {
                writeThreads = Integer.parseInt(writeThreadsStr);
            }
            LOG.info("Using thread pool of [{}] threads in S3 transfer manager.", (Object)writeThreads);
            this.tmx = new TransferManager((AmazonS3)this.s3service, (ThreadPoolExecutor)Executors.newFixedThreadPool(writeThreads, (ThreadFactory)new NamedThreadFactory("s3-transfer-manager-worker")));
            int asyncWritePoolSize = 10;
            String maxConnsStr = prop.getProperty("maxConnections");
            if (maxConnsStr != null) {
                asyncWritePoolSize = Integer.parseInt(maxConnsStr) - writeThreads;
            }
            this.asyncWriteExecuter = (ThreadPoolExecutor)Executors.newFixedThreadPool(asyncWritePoolSize, (ThreadFactory)new NamedThreadFactory("s3-write-worker"));
            String renameKeyProp = prop.getProperty("s3RenameKeys");
            boolean bl = renameKeyBool = renameKeyProp == null || "".equals(renameKeyProp) ? true : Boolean.parseBoolean(renameKeyProp);
            if (renameKeyBool) {
                this.renameKeys();
            }
            LOG.debug("S3 Backend initialized in [{}] ms", (Object)(System.currentTimeMillis() - this.startTime.getTime()));
        }
        catch (Exception e) {
            LOG.debug("  error ", (Throwable)e);
            throw new DataStoreException("Could not initialize S3 from " + prop, (Throwable)e);
        }
        finally {
            if (contextClassLoader != null) {
                Thread.currentThread().setContextClassLoader(contextClassLoader);
            }
        }
    }

    public void write(DataIdentifier identifier, File file) throws DataStoreException {
        this.write(identifier, file, false, null);
    }

    public void writeAsync(DataIdentifier identifier, File file, AsyncUploadCallback callback) throws DataStoreException {
        if (callback == null) {
            throw new IllegalArgumentException("callback parameter cannot be null in asyncUpload");
        }
        this.asyncWriteExecuter.execute(new AsyncUploadJob(identifier, file, callback));
    }

    public boolean exists(DataIdentifier identifier) throws DataStoreException {
        long start = System.currentTimeMillis();
        String key = S3Backend.getKeyName(identifier);
        ClassLoader contextClassLoader = Thread.currentThread().getContextClassLoader();
        try {
            Thread.currentThread().setContextClassLoader(this.getClass().getClassLoader());
            ObjectMetadata objectMetaData = this.s3service.getObjectMetadata(this.bucket, key);
            if (objectMetaData != null) {
                LOG.debug("exists [{}]: [true] took [{}] ms.", (Object)identifier, (Object)(System.currentTimeMillis() - start));
                boolean bl = true;
                return bl;
            }
            boolean bl = false;
            return bl;
        }
        catch (AmazonServiceException e) {
            if (e.getStatusCode() == 404) {
                LOG.debug("exists [{}]: [false] took [{}] ms.", (Object)identifier, (Object)(System.currentTimeMillis() - start));
                boolean bl = false;
                return bl;
            }
            throw new DataStoreException("Error occured to getObjectMetadata for key [" + identifier.toString() + "]", (Throwable)e);
        }
        finally {
            if (contextClassLoader != null) {
                Thread.currentThread().setContextClassLoader(contextClassLoader);
            }
        }
    }

    public boolean exists(DataIdentifier identifier, boolean touch) throws DataStoreException {
        boolean retVal;
        long start;
        block12: {
            start = System.currentTimeMillis();
            String key = S3Backend.getKeyName(identifier);
            ObjectMetadata objectMetaData = null;
            retVal = false;
            ClassLoader contextClassLoader = Thread.currentThread().getContextClassLoader();
            try {
                Thread.currentThread().setContextClassLoader(this.getClass().getClassLoader());
                objectMetaData = this.s3service.getObjectMetadata(this.bucket, key);
                if (objectMetaData != null) {
                    retVal = true;
                    if (touch) {
                        CopyObjectRequest copReq = new CopyObjectRequest(this.bucket, key, this.bucket, key);
                        copReq.setNewObjectMetadata(objectMetaData);
                        this.s3service.copyObject(copReq);
                        LOG.debug("[{}] touched took [{}] ms. ", (Object)identifier, (Object)(System.currentTimeMillis() - start));
                    }
                } else {
                    retVal = false;
                }
            }
            catch (AmazonServiceException e) {
                if (e.getStatusCode() == 404) {
                    retVal = false;
                    break block12;
                }
                throw new DataStoreException("Error occured to find exists for key [" + identifier.toString() + "]", (Throwable)e);
            }
            catch (Exception e) {
                throw new DataStoreException("Error occured to find exists for key  " + identifier.toString(), (Throwable)e);
            }
            finally {
                if (contextClassLoader != null) {
                    Thread.currentThread().setContextClassLoader(contextClassLoader);
                }
            }
        }
        LOG.debug("exists [{}]: [{}] took [{}] ms.", new Object[]{identifier, retVal, System.currentTimeMillis() - start});
        return retVal;
    }

    public InputStream read(DataIdentifier identifier) throws DataStoreException {
        long start = System.currentTimeMillis();
        String key = S3Backend.getKeyName(identifier);
        ClassLoader contextClassLoader = Thread.currentThread().getContextClassLoader();
        try {
            Thread.currentThread().setContextClassLoader(this.getClass().getClassLoader());
            S3Object object = this.s3service.getObject(this.bucket, key);
            S3ObjectInputStream in = object.getObjectContent();
            LOG.debug("[{}] read took [{}]ms", (Object)identifier, (Object)(System.currentTimeMillis() - start));
            S3ObjectInputStream s3ObjectInputStream = in;
            return s3ObjectInputStream;
        }
        catch (AmazonServiceException e) {
            throw new DataStoreException("Object not found: " + key, (Throwable)e);
        }
        finally {
            if (contextClassLoader != null) {
                Thread.currentThread().setContextClassLoader(contextClassLoader);
            }
        }
    }

    public Iterator<DataIdentifier> getAllIdentifiers() throws DataStoreException {
        long start = System.currentTimeMillis();
        ClassLoader contextClassLoader = Thread.currentThread().getContextClassLoader();
        try {
            Thread.currentThread().setContextClassLoader(this.getClass().getClassLoader());
            HashSet<DataIdentifier> ids = new HashSet<DataIdentifier>();
            ObjectListing prevObjectListing = this.s3service.listObjects(this.bucket);
            while (true) {
                for (S3ObjectSummary s3ObjSumm : prevObjectListing.getObjectSummaries()) {
                    String id = S3Backend.getIdentifierName(s3ObjSumm.getKey());
                    if (id == null) continue;
                    ids.add(new DataIdentifier(id));
                }
                if (!prevObjectListing.isTruncated()) break;
                prevObjectListing = this.s3service.listNextBatchOfObjects(prevObjectListing);
            }
            LOG.debug("getAllIdentifiers returned size [{}] took [{}] ms.", (Object)ids.size(), (Object)(System.currentTimeMillis() - start));
            Iterator<DataIdentifier> iterator = ids.iterator();
            return iterator;
        }
        catch (AmazonServiceException e) {
            throw new DataStoreException("Could not list objects", (Throwable)e);
        }
        finally {
            if (contextClassLoader != null) {
                Thread.currentThread().setContextClassLoader(contextClassLoader);
            }
        }
    }

    public long getLastModified(DataIdentifier identifier) throws DataStoreException {
        long start = System.currentTimeMillis();
        String key = S3Backend.getKeyName(identifier);
        ClassLoader contextClassLoader = Thread.currentThread().getContextClassLoader();
        try {
            Thread.currentThread().setContextClassLoader(this.getClass().getClassLoader());
            ObjectMetadata object = this.s3service.getObjectMetadata(this.bucket, key);
            long lastModified = object.getLastModified().getTime();
            LOG.debug("Identifier [{}]'s lastModified = [{}] took [{}]ms.", new Object[]{identifier, lastModified, System.currentTimeMillis() - start});
            long l = lastModified;
            return l;
        }
        catch (AmazonServiceException e) {
            if (e.getStatusCode() == 404) {
                LOG.info("getLastModified:Identifier [{}] not found. Took [{}] ms.", (Object)identifier, (Object)(System.currentTimeMillis() - start));
            }
            throw new DataStoreException((Throwable)e);
        }
        finally {
            if (contextClassLoader != null) {
                Thread.currentThread().setContextClassLoader(contextClassLoader);
            }
        }
    }

    public long getLength(DataIdentifier identifier) throws DataStoreException {
        long start = System.currentTimeMillis();
        String key = S3Backend.getKeyName(identifier);
        ClassLoader contextClassLoader = Thread.currentThread().getContextClassLoader();
        try {
            Thread.currentThread().setContextClassLoader(this.getClass().getClassLoader());
            ObjectMetadata object = this.s3service.getObjectMetadata(this.bucket, key);
            long length = object.getContentLength();
            LOG.debug("Identifier [{}]'s length = [{}] took [{}]ms.", new Object[]{identifier, length, System.currentTimeMillis() - start});
            long l = length;
            return l;
        }
        catch (AmazonServiceException e) {
            throw new DataStoreException("Could not length of dataIdentifier " + identifier, (Throwable)e);
        }
        finally {
            if (contextClassLoader != null) {
                Thread.currentThread().setContextClassLoader(contextClassLoader);
            }
        }
    }

    public void deleteRecord(DataIdentifier identifier) throws DataStoreException {
        long start = System.currentTimeMillis();
        String key = S3Backend.getKeyName(identifier);
        ClassLoader contextClassLoader = Thread.currentThread().getContextClassLoader();
        try {
            Thread.currentThread().setContextClassLoader(this.getClass().getClassLoader());
            this.s3service.deleteObject(this.bucket, key);
            LOG.debug("Identifier [{}] deleted. It took [{}]ms.", new Object[]{identifier, System.currentTimeMillis() - start});
        }
        catch (AmazonServiceException e) {
            throw new DataStoreException("Could not getLastModified of dataIdentifier " + identifier, (Throwable)e);
        }
        finally {
            if (contextClassLoader != null) {
                Thread.currentThread().setContextClassLoader(contextClassLoader);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public Set<DataIdentifier> deleteAllOlderThan(long min) throws DataStoreException {
        long start = System.currentTimeMillis();
        min -= 1000L;
        HashSet<DataIdentifier> deleteIdSet = new HashSet<DataIdentifier>(30);
        ClassLoader contextClassLoader = Thread.currentThread().getContextClassLoader();
        try {
            Thread.currentThread().setContextClassLoader(this.getClass().getClassLoader());
            ObjectListing prevObjectListing = this.s3service.listObjects(this.bucket);
            while (true) {
                ArrayList<DeleteObjectsRequest.KeyVersion> deleteList = new ArrayList<DeleteObjectsRequest.KeyVersion>();
                for (S3ObjectSummary s3ObjSumm : prevObjectListing.getObjectSummaries()) {
                    DataIdentifier identifier = new DataIdentifier(S3Backend.getIdentifierName(s3ObjSumm.getKey()));
                    long lastModified = s3ObjSumm.getLastModified().getTime();
                    LOG.debug("Identifier [{}]'s lastModified = [{}]", (Object)identifier, (Object)lastModified);
                    if (this.store.isInUse(identifier) || lastModified >= min) continue;
                    LOG.debug("add id [{}] to delete lists", (Object)s3ObjSumm.getKey());
                    deleteList.add(new DeleteObjectsRequest.KeyVersion(s3ObjSumm.getKey()));
                    deleteIdSet.add(identifier);
                }
                if (deleteList.size() > 0) {
                    DeleteObjectsRequest delObjsReq = new DeleteObjectsRequest(this.bucket);
                    delObjsReq.setKeys(deleteList);
                    DeleteObjectsResult dobjs = this.s3service.deleteObjects(delObjsReq);
                    if (dobjs.getDeletedObjects().size() != deleteList.size()) {
                        throw new DataStoreException("Incomplete delete object request. only  " + dobjs.getDeletedObjects().size() + " out of " + deleteList.size() + " are deleted");
                    }
                    LOG.debug("[{}] records deleted from datastore", deleteList);
                }
                if (!prevObjectListing.isTruncated()) {
                    break;
                }
                prevObjectListing = this.s3service.listNextBatchOfObjects(prevObjectListing);
            }
        }
        finally {
            if (contextClassLoader != null) {
                Thread.currentThread().setContextClassLoader(contextClassLoader);
            }
        }
        LOG.info("deleteAllOlderThan: min=[{}] exit. Deleted[{}] records. Number of records deleted [{}] took [{}]ms", new Object[]{min, deleteIdSet, deleteIdSet.size(), System.currentTimeMillis() - start});
        return deleteIdSet;
    }

    public void close() {
        this.tmx.abortMultipartUploads(this.bucket, this.startTime);
        this.tmx.shutdownNow();
        this.s3service.shutdown();
        this.asyncWriteExecuter.shutdownNow();
        LOG.info("S3Backend closed.");
    }

    public String getBucket() {
        return this.bucket;
    }

    public void setBucket(String bucket) {
        this.bucket = bucket;
    }

    public void setProperties(Properties properties) {
        this.properties = properties;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void write(DataIdentifier identifier, File file, boolean asyncUpload, AsyncUploadCallback callback) throws DataStoreException {
        long start;
        block16: {
            String key = S3Backend.getKeyName(identifier);
            ObjectMetadata objectMetaData = null;
            start = System.currentTimeMillis();
            ClassLoader contextClassLoader = Thread.currentThread().getContextClassLoader();
            try {
                block15: {
                    Thread.currentThread().setContextClassLoader(this.getClass().getClassLoader());
                    try {
                        objectMetaData = this.s3service.getObjectMetadata(this.bucket, key);
                    }
                    catch (AmazonServiceException ase) {
                        if (ase.getStatusCode() == 404) break block15;
                        throw ase;
                    }
                }
                if (objectMetaData != null) {
                    long l = objectMetaData.getContentLength();
                    if (l != file.length()) {
                        throw new DataStoreException("Collision: " + key + " new length: " + file.length() + " old length: " + l);
                    }
                    LOG.debug("[{}]'s exists, lastmodified = [{}]", (Object)key, (Object)objectMetaData.getLastModified().getTime());
                    CopyObjectRequest copReq = new CopyObjectRequest(this.bucket, key, this.bucket, key);
                    copReq.setNewObjectMetadata(objectMetaData);
                    this.s3service.copyObject(copReq);
                    LOG.debug("lastModified of [{}] updated successfully.", (Object)identifier);
                    if (callback != null) {
                        callback.onSuccess(new AsyncUploadResult(identifier, file));
                    }
                }
                if (objectMetaData != null) break block16;
                try {
                    Upload up = this.tmx.upload(new PutObjectRequest(this.bucket, key, file));
                    if (asyncUpload) {
                        up.addProgressListener((ProgressListener)new S3UploadProgressListener(up, identifier, file, callback));
                        LOG.debug("added upload progress listener to identifier [{}]", (Object)identifier);
                    } else {
                        up.waitForUploadResult();
                        LOG.debug("synchronous upload to identifier [{}] completed.", (Object)identifier);
                        if (callback != null) {
                            callback.onSuccess(new AsyncUploadResult(identifier, file));
                        }
                    }
                }
                catch (Exception e2) {
                    if (!asyncUpload) {
                        callback.onAbort(new AsyncUploadResult(identifier, file));
                    }
                    throw new DataStoreException("Could not upload " + key, (Throwable)e2);
                }
            }
            finally {
                if (contextClassLoader != null) {
                    Thread.currentThread().setContextClassLoader(contextClassLoader);
                }
            }
        }
        LOG.debug("write of [{}], length=[{}], in async mode [{}], in [{}]ms", new Object[]{identifier, file.length(), asyncUpload, System.currentTimeMillis() - start});
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void renameKeys() throws DataStoreException {
        block11: {
            long startTime = System.currentTimeMillis();
            ClassLoader contextClassLoader = Thread.currentThread().getContextClassLoader();
            long count = 0L;
            try {
                int endIndex;
                Thread.currentThread().setContextClassLoader(this.getClass().getClassLoader());
                ObjectListing prevObjectListing = this.s3service.listObjects(this.bucket, KEY_PREFIX);
                ArrayList<DeleteObjectsRequest.KeyVersion> deleteList = new ArrayList<DeleteObjectsRequest.KeyVersion>();
                int nThreads = Integer.parseInt(this.properties.getProperty("maxConnections"));
                ExecutorService executor = Executors.newFixedThreadPool(nThreads, (ThreadFactory)new NamedThreadFactory("s3-object-rename-worker"));
                boolean taskAdded = false;
                while (true) {
                    for (S3ObjectSummary s3ObjSumm : prevObjectListing.getObjectSummaries()) {
                        executor.execute(new KeyRenameThread(s3ObjSumm.getKey()));
                        taskAdded = true;
                        ++count;
                        deleteList.add(new DeleteObjectsRequest.KeyVersion(s3ObjSumm.getKey()));
                    }
                    if (!prevObjectListing.isTruncated()) break;
                    prevObjectListing = this.s3service.listNextBatchOfObjects(prevObjectListing);
                }
                executor.shutdown();
                try {
                    while (taskAdded && !executor.awaitTermination(10L, TimeUnit.SECONDS)) {
                        LOG.info("Rename S3 keys tasks timedout. Waiting again");
                    }
                }
                catch (InterruptedException ie) {
                    // empty catch block
                }
                LOG.info("Renamed [{}] keys, time taken [{}]sec", (Object)count, (Object)((System.currentTimeMillis() - startTime) / 1000L));
                if (deleteList.size() <= 0) break block11;
                DeleteObjectsRequest delObjsReq = new DeleteObjectsRequest(this.bucket);
                int batchSize = 500;
                int startIndex = 0;
                int size = deleteList.size();
                int n = endIndex = batchSize < size ? batchSize : size;
                while (endIndex <= size) {
                    delObjsReq.setKeys(Collections.unmodifiableList(deleteList.subList(startIndex, endIndex)));
                    DeleteObjectsResult dobjs = this.s3service.deleteObjects(delObjsReq);
                    LOG.info("Records[{}] deleted in datastore from index [{}] to [{}]", new Object[]{dobjs.getDeletedObjects().size(), startIndex, endIndex - 1});
                    if (endIndex == size) {
                        break;
                    }
                    startIndex = endIndex;
                    endIndex = startIndex + batchSize < size ? startIndex + batchSize : size;
                }
            }
            finally {
                if (contextClassLoader != null) {
                    Thread.currentThread().setContextClassLoader(contextClassLoader);
                }
            }
        }
    }

    private static String convertKey(String oldKey) throws IllegalArgumentException {
        if (!oldKey.startsWith(KEY_PREFIX)) {
            throw new IllegalArgumentException("[" + oldKey + "] doesn't start with prefix [" + KEY_PREFIX + "]");
        }
        String key = oldKey.substring(KEY_PREFIX.length());
        return key.substring(0, 4) + DASH + key.substring(4);
    }

    private static String getKeyName(DataIdentifier identifier) {
        String key = identifier.toString();
        return key.substring(0, 4) + DASH + key.substring(4);
    }

    private static String getIdentifierName(String key) {
        if (!key.contains(DASH)) {
            return null;
        }
        return key.substring(0, 4) + key.substring(5);
    }

    private class AsyncUploadJob
    implements Runnable {
        private DataIdentifier identifier;
        private File file;
        private AsyncUploadCallback callback;

        public AsyncUploadJob(DataIdentifier identifier, File file, AsyncUploadCallback callback) {
            this.identifier = identifier;
            this.file = file;
            this.callback = callback;
        }

        @Override
        public void run() {
            try {
                S3Backend.this.write(this.identifier, this.file, true, this.callback);
            }
            catch (DataStoreException e) {
                LOG.error("Could not upload [" + this.identifier + "], file[" + this.file + "]", (Throwable)e);
            }
        }
    }

    private class S3UploadProgressListener
    implements ProgressListener {
        private File file;
        private DataIdentifier identifier;
        private AsyncUploadCallback callback;
        private Upload upload;

        public S3UploadProgressListener(Upload upload, DataIdentifier identifier, File file, AsyncUploadCallback callback) {
            this.identifier = identifier;
            this.file = file;
            this.callback = callback;
            this.upload = upload;
        }

        public void progressChanged(ProgressEvent progressEvent) {
            switch (progressEvent.getEventCode()) {
                case 4: {
                    this.callback.onSuccess(new AsyncUploadResult(this.identifier, this.file));
                    break;
                }
                case 8: {
                    AsyncUploadResult result = new AsyncUploadResult(this.identifier, this.file);
                    try {
                        AmazonClientException e = this.upload.waitForException();
                        if (e != null) {
                            result.setException((Exception)e);
                        }
                    }
                    catch (InterruptedException e) {
                        Thread.currentThread().interrupt();
                    }
                    this.callback.onFailure(result);
                    break;
                }
            }
        }
    }

    private class KeyRenameThread
    implements Runnable {
        private String oldKey;

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void run() {
            ClassLoader contextClassLoader = Thread.currentThread().getContextClassLoader();
            try {
                Thread.currentThread().setContextClassLoader(this.getClass().getClassLoader());
                String newS3Key = S3Backend.convertKey(this.oldKey);
                CopyObjectRequest copReq = new CopyObjectRequest(S3Backend.this.bucket, this.oldKey, S3Backend.this.bucket, newS3Key);
                S3Backend.this.s3service.copyObject(copReq);
                LOG.debug("[{}] renamed to [{}] ", (Object)this.oldKey, (Object)newS3Key);
            }
            finally {
                if (contextClassLoader != null) {
                    Thread.currentThread().setContextClassLoader(contextClassLoader);
                }
            }
        }

        public KeyRenameThread(String oldKey) {
            this.oldKey = oldKey;
        }
    }
}

