/*
 * Decompiled with CFR 0.152.
 */
package org.apache.jena.tdb.base.file;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import org.apache.jena.atlas.io.IO;
import org.apache.jena.tdb.TDBException;
import org.apache.jena.tdb.base.file.FileException;
import org.apache.jena.tdb.base.file.Location;
import org.apache.jena.tdb.sys.ProcessUtils;
import org.apache.jena.tdb.sys.SystemTDB;

public class LocationLock {
    private static final int NO_OWNER = 0;
    private static final String LOCK_FILENAME = "tdb.lock";
    private Location location;

    public LocationLock(Location location) {
        if (location == null) {
            throw new NullPointerException("Location cannot be null");
        }
        this.location = location;
    }

    public boolean canLock() {
        return !this.location.isMem();
    }

    public boolean isLocked() {
        if (this.location.isMem()) {
            return false;
        }
        return this.getOwner() != 0;
    }

    public boolean isOwned() {
        if (this.location.isMem()) {
            return false;
        }
        int owner = this.getOwner();
        if (owner == 0) {
            return false;
        }
        return owner == ProcessUtils.getPid(0);
    }

    public int getOwner() {
        if (this.location.isMem()) {
            return 0;
        }
        File lockFile = this.getLockFile();
        if (lockFile.exists()) {
            this.checkLockFileForRead(lockFile);
            try {
                String rawLockInfo = IO.readWholeFileAsUTF8((String)lockFile.getAbsolutePath());
                int owner = Integer.parseInt(rawLockInfo);
                return owner;
            }
            catch (IOException e) {
                throw new FileException("Unable to check TDB lock owner due to an IO error reading the lock file", e);
            }
            catch (NumberFormatException e) {
                throw new FileException("Unable to check TDB lock owner as the lock file contains invalid data", e);
            }
        }
        return 0;
    }

    public boolean canObtain() {
        if (this.location.isMem()) {
            return false;
        }
        int owner = this.getOwner();
        int pid = ProcessUtils.getPid(0);
        if (owner == 0 ? pid != 0 : owner == pid) {
            return true;
        }
        return !ProcessUtils.isAlive(owner);
    }

    public void obtain() {
        if (this.location.isMem()) {
            return;
        }
        int owner = this.getOwner();
        if (owner == 0) {
            int pid = ProcessUtils.getPid(0);
            if (pid == 0) {
                SystemTDB.errlog.warn("Location " + this.location.getDirectoryPath() + " cannot be locked as unable to obtain PID of current process, if another JVM accessed this location while this process is accessing it then data corruption may occur");
                return;
            }
            this.takeLock(pid);
        } else if (owner != ProcessUtils.getPid(0)) {
            if (ProcessUtils.isAlive(owner)) {
                throw new TDBException("Location " + this.location.getDirectoryPath() + " is currently locked by PID " + owner + ".  TDB databases do not permit concurrent usage across JVMs so in order to prevent possible data corruption you cannot open this location from the JVM that does not own the lock for the dataset");
            }
            this.takeLock(ProcessUtils.getPid(0));
        }
    }

    private void takeLock(int pid) {
        File lockFile = this.getLockFile();
        this.checkLockFileForWrite(lockFile);
        try (BufferedWriter writer = new BufferedWriter(new FileWriter(lockFile));){
            writer.write(Integer.toString(pid));
            writer.close();
        }
        catch (IOException e) {
            throw new TDBException("Failed to obtain a lock on the location " + this.location.getDirectoryPath(), e);
        }
    }

    public void release() {
        if (this.location.isMem()) {
            return;
        }
        int owner = this.getOwner();
        if (owner == 0) {
            return;
        }
        if (owner != ProcessUtils.getPid(0)) {
            throw new TDBException("Cannot release the lock on location " + this.location.getDirectoryPath() + " since this process does not own the lock");
        }
        File lockFile = this.getLockFile();
        if (!lockFile.exists()) {
            return;
        }
        if (!lockFile.delete()) {
            throw new TDBException("Failed to release the lock on location " + this.location.getDirectoryPath() + ", it may be necessary to manually remove the lock file");
        }
    }

    private File getLockFile() {
        return new File(this.location.getPath(LOCK_FILENAME));
    }

    private void checkLockFileForRead(File lockFile) {
        if (!lockFile.exists()) {
            return;
        }
        if (!lockFile.isFile() || !lockFile.canRead()) {
            throw new FileException("Unable to check TDB lock owner for this location since the expected lock file is not a file/not readable");
        }
    }

    private void checkLockFileForWrite(File lockFile) {
        if (!lockFile.exists()) {
            return;
        }
        if (!lockFile.isFile() || !lockFile.canWrite()) {
            throw new FileException("Unable to check TDB lock owner for this location since the expected lock file is not a file/not writable");
        }
    }
}

