/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.controller;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Optional;
import java.util.OptionalLong;
import java.util.Random;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import org.apache.kafka.common.Endpoint;
import org.apache.kafka.common.Uuid;
import org.apache.kafka.common.errors.BrokerIdNotRegisteredException;
import org.apache.kafka.common.errors.DuplicateBrokerRegistrationException;
import org.apache.kafka.common.errors.InconsistentClusterIdException;
import org.apache.kafka.common.errors.StaleBrokerEpochException;
import org.apache.kafka.common.errors.UnsupportedVersionException;
import org.apache.kafka.common.message.BrokerRegistrationRequestData;
import org.apache.kafka.common.metadata.BrokerRegistrationChangeRecord;
import org.apache.kafka.common.metadata.FenceBrokerRecord;
import org.apache.kafka.common.metadata.RegisterBrokerRecord;
import org.apache.kafka.common.metadata.UnfenceBrokerRecord;
import org.apache.kafka.common.metadata.UnregisterBrokerRecord;
import org.apache.kafka.common.protocol.ApiMessage;
import org.apache.kafka.common.security.auth.SecurityProtocol;
import org.apache.kafka.common.utils.LogContext;
import org.apache.kafka.common.utils.Time;
import org.apache.kafka.controller.BrokerHeartbeatManager;
import org.apache.kafka.controller.ControllerResult;
import org.apache.kafka.controller.FeatureControlManager;
import org.apache.kafka.metadata.BrokerRegistration;
import org.apache.kafka.metadata.BrokerRegistrationFencingChange;
import org.apache.kafka.metadata.BrokerRegistrationInControlledShutdownChange;
import org.apache.kafka.metadata.BrokerRegistrationReply;
import org.apache.kafka.metadata.FinalizedControllerFeatures;
import org.apache.kafka.metadata.VersionRange;
import org.apache.kafka.metadata.placement.ReplicaPlacer;
import org.apache.kafka.metadata.placement.StripedReplicaPlacer;
import org.apache.kafka.metadata.placement.UsableBroker;
import org.apache.kafka.server.common.ApiMessageAndVersion;
import org.apache.kafka.server.common.MetadataVersion;
import org.apache.kafka.timeline.SnapshotRegistry;
import org.apache.kafka.timeline.TimelineHashMap;
import org.slf4j.Logger;

public class ClusterControlManager {
    static final long DEFAULT_SESSION_TIMEOUT_NS = TimeUnit.NANOSECONDS.convert(18L, TimeUnit.SECONDS);
    private final LogContext logContext;
    private final String clusterId;
    private final Logger log;
    private final Time time;
    private final long sessionTimeoutNs;
    private final ReplicaPlacer replicaPlacer;
    private final TimelineHashMap<Integer, BrokerRegistration> brokerRegistrations;
    private final TimelineHashMap<Integer, Long> registerBrokerRecordOffsets;
    private BrokerHeartbeatManager heartbeatManager;
    private Optional<ReadyBrokersFuture> readyBrokersFuture;
    private final FeatureControlManager featureControl;
    private final boolean zkMigrationEnabled;

    private ClusterControlManager(LogContext logContext, String clusterId, Time time, SnapshotRegistry snapshotRegistry, long sessionTimeoutNs, ReplicaPlacer replicaPlacer, FeatureControlManager featureControl, boolean zkMigrationEnabled) {
        this.logContext = logContext;
        this.clusterId = clusterId;
        this.log = logContext.logger(ClusterControlManager.class);
        this.time = time;
        this.sessionTimeoutNs = sessionTimeoutNs;
        this.replicaPlacer = replicaPlacer;
        this.brokerRegistrations = new TimelineHashMap(snapshotRegistry, 0);
        this.registerBrokerRecordOffsets = new TimelineHashMap(snapshotRegistry, 0);
        this.heartbeatManager = null;
        this.readyBrokersFuture = Optional.empty();
        this.featureControl = featureControl;
        this.zkMigrationEnabled = zkMigrationEnabled;
    }

    ReplicaPlacer replicaPlacer() {
        return this.replicaPlacer;
    }

    public void activate() {
        this.heartbeatManager = new BrokerHeartbeatManager(this.logContext, this.time, this.sessionTimeoutNs);
        for (BrokerRegistration registration : this.brokerRegistrations.values()) {
            this.heartbeatManager.touch(registration.id(), registration.fenced(), -1L);
        }
    }

    public void deactivate() {
        this.heartbeatManager = null;
    }

    Map<Integer, BrokerRegistration> brokerRegistrations() {
        return this.brokerRegistrations;
    }

    Map<Integer, Map<String, VersionRange>> brokerSupportedVersions() {
        return this.brokerRegistrations().entrySet().stream().collect(Collectors.toMap(Map.Entry::getKey, entry -> ((BrokerRegistration)entry.getValue()).supportedFeatures()));
    }

    Set<Integer> fencedBrokerIds() {
        return this.brokerRegistrations.values().stream().filter(BrokerRegistration::fenced).map(BrokerRegistration::id).collect(Collectors.toSet());
    }

    boolean zkRegistrationAllowed() {
        return this.zkMigrationEnabled && this.featureControl.metadataVersion().isMigrationSupported();
    }

    public ControllerResult<BrokerRegistrationReply> registerBroker(BrokerRegistrationRequestData request, long brokerEpoch, FinalizedControllerFeatures finalizedFeatures) {
        if (this.heartbeatManager == null) {
            throw new RuntimeException("ClusterControlManager is not active.");
        }
        if (!this.clusterId.equals(request.clusterId())) {
            throw new InconsistentClusterIdException("Expected cluster ID " + this.clusterId + ", but got cluster ID " + request.clusterId());
        }
        int brokerId = request.brokerId();
        BrokerRegistration existing = (BrokerRegistration)this.brokerRegistrations.get((Object)brokerId);
        if (existing != null) {
            if (this.heartbeatManager.hasValidSession(brokerId)) {
                if (!existing.incarnationId().equals((Object)request.incarnationId())) {
                    throw new DuplicateBrokerRegistrationException("Another broker is registered with that broker id.");
                }
            } else if (!existing.incarnationId().equals((Object)request.incarnationId())) {
                this.heartbeatManager.remove(brokerId);
            }
        }
        if (request.isMigratingZkBroker() && !this.zkRegistrationAllowed()) {
            throw new BrokerIdNotRegisteredException("Controller does not support registering ZK brokers.");
        }
        RegisterBrokerRecord record = new RegisterBrokerRecord().setBrokerId(brokerId).setIsMigratingZkBroker(request.isMigratingZkBroker()).setIncarnationId(request.incarnationId()).setBrokerEpoch(brokerEpoch).setRack(request.rack());
        for (BrokerRegistrationRequestData.Listener listener : request.listeners()) {
            record.endPoints().add(new RegisterBrokerRecord.BrokerEndpoint().setHost(listener.host()).setName(listener.name()).setPort(listener.port()).setSecurityProtocol(listener.securityProtocol()));
        }
        for (BrokerRegistrationRequestData.Feature feature : request.features()) {
            record.features().add(this.processRegistrationFeature(brokerId, finalizedFeatures, feature));
        }
        if (request.features().find("metadata.version") == null) {
            record.features().add(this.processRegistrationFeature(brokerId, finalizedFeatures, new BrokerRegistrationRequestData.Feature().setName("metadata.version").setMinSupportedVersion(MetadataVersion.MINIMUM_KRAFT_VERSION.featureLevel()).setMaxSupportedVersion(MetadataVersion.MINIMUM_KRAFT_VERSION.featureLevel())));
        }
        this.heartbeatManager.register(brokerId, record.fenced());
        ArrayList<ApiMessageAndVersion> records = new ArrayList<ApiMessageAndVersion>();
        records.add(new ApiMessageAndVersion((ApiMessage)record, this.featureControl.metadataVersion().registerBrokerRecordVersion()));
        return ControllerResult.atomicOf(records, new BrokerRegistrationReply(brokerEpoch));
    }

    RegisterBrokerRecord.BrokerFeature processRegistrationFeature(int brokerId, FinalizedControllerFeatures finalizedFeatures, BrokerRegistrationRequestData.Feature feature) {
        Optional<Short> finalized = finalizedFeatures.get(feature.name());
        if (finalized.isPresent()) {
            if (!VersionRange.of(feature.minSupportedVersion(), feature.maxSupportedVersion()).contains(finalized.get())) {
                throw new UnsupportedVersionException("Unable to register because the broker does not support version " + finalized.get() + " of " + feature.name() + ". It wants a version between " + feature.minSupportedVersion() + " and " + feature.maxSupportedVersion() + ", inclusive.");
            }
        } else {
            this.log.warn("Broker {} registered with feature {} that is unknown to the controller", (Object)brokerId, (Object)feature.name());
        }
        return new RegisterBrokerRecord.BrokerFeature().setName(feature.name()).setMinSupportedVersion(feature.minSupportedVersion()).setMaxSupportedVersion(feature.maxSupportedVersion());
    }

    public OptionalLong registerBrokerRecordOffset(int brokerId) {
        Long registrationOffset = (Long)this.registerBrokerRecordOffsets.get((Object)brokerId);
        if (registrationOffset != null) {
            return OptionalLong.of(registrationOffset);
        }
        return OptionalLong.empty();
    }

    public void replay(RegisterBrokerRecord record, long offset) {
        Object endpoint;
        this.registerBrokerRecordOffsets.put((Object)record.brokerId(), (Object)offset);
        int brokerId = record.brokerId();
        ArrayList<Endpoint> listeners = new ArrayList<Endpoint>();
        Iterator iterator = record.endPoints().iterator();
        while (iterator.hasNext()) {
            endpoint = (RegisterBrokerRecord.BrokerEndpoint)iterator.next();
            listeners.add(new Endpoint(((RegisterBrokerRecord.BrokerEndpoint)endpoint).name(), SecurityProtocol.forId((short)((RegisterBrokerRecord.BrokerEndpoint)endpoint).securityProtocol()), ((RegisterBrokerRecord.BrokerEndpoint)endpoint).host(), ((RegisterBrokerRecord.BrokerEndpoint)endpoint).port()));
        }
        HashMap<String, VersionRange> features = new HashMap<String, VersionRange>();
        endpoint = record.features().iterator();
        while (endpoint.hasNext()) {
            RegisterBrokerRecord.BrokerFeature feature = (RegisterBrokerRecord.BrokerFeature)endpoint.next();
            features.put(feature.name(), VersionRange.of(feature.minSupportedVersion(), feature.maxSupportedVersion()));
        }
        BrokerRegistration prevRegistration = (BrokerRegistration)this.brokerRegistrations.put((Object)brokerId, (Object)new BrokerRegistration(brokerId, record.brokerEpoch(), record.incarnationId(), listeners, features, Optional.ofNullable(record.rack()), record.fenced(), record.inControlledShutdown(), record.isMigratingZkBroker()));
        if (this.heartbeatManager != null) {
            if (prevRegistration != null) {
                this.heartbeatManager.remove(brokerId);
            }
            this.heartbeatManager.register(brokerId, record.fenced());
        }
        if (prevRegistration == null) {
            this.log.info("Registered new broker: {}", (Object)record);
        } else if (prevRegistration.incarnationId().equals((Object)record.incarnationId())) {
            this.log.info("Re-registered broker incarnation: {}", (Object)record);
        } else {
            this.log.info("Re-registered broker id {}: {}", (Object)brokerId, (Object)record);
        }
    }

    public void replay(UnregisterBrokerRecord record) {
        this.registerBrokerRecordOffsets.remove((Object)record.brokerId());
        int brokerId = record.brokerId();
        BrokerRegistration registration = (BrokerRegistration)this.brokerRegistrations.get((Object)brokerId);
        if (registration == null) {
            throw new RuntimeException(String.format("Unable to replay %s: no broker registration found for that id", record));
        }
        if (registration.epoch() != record.brokerEpoch()) {
            throw new RuntimeException(String.format("Unable to replay %s: no broker registration with that epoch found", record));
        }
        if (this.heartbeatManager != null) {
            this.heartbeatManager.remove(brokerId);
        }
        this.brokerRegistrations.remove((Object)brokerId);
        this.log.info("Unregistered broker: {}", (Object)record);
    }

    public void replay(FenceBrokerRecord record) {
        this.replayRegistrationChange(record, record.id(), record.epoch(), BrokerRegistrationFencingChange.FENCE.asBoolean(), BrokerRegistrationInControlledShutdownChange.NONE.asBoolean());
    }

    public void replay(UnfenceBrokerRecord record) {
        this.replayRegistrationChange(record, record.id(), record.epoch(), BrokerRegistrationFencingChange.UNFENCE.asBoolean(), BrokerRegistrationInControlledShutdownChange.NONE.asBoolean());
    }

    public void replay(BrokerRegistrationChangeRecord record) {
        BrokerRegistrationFencingChange fencingChange = BrokerRegistrationFencingChange.fromValue(record.fenced()).orElseThrow(() -> new IllegalStateException(String.format("Unable to replay %s: unknown value for fenced field: %x", record, record.fenced())));
        BrokerRegistrationInControlledShutdownChange inControlledShutdownChange = BrokerRegistrationInControlledShutdownChange.fromValue(record.inControlledShutdown()).orElseThrow(() -> new IllegalStateException(String.format("Unable to replay %s: unknown value for inControlledShutdown field: %x", record, record.inControlledShutdown())));
        this.replayRegistrationChange(record, record.brokerId(), record.brokerEpoch(), fencingChange.asBoolean(), inControlledShutdownChange.asBoolean());
    }

    private void replayRegistrationChange(ApiMessage record, int brokerId, long brokerEpoch, Optional<Boolean> fencingChange, Optional<Boolean> inControlledShutdownChange) {
        BrokerRegistration curRegistration = (BrokerRegistration)this.brokerRegistrations.get((Object)brokerId);
        if (curRegistration == null) {
            throw new RuntimeException(String.format("Unable to replay %s: no broker registration found for that id", record.toString()));
        }
        if (curRegistration.epoch() != brokerEpoch) {
            throw new RuntimeException(String.format("Unable to replay %s: no broker registration with that epoch found", record.toString()));
        }
        BrokerRegistration nextRegistration = curRegistration.cloneWith(fencingChange, inControlledShutdownChange);
        if (!curRegistration.equals(nextRegistration)) {
            this.brokerRegistrations.put((Object)brokerId, (Object)nextRegistration);
        } else {
            this.log.info("Ignoring no-op registration change for {}", (Object)curRegistration);
        }
        if (this.heartbeatManager != null) {
            this.heartbeatManager.register(brokerId, nextRegistration.fenced());
        }
        if (this.readyBrokersFuture.isPresent() && this.readyBrokersFuture.get().check()) {
            this.readyBrokersFuture.get().future.complete(null);
            this.readyBrokersFuture = Optional.empty();
        }
    }

    Iterator<UsableBroker> usableBrokers() {
        if (this.heartbeatManager == null) {
            throw new RuntimeException("ClusterControlManager is not active.");
        }
        return this.heartbeatManager.usableBrokers(id -> ((BrokerRegistration)this.brokerRegistrations.get(id)).rack());
    }

    public boolean isUnfenced(int brokerId) {
        BrokerRegistration registration = (BrokerRegistration)this.brokerRegistrations.get((Object)brokerId);
        if (registration == null) {
            return false;
        }
        return !registration.fenced();
    }

    public BrokerRegistration registration(int brokerId) {
        return (BrokerRegistration)this.brokerRegistrations.get((Object)brokerId);
    }

    public boolean inControlledShutdown(int brokerId) {
        BrokerRegistration registration = (BrokerRegistration)this.brokerRegistrations.get((Object)brokerId);
        if (registration == null) {
            return false;
        }
        return registration.inControlledShutdown();
    }

    public boolean isActive(int brokerId) {
        BrokerRegistration registration = (BrokerRegistration)this.brokerRegistrations.get((Object)brokerId);
        if (registration == null) {
            return false;
        }
        return !registration.inControlledShutdown() && !registration.fenced();
    }

    BrokerHeartbeatManager heartbeatManager() {
        if (this.heartbeatManager == null) {
            throw new RuntimeException("ClusterControlManager is not active.");
        }
        return this.heartbeatManager;
    }

    public void checkBrokerEpoch(int brokerId, long brokerEpoch) {
        BrokerRegistration registration = (BrokerRegistration)this.brokerRegistrations.get((Object)brokerId);
        if (registration == null) {
            throw new StaleBrokerEpochException("No broker registration found for broker id " + brokerId);
        }
        if (registration.epoch() != brokerEpoch) {
            throw new StaleBrokerEpochException("Expected broker epoch " + registration.epoch() + ", but got broker epoch " + brokerEpoch);
        }
    }

    public void addReadyBrokersFuture(CompletableFuture<Void> future, int minBrokers) {
        this.readyBrokersFuture = Optional.of(new ReadyBrokersFuture(future, minBrokers));
        if (this.readyBrokersFuture.get().check()) {
            this.readyBrokersFuture.get().future.complete(null);
            this.readyBrokersFuture = Optional.empty();
        }
    }

    class ReadyBrokersFuture {
        private final CompletableFuture<Void> future;
        private final int minBrokers;

        ReadyBrokersFuture(CompletableFuture<Void> future, int minBrokers) {
            this.future = future;
            this.minBrokers = minBrokers;
        }

        boolean check() {
            int numUnfenced = 0;
            for (BrokerRegistration registration : ClusterControlManager.this.brokerRegistrations.values()) {
                if (!registration.fenced()) {
                    ++numUnfenced;
                }
                if (numUnfenced < this.minBrokers) continue;
                return true;
            }
            return false;
        }
    }

    static class Builder {
        private LogContext logContext = null;
        private String clusterId = null;
        private Time time = Time.SYSTEM;
        private SnapshotRegistry snapshotRegistry = null;
        private long sessionTimeoutNs = DEFAULT_SESSION_TIMEOUT_NS;
        private ReplicaPlacer replicaPlacer = null;
        private FeatureControlManager featureControl = null;
        private boolean zkMigrationEnabled = false;

        Builder() {
        }

        Builder setLogContext(LogContext logContext) {
            this.logContext = logContext;
            return this;
        }

        Builder setClusterId(String clusterId) {
            this.clusterId = clusterId;
            return this;
        }

        Builder setTime(Time time) {
            this.time = time;
            return this;
        }

        Builder setSnapshotRegistry(SnapshotRegistry snapshotRegistry) {
            this.snapshotRegistry = snapshotRegistry;
            return this;
        }

        Builder setSessionTimeoutNs(long sessionTimeoutNs) {
            this.sessionTimeoutNs = sessionTimeoutNs;
            return this;
        }

        Builder setReplicaPlacer(ReplicaPlacer replicaPlacer) {
            this.replicaPlacer = replicaPlacer;
            return this;
        }

        Builder setFeatureControlManager(FeatureControlManager featureControl) {
            this.featureControl = featureControl;
            return this;
        }

        Builder setZkMigrationEnabled(boolean zkMigrationEnabled) {
            this.zkMigrationEnabled = zkMigrationEnabled;
            return this;
        }

        ClusterControlManager build() {
            if (this.logContext == null) {
                this.logContext = new LogContext();
            }
            if (this.clusterId == null) {
                this.clusterId = Uuid.randomUuid().toString();
            }
            if (this.snapshotRegistry == null) {
                this.snapshotRegistry = new SnapshotRegistry(this.logContext);
            }
            if (this.replicaPlacer == null) {
                this.replicaPlacer = new StripedReplicaPlacer(new Random());
            }
            if (this.featureControl == null) {
                throw new RuntimeException("You must specify FeatureControlManager");
            }
            return new ClusterControlManager(this.logContext, this.clusterId, this.time, this.snapshotRegistry, this.sessionTimeoutNs, this.replicaPlacer, this.featureControl, this.zkMigrationEnabled);
        }
    }
}

