/*
 * Decompiled with CFR 0.152.
 */
package org.apache.shindig.gadgets.http;

import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.inject.Inject;
import com.google.inject.name.Named;
import java.io.ByteArrayInputStream;
import java.io.Externalizable;
import java.io.IOException;
import java.io.InputStream;
import java.io.ObjectInput;
import java.io.ObjectOutput;
import java.nio.ByteBuffer;
import java.nio.charset.Charset;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import org.apache.shindig.common.util.DateUtil;
import org.apache.shindig.gadgets.encoding.EncodingDetector;
import org.apache.shindig.gadgets.http.HttpResponseBuilder;

/*
 * This class specifies class file version 49.0 but uses Java 6 signatures.  Assumed Java 6.
 */
public final class HttpResponse
implements Externalizable {
    private static final long serialVersionUID = 7526471155622776147L;
    public static final int SC_CONTINUE = 100;
    public static final int SC_SWITCHING_PROTOCOLS = 101;
    public static final int SC_OK = 200;
    public static final int SC_CREATED = 201;
    public static final int SC_ACCEPTED = 202;
    public static final int SC_NON_AUTHORITATIVE_INFORMATION = 203;
    public static final int SC_NO_CONTENT = 204;
    public static final int SC_RESET_CONTENT = 205;
    public static final int SC_PARTIAL_CONTENT = 206;
    public static final int SC_MULTIPLE_CHOICES = 300;
    public static final int SC_MOVED_PERMANENTLY = 301;
    public static final int SC_FOUND = 302;
    public static final int SC_SEE_OTHER = 303;
    public static final int SC_NOT_MODIFIED = 304;
    public static final int SC_USE_PROXY = 305;
    public static final int SC_TEMPORARY_REDIRECT = 307;
    public static final int SC_BAD_REQUEST = 400;
    public static final int SC_UNAUTHORIZED = 401;
    public static final int SC_PAYMENT_REQUIRED = 402;
    public static final int SC_FORBIDDEN = 403;
    public static final int SC_NOT_FOUND = 404;
    public static final int SC_METHOD_NOT_ALLOWED = 405;
    public static final int SC_NOT_ACCEPTABLE = 406;
    public static final int SC_PROXY_AUTHENTICATION_REQUIRED = 407;
    public static final int SC_REQUEST_TIMEOUT = 408;
    public static final int SC_CONFLICT = 409;
    public static final int SC_GONE = 410;
    public static final int SC_LENGTH_REQUIRED = 411;
    public static final int SC_PRECONDITION_FAILED = 412;
    public static final int SC_REQUEST_ENTITY_TOO_LARGE = 413;
    public static final int SC_REQUEST_URI_TOO_LONG = 414;
    public static final int SC_UNSUPPORTED_MEDIA_TYPE = 415;
    public static final int SC_REQUESTED_RANGE_NOT_SATISFIABLE = 416;
    public static final int SC_EXPECTATION_FAILED = 417;
    public static final int SC_INTERNAL_SERVER_ERROR = 500;
    public static final int SC_NOT_IMPLEMENTED = 501;
    public static final int SC_BAD_GATEWAY = 502;
    public static final int SC_SERVICE_UNAVAILABLE = 503;
    public static final int SC_GATEWAY_TIMEOUT = 504;
    public static final int SC_HTTP_VERSION_NOT_SUPPORTED = 505;
    private static final Set<String> BINARY_CONTENT_TYPES = ImmutableSet.of((Object[])new String[]{"image/jpeg", "image/png", "image/gif", "image/jpg", "application/x-shockwave-flash", "application/octet-stream", "application/ogg", "application/zip", "audio/mpeg", "audio/x-ms-wma", "audio/vnd.rn-realaudio", "audio/x-wav", "video/mpeg", "video/mp4", "video/quicktime", "video/x-ms-wmv", "video/x-flv", "video/flv", "video/x-ms-asf", "application/pdf"});
    private static final Set<Integer> NEGATIVE_CACHING_EXEMPT_STATUS = ImmutableSet.of((Object[])new Integer[]{401, 403});
    static final long DEFAULT_NEGATIVE_CACHE_TTL = 30000L;
    static final long DEFAULT_TTL = 300000L;
    static final String DEFAULT_ENCODING = "UTF-8";
    @Inject(optional=true)
    @Named(value="shindig.cache.http.negativeCacheTtl")
    private static long negativeCacheTtl = 30000L;
    @Inject(optional=true)
    @Named(value="shindig.cache.http.defaultTtl")
    private static long defaultTtl = 300000L;
    @Inject(optional=true)
    @Named(value="shindig.http.fast-encoding-detection")
    private static boolean fastEncodingDetection = true;
    private static final Map<String, Charset> encodingToCharset = Maps.newConcurrentHashMap();
    private transient String responseString;
    private transient long date;
    private transient String encoding;
    private transient Map<String, String> metadata;
    private int httpStatusCode;
    private Map<String, List<String>> headers;
    private byte[] responseBytes;

    public HttpResponse() {
    }

    HttpResponse(HttpResponseBuilder builder) {
        this.httpStatusCode = builder.getHttpStatusCode();
        TreeMap headerCopy = Maps.newTreeMap((Comparator)String.CASE_INSENSITIVE_ORDER);
        headerCopy.putAll(builder.getHeaders());
        this.responseBytes = builder.getResponse();
        HashMap metadataCopy = Maps.newHashMap(builder.getMetadata());
        this.metadata = Collections.unmodifiableMap(metadataCopy);
        this.date = HttpResponse.getAndUpdateDate(headerCopy);
        this.encoding = HttpResponse.getAndUpdateEncoding(headerCopy, this.responseBytes);
        this.headers = Collections.unmodifiableMap(headerCopy);
    }

    private HttpResponse(int httpStatusCode, String body) {
        this(new HttpResponseBuilder().setHttpStatusCode(httpStatusCode).setResponseString(body));
    }

    public HttpResponse(String body) {
        this(200, body);
    }

    public static HttpResponse error() {
        return new HttpResponse(500, "");
    }

    public static HttpResponse timeout() {
        return new HttpResponse(504, "");
    }

    public static HttpResponse notFound() {
        return new HttpResponse(404, "");
    }

    public int getHttpStatusCode() {
        return this.httpStatusCode;
    }

    public boolean isError() {
        return this.httpStatusCode >= 400;
    }

    public String getEncoding() {
        return this.encoding;
    }

    public int getContentLength() {
        return this.responseBytes.length;
    }

    public InputStream getResponse() {
        return new ByteArrayInputStream(this.responseBytes);
    }

    public String getResponseAsString() {
        if (this.responseString == null) {
            Charset charset = encodingToCharset.get(this.encoding);
            if (charset == null) {
                charset = Charset.forName(this.encoding);
                encodingToCharset.put(this.encoding, charset);
            }
            this.responseString = charset.decode(ByteBuffer.wrap(this.responseBytes)).toString();
            if (this.responseString.length() > 0 && this.responseString.codePointAt(0) == 65279) {
                this.responseString = this.responseString.substring(1);
            }
        }
        return this.responseString;
    }

    public Map<String, List<String>> getHeaders() {
        return this.headers;
    }

    public List<String> getHeaders(String name) {
        List<String> ret = this.headers.get(name);
        if (ret == null) {
            return Collections.emptyList();
        }
        return ret;
    }

    public String getHeader(String name) {
        List<String> headerList = this.getHeaders(name);
        if (headerList.isEmpty()) {
            return null;
        }
        return headerList.get(0);
    }

    public Map<String, String> getMetadata() {
        return this.metadata;
    }

    public long getCacheExpiration() {
        if (this.isError() && !NEGATIVE_CACHING_EXEMPT_STATUS.contains(this.httpStatusCode)) {
            return this.date + negativeCacheTtl;
        }
        if (this.isStrictNoCache()) {
            return -1L;
        }
        long maxAge = this.getCacheControlMaxAge();
        if (maxAge != -1L) {
            return this.date + maxAge;
        }
        long expiration = this.getExpiresTime();
        if (expiration != -1L) {
            return expiration;
        }
        return this.date + defaultTtl;
    }

    public long getCacheTtl() {
        long expiration = this.getCacheExpiration();
        if (expiration != -1L) {
            return expiration - System.currentTimeMillis();
        }
        return -1L;
    }

    public boolean isStrictNoCache() {
        String cacheControl = this.getHeader("Cache-Control");
        if (cacheControl != null) {
            String[] directives;
            for (String directive : directives = cacheControl.split(",")) {
                if (!(directive = directive.trim()).equalsIgnoreCase("no-cache") && !directive.equalsIgnoreCase("no-store") && !directive.equalsIgnoreCase("private")) continue;
                return true;
            }
        }
        for (String pragma : this.getHeaders("Pragma")) {
            if (!"no-cache".equalsIgnoreCase(pragma)) continue;
            return true;
        }
        return false;
    }

    private long getExpiresTime() {
        Date expiresDate;
        String expires = this.getHeader("Expires");
        if (expires != null && (expiresDate = DateUtil.parseDate((String)expires)) != null) {
            return expiresDate.getTime();
        }
        return -1L;
    }

    private long getCacheControlMaxAge() {
        String cacheControl = this.getHeader("Cache-Control");
        if (cacheControl != null) {
            String[] directives;
            for (String directive : directives = cacheControl.split(",")) {
                String[] parts;
                if (!(directive = directive.trim()).startsWith("max-age") || (parts = directive.split("=")).length != 2) continue;
                try {
                    return Long.parseLong(parts[1]) * 1000L;
                }
                catch (NumberFormatException ignore) {
                    return -1L;
                }
            }
        }
        return -1L;
    }

    private static long getAndUpdateDate(Map<String, List<String>> headers) {
        Date d;
        String dateStr;
        long timestamp = -1L;
        List<String> dates = headers.get("Date");
        String string = dates == null ? null : (dateStr = dates.isEmpty() ? null : dates.get(0));
        if (dateStr != null && (d = DateUtil.parseDate(dateStr)) != null) {
            timestamp = d.getTime();
        }
        if (timestamp == -1L) {
            timestamp = System.currentTimeMillis();
            headers.put("Date", Lists.newArrayList((Object[])new String[]{DateUtil.formatDate((long)timestamp)}));
        }
        return timestamp;
    }

    private static String getAndUpdateEncoding(Map<String, List<String>> headers, byte[] body) {
        String contentType;
        if (body == null || body.length == 0) {
            return DEFAULT_ENCODING;
        }
        List<String> values = headers.get("Content-Type");
        String string = values == null ? null : (contentType = values.isEmpty() ? null : values.get(0));
        if (contentType != null) {
            int offset;
            String[] parts = contentType.split(";");
            if (BINARY_CONTENT_TYPES.contains(parts[0])) {
                return DEFAULT_ENCODING;
            }
            if (parts.length == 2 && (offset = parts[1].indexOf("charset=")) != -1) {
                String charset = parts[1].substring(offset + 8).toUpperCase();
                if (charset.charAt(0) == '\"') {
                    charset = charset.substring(1, charset.length() - 1);
                }
                return charset;
            }
            String encoding = EncodingDetector.detectEncoding(body, fastEncodingDetection);
            headers.put("Content-Type", Lists.newArrayList((Object[])new String[]{contentType + "; charset=" + encoding}));
            return encoding;
        }
        contentType = "application/octet-stream";
        return DEFAULT_ENCODING;
    }

    public boolean equals(Object obj) {
        if (obj == this) {
            return true;
        }
        if (!(obj instanceof HttpResponse)) {
            return false;
        }
        HttpResponse response = (HttpResponse)obj;
        return this.httpStatusCode == response.httpStatusCode && ((Object)this.headers).equals(response.headers) && Arrays.equals(this.responseBytes, response.responseBytes);
    }

    public String toString() {
        StringBuilder buf = new StringBuilder("HTTP/1.1 ").append(this.httpStatusCode).append("\r\n\r\n");
        for (Map.Entry<String, List<String>> entry : this.headers.entrySet()) {
            String name = entry.getKey();
            for (String value : entry.getValue()) {
                buf.append(name).append(": ").append(value).append('\n');
            }
        }
        buf.append("\r\n").append(this.getResponseAsString()).append("\r\n");
        return buf.toString();
    }

    byte[] getResponseAsBytes() {
        return this.responseBytes;
    }

    @Override
    public void readExternal(ObjectInput in) throws IOException, ClassNotFoundException {
        this.httpStatusCode = in.readInt();
        Map headerCopy = (Map)in.readObject();
        int bodyLength = in.readInt();
        this.responseBytes = new byte[bodyLength];
        in.read(this.responseBytes, 0, bodyLength);
        this.date = HttpResponse.getAndUpdateDate(headerCopy);
        this.encoding = HttpResponse.getAndUpdateEncoding(headerCopy, this.responseBytes);
        this.headers = Collections.unmodifiableMap(headerCopy);
        this.metadata = Collections.emptyMap();
    }

    @Override
    public void writeExternal(ObjectOutput out) throws IOException {
        out.writeInt(this.httpStatusCode);
        out.writeObject(this.headers);
        out.writeInt(this.responseBytes.length);
        out.write(this.responseBytes);
    }
}

