/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements. See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership. The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations under the License.
 */
package org.apache.shindig.social.core.util.xstream;

import com.google.inject.Injector;

import com.thoughtworks.xstream.converters.reflection.ObjectAccessException;
import com.thoughtworks.xstream.core.util.OrderRetainingMap;

import java.beans.BeanInfo;
import java.beans.IntrospectionException;
import java.beans.Introspector;
import java.beans.PropertyDescriptor;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;
import java.util.WeakHashMap;

/**
 * 
 */
public class GuiceBeanProvider {

  protected static final Object[] NO_PARAMS = new Object[0];
  private final Comparator<String> propertyNameComparator;
  private final transient Map<Class<?>, Map<String, PropertyDescriptor>> propertyNameCache = new WeakHashMap<Class<?>, Map<String, PropertyDescriptor>>();
  private Injector injector;

  public GuiceBeanProvider(Injector injector) {
    this(injector, null);
  }

  public GuiceBeanProvider(Injector injector,
      final Comparator<String> propertyNameComparator) {
    this.propertyNameComparator = propertyNameComparator;
    this.injector = injector;
  }

  public Object newInstance(Class<?> type) {
    return injector.getInstance(type);
  }

  public void visitSerializableProperties(Object object, Visitor visitor) {
    PropertyDescriptor[] propertyDescriptors = getSerializableProperties(object);
    for (int i = 0; i < propertyDescriptors.length; i++) {
      PropertyDescriptor property = propertyDescriptors[i];
      try {
        Method readMethod = property.getReadMethod();
        String name = property.getName();
        Class<?> definedIn = readMethod.getDeclaringClass();
        if (visitor.shouldVisit(name, definedIn)) {
          Object value = readMethod.invoke(object, new Object[0]);
          visitor.visit(name, property.getPropertyType(), definedIn, value);
        }
      } catch (IllegalArgumentException e) {
        throw new ObjectAccessException("Could not get property "
            + object.getClass() + "." + property.getName(), e);
      } catch (IllegalAccessException e) {
        throw new ObjectAccessException("Could not get property "
            + object.getClass() + "." + property.getName(), e);
      } catch (InvocationTargetException e) {
        throw new ObjectAccessException("Could not get property "
            + object.getClass() + "." + property.getName(), e);
      }
    }
  }

  public void writeProperty(Object object, String propertyName, Object value) {
    PropertyDescriptor property = getProperty(propertyName, object.getClass());
    try {
      property.getWriteMethod().invoke(object, new Object[] { value });
    } catch (IllegalArgumentException e) {
      throw new ObjectAccessException("Could not set property "
          + object.getClass() + "." + property.getName(), e);
    } catch (IllegalAccessException e) {
      throw new ObjectAccessException("Could not set property "
          + object.getClass() + "." + property.getName(), e);
    } catch (InvocationTargetException e) {
      throw new ObjectAccessException("Could not set property "
          + object.getClass() + "." + property.getName(), e);
    }
  }

  public Class<?> getPropertyType(Object object, String name) {
    return getProperty(name, object.getClass()).getPropertyType();
  }

  public boolean propertyDefinedInClass(String name, Class<?> type) {
    return getProperty(name, type) != null;
  }

  private PropertyDescriptor[] getSerializableProperties(Object object) {
    Map<String, PropertyDescriptor> nameMap = getNameMap(object.getClass());
    List<PropertyDescriptor> result = new ArrayList<PropertyDescriptor>(nameMap
        .size());
    Set<String> names = nameMap.keySet();
    if (propertyNameComparator != null) {
      Set<String> sortedSet = new TreeSet<String>(propertyNameComparator);
      sortedSet.addAll(names);
      names = sortedSet;
    }
    for (final String name : names) {
      final PropertyDescriptor descriptor = nameMap.get(name);
      if (canStreamProperty(descriptor)) {
        result.add(descriptor);
      }
    }
    return result.toArray(new PropertyDescriptor[result.size()]);
  }

  protected boolean canStreamProperty(PropertyDescriptor descriptor) {
    return descriptor.getReadMethod() != null
        && descriptor.getWriteMethod() != null;
  }

  public boolean propertyWriteable(String name, Class<?> type) {
    PropertyDescriptor property = getProperty(name, type);
    return property.getWriteMethod() != null;
  }

  private PropertyDescriptor getProperty(String name, Class<?> type) {
    return (PropertyDescriptor) getNameMap(type).get(name);
  }

  private Map<String, PropertyDescriptor> getNameMap(Class<?> type) {
    Map<String, PropertyDescriptor> nameMap = propertyNameCache.get(type);
    if (nameMap == null) {
      BeanInfo beanInfo;
      try {
        beanInfo = Introspector.getBeanInfo(type, Object.class);
      } catch (IntrospectionException e) {
        throw new ObjectAccessException("Cannot get BeanInfo of type "
            + type.getName(), e);
      }
      nameMap = new OrderRetainingMap();
      propertyNameCache.put(type, nameMap);
      PropertyDescriptor[] propertyDescriptors = beanInfo
          .getPropertyDescriptors();
      for (int i = 0; i < propertyDescriptors.length; i++) {
        PropertyDescriptor descriptor = propertyDescriptors[i];
        nameMap.put(descriptor.getName(), descriptor);
      }
    }
    return nameMap;
  }

  interface Visitor {
    boolean shouldVisit(String name, Class<?> definedIn);

    void visit(String name, Class<?> type, Class<?> definedIn, Object value);
  }

}
