/*
 * Decompiled with CFR 0.152.
 */
package org.apache.storm.windowing;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.locks.ReentrantLock;
import org.apache.storm.windowing.Event;
import org.apache.storm.windowing.EventImpl;
import org.apache.storm.windowing.EvictionPolicy;
import org.apache.storm.windowing.TriggerHandler;
import org.apache.storm.windowing.TriggerPolicy;
import org.apache.storm.windowing.WindowLifecycleListener;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class WindowManager<T>
implements TriggerHandler {
    private static final Logger LOG = LoggerFactory.getLogger(WindowManager.class);
    public static final int EXPIRE_EVENTS_THRESHOLD = 100;
    private final WindowLifecycleListener<T> windowLifecycleListener;
    private final ConcurrentLinkedQueue<Event<T>> queue;
    private final List<T> expiredEvents;
    private final Set<Event<T>> prevWindowEvents;
    private final AtomicInteger eventsSinceLastExpiry;
    private final ReentrantLock lock;
    private EvictionPolicy<T> evictionPolicy;
    private TriggerPolicy<T> triggerPolicy;

    public WindowManager(WindowLifecycleListener<T> lifecycleListener) {
        this.windowLifecycleListener = lifecycleListener;
        this.queue = new ConcurrentLinkedQueue();
        this.expiredEvents = new ArrayList<T>();
        this.prevWindowEvents = new HashSet<Event<T>>();
        this.eventsSinceLastExpiry = new AtomicInteger();
        this.lock = new ReentrantLock(true);
    }

    public void setEvictionPolicy(EvictionPolicy<T> evictionPolicy) {
        this.evictionPolicy = evictionPolicy;
    }

    public void setTriggerPolicy(TriggerPolicy<T> triggerPolicy) {
        this.triggerPolicy = triggerPolicy;
    }

    public void add(T event) {
        this.add(event, System.currentTimeMillis());
    }

    public void add(T event, long ts) {
        this.add(new EventImpl<T>(event, ts));
    }

    public void add(Event<T> windowEvent) {
        if (!windowEvent.isWatermark()) {
            this.queue.add(windowEvent);
        } else {
            LOG.debug("Got watermark event with ts {}", (Object)windowEvent.getTimestamp());
        }
        this.track(windowEvent);
        this.compactWindow();
    }

    @Override
    public boolean onTrigger() {
        List<Event<T>> windowEvents = null;
        ArrayList<T> expired = null;
        try {
            this.lock.lock();
            windowEvents = this.scanEvents(true);
            expired = new ArrayList<T>(this.expiredEvents);
            this.expiredEvents.clear();
        }
        finally {
            this.lock.unlock();
        }
        ArrayList<T> events = new ArrayList<T>();
        ArrayList<T> newEvents = new ArrayList<T>();
        for (Event<T> event : windowEvents) {
            events.add(event.get());
            if (this.prevWindowEvents.contains(event)) continue;
            newEvents.add(event.get());
        }
        this.prevWindowEvents.clear();
        if (!events.isEmpty()) {
            this.prevWindowEvents.addAll(windowEvents);
            LOG.debug("invoking windowLifecycleListener onActivation, [{}] events in window.", (Object)events.size());
            this.windowLifecycleListener.onActivation(events, newEvents, expired, this.evictionPolicy.getContext().getReferenceTime());
        } else {
            LOG.debug("No events in the window, skipping onActivation");
        }
        this.triggerPolicy.reset();
        return !events.isEmpty();
    }

    public void shutdown() {
        LOG.debug("Shutting down WindowManager");
        if (this.triggerPolicy != null) {
            this.triggerPolicy.shutdown();
        }
    }

    private void compactWindow() {
        if (this.eventsSinceLastExpiry.incrementAndGet() >= 100) {
            this.scanEvents(false);
        }
    }

    private void track(Event<T> windowEvent) {
        this.evictionPolicy.track(windowEvent);
        this.triggerPolicy.track(windowEvent);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private List<Event<T>> scanEvents(boolean fullScan) {
        LOG.debug("Scan events, eviction policy {}", this.evictionPolicy);
        ArrayList<T> eventsToExpire = new ArrayList<T>();
        ArrayList<Event<T>> eventsToProcess = new ArrayList<Event<T>>();
        try {
            this.lock.lock();
            Iterator<Event<T>> it = this.queue.iterator();
            while (it.hasNext()) {
                Event<T> windowEvent = it.next();
                EvictionPolicy.Action action = this.evictionPolicy.evict(windowEvent);
                if (action == EvictionPolicy.Action.EXPIRE) {
                    eventsToExpire.add(windowEvent.get());
                    it.remove();
                    continue;
                }
                if (!fullScan || action == EvictionPolicy.Action.STOP) break;
                if (action != EvictionPolicy.Action.PROCESS) continue;
                eventsToProcess.add(windowEvent);
            }
            this.expiredEvents.addAll(eventsToExpire);
        }
        finally {
            this.lock.unlock();
        }
        this.eventsSinceLastExpiry.set(0);
        LOG.debug("[{}] events expired from window.", (Object)eventsToExpire.size());
        if (!eventsToExpire.isEmpty()) {
            LOG.debug("invoking windowLifecycleListener.onExpiry");
            this.windowLifecycleListener.onExpiry(eventsToExpire);
        }
        return eventsToProcess;
    }

    public long getEarliestEventTs(long startTs, long endTs) {
        long minTs = Long.MAX_VALUE;
        for (Event<T> event : this.queue) {
            if (event.getTimestamp() <= startTs || event.getTimestamp() > endTs) continue;
            minTs = Math.min(minTs, event.getTimestamp());
        }
        return minTs;
    }

    public int getEventCount(long referenceTime) {
        int count = 0;
        for (Event<T> event : this.queue) {
            if (event.getTimestamp() > referenceTime) continue;
            ++count;
        }
        return count;
    }

    public List<Long> getSlidingCountTimestamps(long startTs, long endTs, int slidingCount) {
        ArrayList<Long> timestamps = new ArrayList<Long>();
        if (endTs > startTs) {
            int count = 0;
            long ts = Long.MIN_VALUE;
            for (Event<T> event : this.queue) {
                if (event.getTimestamp() <= startTs || event.getTimestamp() > endTs) continue;
                ts = Math.max(ts, event.getTimestamp());
                if (++count % slidingCount != 0) continue;
                timestamps.add(ts);
            }
        }
        return timestamps;
    }

    public String toString() {
        return "WindowManager{evictionPolicy=" + this.evictionPolicy + ", triggerPolicy=" + this.triggerPolicy + '}';
    }
}

