/*******************************************************************************
 * Copyright (c) 2000, 2022 IBM Corporation and others.
 *
 * This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.aspectj.org.eclipse.jdt.internal.compiler.parser;

import java.util.HashMap;
import java.util.Map;

/**
 * IMPORTANT NOTE: These constants are dedicated to the internal Scanner implementation.
 * It is mirrored in org.eclipse.jdt.core.compiler public package where it is API.
 * The mirror implementation is using the backward compatible ITerminalSymbols constant
 * definitions (stable with 2.0), whereas the internal implementation uses TerminalTokens
 * which constant values reflect the latest parser generation state.
 */
/**
 * Maps each terminal symbol in the java-grammar into a unique integer.
 * This integer is used to represent the terminal when computing a parsing action.
 *
 * Disclaimer : These constant values are generated automatically using a Java
 * grammar, therefore their actual values are subject to change if new keywords
 * were added to the language (for instance, 'assert' is a keyword in 1.4).
 */
public enum TerminalToken {

	// special tokens not part of grammar - not autogenerated
							TokenNameInvalid(-1),
							TokenNameNotAToken(0),
							TokenNameWHITESPACE(1000),
							TokenNameCOMMENT_LINE(1001),
							TokenNameCOMMENT_BLOCK(1002),
							TokenNameCOMMENT_JAVADOC(1003),
							TokenNameSingleQuoteStringLiteral(1004),
							TokenNameCOMMENT_MARKDOWN(1005),

	// BEGIN_AUTOGENERATED_REGION
							TokenNameIdentifier(12),
							TokenNameabstract(49),
							TokenNameassert(87),
							TokenNameboolean(119),
							TokenNamebreak(88),
							TokenNamebyte(120),
							TokenNamecase(98),
							TokenNamecatch(99),
							TokenNamechar(121),
							TokenNameclass(76),
							TokenNamecontinue(89),
							TokenNameconst(143),
							TokenNamedefault(96),
							TokenNamedo(90),
							TokenNamedouble(122),
							TokenNameelse(103),
							TokenNameenum(81),
							TokenNameextends(97),
							TokenNamefalse(60),
							TokenNamefinal(50),
							TokenNamefinally(102),
							TokenNamefloat(123),
							TokenNamefor(91),
							TokenNamegoto(144),
							TokenNameif(92),
							TokenNameimplements(106),
							TokenNameimport(100),
							TokenNameinstanceof(17),
							TokenNameint(124),
							TokenNameinterface(80),
							TokenNamelong(125),
							TokenNamenative(51),
							TokenNamenew(43),
							TokenNamenon_sealed(52),
							TokenNamenull(61),
							TokenNamepackage(95),
							TokenNameprivate(53),
							TokenNameprotected(54),
							TokenNamepublic(55),
							TokenNamereturn(93),
							TokenNameshort(126),
							TokenNamestatic(44),
							TokenNamestrictfp(56),
							TokenNamesuper(45),
							TokenNameswitch(71),
							TokenNamesynchronized(47),
							TokenNamethis(46),
							TokenNamethrow(85),
							TokenNamethrows(104),
							TokenNametransient(57),
							TokenNametrue(62),
							TokenNametry(94),
							TokenNamevoid(127),
							TokenNamevolatile(58),
							TokenNamewhile(86),
							TokenNamemodule(128),
							TokenNameopen(130),
							TokenNamerequires(131),
							TokenNametransitive(136),
							TokenNameexports(132),
							TokenNameopens(133),
							TokenNameto(140),
							TokenNameuses(134),
							TokenNameprovides(135),
							TokenNamewith(141),
							TokenNameaspect(24),
							TokenNamepointcut(28),
							TokenNamearound(33),
							TokenNamebefore(30),
							TokenNameafter(31),
							TokenNamedeclare(32),
							TokenNameprivileged(27),
							TokenNameIntegerLiteral(63),
							TokenNameLongLiteral(64),
							TokenNameFloatingPointLiteral(65),
							TokenNameDoubleLiteral(66),
							TokenNameCharacterLiteral(67),
							TokenNameStringLiteral(68),
							TokenNameTextBlock(69),
							TokenNamePLUS_PLUS(3),
							TokenNameMINUS_MINUS(4),
							TokenNameEQUAL_EQUAL(23),
							TokenNameLESS_EQUAL(18),
							TokenNameGREATER_EQUAL(19),
							TokenNameNOT_EQUAL(20),
							TokenNameLEFT_SHIFT(21),
							TokenNameRIGHT_SHIFT(13),
							TokenNameUNSIGNED_RIGHT_SHIFT(16),
							TokenNamePLUS_EQUAL(108),
							TokenNameMINUS_EQUAL(109),
							TokenNameMULTIPLY_EQUAL(110),
							TokenNameDIVIDE_EQUAL(111),
							TokenNameAND_EQUAL(112),
							TokenNameOR_EQUAL(113),
							TokenNameXOR_EQUAL(114),
							TokenNameREMAINDER_EQUAL(115),
							TokenNameLEFT_SHIFT_EQUAL(116),
							TokenNameRIGHT_SHIFT_EQUAL(117),
							TokenNameUNSIGNED_RIGHT_SHIFT_EQUAL(118),
							TokenNameOR_OR(38),
							TokenNameAND_AND(37),
							TokenNamePLUS(2),
							TokenNameMINUS(6),
							TokenNameNOT(74),
							TokenNameREMAINDER(10),
							TokenNameXOR(34),
							TokenNameAND(22),
							TokenNameMULTIPLY(8),
							TokenNameOR(36),
							TokenNameTWIDDLE(77),
							TokenNameDIVIDE(11),
							TokenNameGREATER(15),
							TokenNameLESS(7),
							TokenNameLPAREN(14),
							TokenNameRPAREN(25),
							TokenNameLBRACE(70),
							TokenNameRBRACE(42),
							TokenNameLBRACKET(5),
							TokenNameRBRACKET(75),
							TokenNameSEMICOLON(29),
							TokenNameQUESTION(35),
							TokenNameCOLON(72),
							TokenNameCOMMA(39),
							TokenNameDOT(1),
							TokenNameEQUAL(79),
							TokenNameAT(40),
							TokenNameELLIPSIS(105),
							TokenNameARROW(129),
							TokenNameCOLON_COLON(9),
							TokenNameBeginLambda(73),
							TokenNameBeginIntersectionCast(78),
							TokenNameBeginTypeArguments(101),
							TokenNameElidedSemicolonAndRightBrace(82),
							TokenNameAT308(26),
							TokenNameAT308DOTDOTDOT(142),
							TokenNameCaseArrow(83),
							TokenNameRestrictedIdentifierYield(107),
							TokenNameRestrictedIdentifierrecord(84),
							TokenNameRestrictedIdentifiersealed(48),
							TokenNameRestrictedIdentifierpermits(137),
							TokenNameBeginCasePattern(138),
							TokenNameRestrictedIdentifierWhen(139),
							TokenNameUNDERSCORE(41),
							TokenNameEOF(59),
							TokenNameERROR(145);

// END_AUTOGENERATED_REGION

	private final static Map<Integer, TerminalToken> tokenMap = new HashMap<>(256);
	static {
		for (TerminalToken t : TerminalToken.values()) {
			tokenMap.put(t.tokenNumber(), t);
		}
	}

	private final int tokenNumber;

	// Constructor
	TerminalToken(int number) {
		this.tokenNumber = number;
	}

	public int tokenNumber() {
		return this.tokenNumber;
	}

	public static boolean isRestrictedKeyword(TerminalToken tokenType) {
		return switch (tokenType) {
			case TokenNameRestrictedIdentifierYield, TokenNameRestrictedIdentifierrecord, TokenNameRestrictedIdentifierWhen,
					TokenNameRestrictedIdentifiersealed, TokenNameRestrictedIdentifierpermits -> true;
			default -> false;
		};
	}

	public static TerminalToken getRestrictedKeyword(char [] text) {
		if (text != null) {
			int len = text.length;
			if (len == 4 && text[0] == 'w' ||
				len == 5 && text[0] == 'y' ||
				len == 6 && (text[0] == 'r' || text[0] == 's') ||
				len == 7 && text[0] == 'p') {
				return getRestrictedKeyword(new String(text));
			}
		}
		return TokenNameNotAToken;
	}

	public static TerminalToken getRestrictedKeyword(String text) {
		return switch (text) {
			case "yield"   -> TokenNameRestrictedIdentifierYield;   //$NON-NLS-1$
			case "record"  -> TokenNameRestrictedIdentifierrecord;  //$NON-NLS-1$
			case "when"    -> TokenNameRestrictedIdentifierWhen;    //$NON-NLS-1$
			case "sealed"  -> TokenNameRestrictedIdentifiersealed;  //$NON-NLS-1$
			case "permits" -> TokenNameRestrictedIdentifierpermits; //$NON-NLS-1$
			default        -> TokenNameNotAToken;
		};
	}

	public static TerminalToken of(int act) {
		TerminalToken token = tokenMap.get(act);
		if (token == null) {
			// Really shouldn't occur -- perhaps if parser non-terminals are looked up?
			throw new IllegalArgumentException("Unknown token number = " + act); //$NON-NLS-1$
		}
		return token;
	}

	static TerminalToken maybeOf(int act) {
		return tokenMap.get(act);
	}

}

