/**********************************************************************
Copyright (c) 2009 Erik Bengtson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
   ...
**********************************************************************/
package org.datanucleus.store;

import org.datanucleus.ClassLoaderResolver;
import org.datanucleus.FetchPlan;
import org.datanucleus.OMFContext;
import org.datanucleus.ObjectManagerFactoryImpl;
import org.datanucleus.Transaction;
import org.datanucleus.api.ApiAdapter;
import org.datanucleus.metadata.AbstractClassMetaData;
import org.datanucleus.metadata.AbstractMemberMetaData;
import org.datanucleus.metadata.MetaDataManager;
import org.datanucleus.state.FetchPlanState;
import org.datanucleus.state.lock.LockManager;
import org.datanucleus.store.query.Query;
import org.datanucleus.store.types.TypeManager;

/**
 * Context of execution for persistence operations
 */
public interface ExecutionContext
{
    public static interface LifecycleListener
    {
        /**
         * Invoked before closing the ObjectManager
         * @param ec execution context
         */   
        void preClose(ExecutionContext ec);
    }

    /**
     * @deprecated Please make use of IdentityUtils.getApplicationIdentityForResultSetRow()
     * and then call findObject(Object oid, FieldValues2 fieldValues2, Class pcClass, boolean ignoreCache)
     */
    Object findObjectUsingAID(Type pcClass, FieldValues2 fv, boolean ignoreCache, boolean checkInheritance);

    /**
     * Accessor for the MetaData Manager.
     * @return The MetaData Manager
     */
    MetaDataManager getMetaDataManager();

    /**
     * Accessor for the LockManager.
     * @return The Lock Manager
     */
    LockManager getLockManager();

    /**
     * Accessor for the current transaction for this ExecutionContext.
     * @return The current transaction
     */
    Transaction getTransaction();

    /**
     * Accessor for the ClassLoader resolver to use in class loading issues.
     * @return The ClassLoader resolver
     */
    ClassLoaderResolver getClassLoaderResolver();
    
    /**
     * TODO should we keep this here? this is api/language dependent
     * @return The type manager
     */
    TypeManager getTypeManager();
    
    ObjectProvider findObjectProvider(Object object);

    ObjectProvider findObjectProvider(Object object, boolean persist);
    
    ApiAdapter getApiAdapter();
    
    ObjectProvider findObjectProviderForEmbedded(Object value, ObjectProvider owner, AbstractMemberMetaData mmd);
    
    ObjectProvider newObjectProviderForMember(AbstractMemberMetaData mmd);

    ObjectProvider newObjectProviderForMember(AbstractMemberMetaData mmd, Type effectiveType);

    ObjectProvider newObjectProviderForMember(AbstractMemberMetaData mmd, AbstractClassMetaData effectiveTypeCmd);

    void deleteObjectInternal(Object pc);
    
    /**
     * Method to persist the passed object (internally).
     * @param pc The object
     * @param ownerSM StateManager of the owner when embedded
     * @param ownerFieldNum Field number in the owner where this is embedded (or -1 if not embedded)
     * @param objectType Type of object (see org.datanucleus.StateManager, e.g StateManager.PC)
     * @return The persisted object
     */
    Object persistObjectInternal(Object pc, ObjectProvider ownerSM, int ownerFieldNum, int objectType);
    
    boolean isClosed();

    FetchPlan getFetchPlan();

    Query newQuery();

    OMFContext getOMFContext();

    StoreManager getStoreManager();

    ObjectProvider newObjectProvider(Object id, Object obj);
    
    Object findObject(Object idForObject, boolean b, boolean c, String name);

    void flushInternal(boolean b);

    void detachObject(Object val, FetchPlanState state);

    void deleteObjects(Object[] array);

    Object attachObjectCopy(Object value, boolean sco);

    Object detachObjectCopy(Object val, FetchPlanState state);

    void refreshObject(Object val);

    void evictFromTransaction(ObjectProvider objSM);

    boolean isFlushing();

    boolean getIgnoreCache();

    Integer getDatastoreReadTimeoutMillis();

    Integer getDatastoreWriteTimeoutMillis();

    boolean isDelayDatastoreOperationsEnabled();

    void markDirty(ObjectProvider otherSM, boolean b);

    Extent getExtent(Class candidateClass, boolean subclasses);

    void attachObject(Object object, boolean elementsWithoutIdentity);

    Object getObjectFromCache(Object idForObject);

    void removeObjectFromCache(Object value, Object idForObject);

    /**
     * Whether an object with the specified identity exists in the cache(s).
     * Used as a check on identity (inheritance-level) validity
     * @param id The identity
     * @return Whether it exists
     */
    boolean hasIdentityInCache(Object id);

    Object findObject(Object oid, FieldValues2 fieldValues2, Class pcClass, boolean ignoreCache);

    boolean getSerializeReadForClass(String fullClassName);

    void hasPersistenceInformationForClass(Class candidateClass);

    void makeObjectTransient(Object pc, FetchPlanState state);

    boolean isInserting(Object value);

    Object getAttachedObjectForId(Object idForObject);

    ObjectManagerFactoryImpl getObjectManagerFactory();

    void persistObjectInternal(Object object, FieldValues2 fieldValues, int pc);

    Object newObjectId(String className, Object object);

    Object newObjectId(Class clazz, Object object);
    
    void deleteObject(Object obj);
}