/**********************************************************************
Copyright (c) 2005 Andy Jefferson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License. 


Contributors:
    ...
**********************************************************************/
package org.datanucleus.store.mapped.scostore;

import org.datanucleus.ClassLoaderResolver;
import org.datanucleus.exceptions.NucleusUserException;
import org.datanucleus.metadata.AbstractClassMetaData;
import org.datanucleus.metadata.AbstractMemberMetaData;
import org.datanucleus.metadata.FieldRole;
import org.datanucleus.metadata.MetaDataUtils;
import org.datanucleus.store.mapped.DatastoreClass;
import org.datanucleus.store.mapped.DatastoreContainerObject;
import org.datanucleus.store.mapped.mapping.JavaTypeMapping;
import org.datanucleus.util.ClassUtils;

/**
 * Backing store for an array stored using a join table.
 * Can be used for all types of arrays :- PC arrays, non-PC arrays, reference arrays.
 */
public abstract class JoinArrayStore extends AbstractArrayStore
{
    /**
     * Constructor.
     * @param joinTable Join table storing the relationship between owner and element
     * @param clr ClassLoader resolver
     */
    public JoinArrayStore(DatastoreContainerObject joinTable, AbstractMemberMetaData ownerFieldMetaData,
                          JavaTypeMapping ownerMapping, JavaTypeMapping elementMapping, JavaTypeMapping orderMapping,
                          JavaTypeMapping relationDiscriminatorMapping, String relationDiscriminatorValue,
                          String elementType, boolean elementsAreEmbedded, boolean elementsAreSerialised,
                          ClassLoaderResolver clr, JoinArrayStoreSpecialization specialization)
    {
        super(joinTable.getStoreManager(), clr, specialization);

        this.containerTable = joinTable;
        setOwner(ownerFieldMetaData, clr);

        this.ownerMapping = ownerMapping;
        this.elementMapping = elementMapping;
        this.orderMapping = orderMapping;
        this.relationDiscriminatorMapping = relationDiscriminatorMapping;
        this.relationDiscriminatorValue = relationDiscriminatorValue;

        this.elementType = elementType;
        this.elementsAreEmbedded = elementsAreEmbedded;
        this.elementsAreSerialised = elementsAreSerialised;

        if (this.elementsAreSerialised)
        {
            this.elementInfo = null;
        }
        else
        {
            Class element_class=clr.classForName(elementType);
            if (ClassUtils.isReferenceType(element_class))
            {
                // Array of reference types (interfaces/Objects)
                String[] implNames = MetaDataUtils.getInstance().getImplementationNamesForReferenceField(ownerMemberMetaData,
                    FieldRole.ROLE_ARRAY_ELEMENT, clr, storeMgr.getMetaDataManager());
                elementInfo = new ElementInfo[implNames.length];
                for (int i=0;i<implNames.length;i++)
                {
                    DatastoreClass table = storeMgr.getDatastoreClass(implNames[i], clr);
                    AbstractClassMetaData cmd = storeMgr.getOMFContext().getMetaDataManager().getMetaDataForClass(implNames[i], clr);
                    elementInfo[i] = new ElementInfo(cmd, table);
                }
            }
            else
            {
                // Collection of PC or non-PC
                emd = storeMgr.getOMFContext().getMetaDataManager().getMetaDataForClass(element_class, clr);
                if (emd != null)
                {
                    elementType  = emd.getFullClassName();
                    if (!elementsAreEmbedded && !elementsAreSerialised)
                    {
                        elementInfo = getElementInformationForClass();
                        if (elementInfo != null && elementInfo.length > 1)
                        {
                            throw new NucleusUserException(LOCALISER.msg("056045", 
                                ownerMemberMetaData.getFullFieldName()));
                        }
                    }
                    else
                    {
                        elementInfo = null;
                    }
                }
                else
                {
                    elementInfo = null;
                }
            }
        }
    }
}