/**********************************************************************
Copyright (c) 2002 Mike Martin (TJDO) and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
2003 Erik Bengtson - removed unused import
2003 Andy Jefferson - coding standards
2004 Andy Jefferson - moved statements to AbstractSetStore
2009 Andy Jefferson - rewritten to be applicable to other RDBMS-like datastores
    ...
**********************************************************************/
package org.datanucleus.store.mapped.scostore;

import java.util.Collection;

import org.datanucleus.ClassLoaderResolver;
import org.datanucleus.exceptions.NucleusUserException;
import org.datanucleus.metadata.AbstractMemberMetaData;
import org.datanucleus.store.ObjectProvider;
import org.datanucleus.store.mapped.DatastoreClass;
import org.datanucleus.store.mapped.DatastoreContainerObject;
import org.datanucleus.store.mapped.mapping.JavaTypeMapping;
import org.datanucleus.store.scostore.MapStore;
import org.datanucleus.util.ClassUtils;

/**
 * Representation of the backing store for the values of a Map.
 * This is used where the user calls Map.values() and then wants to perform some operation on the resulting
 * Collection. The values for a Map can be stored in several ways. There are the following possibilities
 * <UL>
 * <LI><B>Map using join table</B> - join table contains the key and value (or FK's to their tables)</LI>
 * <LI><B>Map using key table</B> - Key table stores the value (or an FK to its table)</LI>
 * <LI><B>Map using value table</B> - Value table stores the key (or an FK to its table)</LI>
 * </UL>
 */
public abstract class MapValueSetStore extends AbstractSetStore
{
    protected final MapStore mapStore;

    protected final JavaTypeMapping keyMapping;

    /**
     * Constructor where a join table is used to store the map relation.
     * @param mapTable Table used by the map
     * @param ownerMmd Metadata for the owning field/property
     * @param ownerMapping the owner mapping
     * @param keyMapping the key mapping
     * @param valueMapping the value mapping
     * @param mapStore Store used by the map
     * @param clr The ClassLoaderResolver
     * @param specialization the specialisation
     */
    public MapValueSetStore(DatastoreContainerObject mapTable, AbstractMemberMetaData ownerMmd,
            JavaTypeMapping ownerMapping, JavaTypeMapping keyMapping, JavaTypeMapping valueMapping,
            MapStore mapStore, ClassLoaderResolver clr, AbstractSetStoreSpecialization specialization)
    {
        super(mapTable.getStoreManager(), clr, specialization);

        this.containerTable = mapTable;
        this.mapStore = mapStore;
        this.ownerMapping = ownerMapping;
        this.keyMapping = keyMapping;
        this.elementMapping = valueMapping;
        this.elementType = elementMapping.getType();
        this.ownerMemberMetaData = ownerMmd;

        initialize(clr);
    }

    /**
     * Constructor where either the key is stored in the value table or the value is stored in the key table.
     * @param mapTable Table storing the map relation (key or value table)
     * @param ownerMmd FieldMetaData for the owning "map" field
     * @param ownerMapping Mapping to the owner from this table
     * @param valueMapping Mapping for this table to the value
     * @param mapStore Store used by the map.
     * @param clr The ClassLoaderResolver
     * @param specialization the specialization
     */
    public MapValueSetStore(DatastoreClass mapTable, AbstractMemberMetaData ownerMmd, 
            JavaTypeMapping ownerMapping, JavaTypeMapping valueMapping, 
            MapStore mapStore, ClassLoaderResolver clr, AbstractSetStoreSpecialization specialization)
    {
        super(mapTable.getStoreManager(), clr, specialization);

        this.containerTable = mapTable;
        this.mapStore = mapStore;
        this.ownerMapping = ownerMapping;
        this.keyMapping = null;
        this.elementMapping = valueMapping;
        this.ownerMemberMetaData = ownerMmd;

        initialize(clr);
    }

    /**
     * Initialise Method.
     * @param clr ClassLoader resolver
     */
    private void initialize(ClassLoaderResolver clr)
    {
        elementType = elementMapping.getType();
        elementsAreEmbedded = isEmbeddedMapping(elementMapping);
        elementsAreSerialised = isEmbeddedMapping(elementMapping);

        Class valueCls = clr.classForName(elementType);
        if (ClassUtils.isReferenceType(valueCls))
        {
            emd = storeMgr.getOMFContext().getMetaDataManager().getMetaDataForImplementationOfReference(valueCls,null,clr);
        }
        else
        {
            emd = storeMgr.getOMFContext().getMetaDataManager().getMetaDataForClass(valueCls, clr);
        }
        if (emd != null)
        {
            elementType = emd.getFullClassName();
            elementInfo = getElementInformationForClass();
        }
    }

    /**
     * Method to add a value to the Map. Not supported.
     * @param sm State Manager for the owner
     * @param element The value to add
     * @return Whether it was added correctly.
     */
    public boolean add(ObjectProvider sm, Object element, int size)
    {
        throw new UnsupportedOperationException("Cannot add to a map through its values collection");
    }

    /**
     * Method to add entries to the Map. Not supported.
     * @param sm State Manager for the owner
     * @param elements The values to add
     * @return Whether it was added correctly.
     */
    public boolean addAll(ObjectProvider sm, Collection elements, int size)
    {
        throw new UnsupportedOperationException("Cannot add to a map through its values collection");
    }

    /**
     * Method to remove a value from the Map.
     * @param sm State Manager for the owner
     * @param element The value to remove
     * @return Whether it was removed correctly.
     */
    public boolean remove(ObjectProvider sm, Object element, int size, boolean allowDependentField)
    {
        if (!validateElementForReading(sm, element))
        {
            return false;
        }

        return remove(sm, element);
    }

    /**
     * Method to remove values from the map via the value collection.
     * @param sm StateManager for the owner
     */
    public boolean removeAll(ObjectProvider sm, Collection elements, int size)
    {
        throw new NucleusUserException("Cannot remove values from a map through its values collection");
    }

    /**
     * Method to clear the map.
     * @param sm StateManager for the owner
     */
    public void clear(ObjectProvider sm)
    {
        if (canClear())
        {
            throw new NucleusUserException("Cannot clear a map through its values collection");
        }

        super.clear(sm);
    }

    protected abstract boolean canClear();
    protected abstract boolean remove(ObjectProvider sm, Object element);
}