/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.cbi.maven.plugins.jarsigner;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Joiner;
import com.google.common.base.Preconditions;
import java.io.BufferedOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.DirectoryStream;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.util.EnumSet;
import java.util.List;
import java.util.Locale;
import java.util.Objects;
import java.util.Set;
import java.util.jar.Attributes;
import java.util.jar.JarEntry;
import java.util.jar.JarInputStream;
import java.util.jar.Manifest;
import java.util.stream.Collectors;
import java.util.stream.StreamSupport;
import org.apache.maven.plugin.logging.Log;
import org.eclipse.cbi.common.security.MessageDigestAlgorithm;
import org.eclipse.cbi.common.util.Paths;
import org.eclipse.cbi.common.util.Zips;
import org.eclipse.cbi.maven.plugins.jarsigner.JarSigner;

public abstract class JarResigner
implements JarSigner {
    private static final String DIGEST_ATTRIBUTE_SUFFIX = "-Digest";
    private final JarSigner delegate;
    private final Log log;

    JarResigner(JarSigner delegate, Log log) {
        this.delegate = (JarSigner)Preconditions.checkNotNull((Object)delegate);
        this.log = (Log)Preconditions.checkNotNull((Object)log);
    }

    final Log log() {
        return this.log;
    }

    public static JarSigner create(Strategy strategy, JarSigner delegate, Log log) {
        JarResigner ret;
        switch (strategy) {
            case DO_NOT_RESIGN: {
                ret = new DoNotResign(delegate, log);
                break;
            }
            case THROW_EXCEPTION: {
                ret = new ThrowException(delegate, log);
                break;
            }
            case RESIGN: {
                ret = new Resign(delegate, log);
                break;
            }
            case RESIGN_WITH_SAME_DIGEST_ALGORITHM: {
                ret = new ResignWithSameDigestAlg(delegate, log);
                break;
            }
            case OVERWRITE: {
                ret = new OverwriteSignature(delegate, log);
                break;
            }
            case OVERWRITE_WITH_SAME_DIGEST_ALGORITHM: {
                ret = new OverwriteSignatureWithSameDigestAlg(delegate, log);
                break;
            }
            default: {
                throw new IllegalStateException("Unknow resigning strategy: " + (Object)((Object)strategy));
            }
        }
        return ret;
    }

    JarSigner delegate() {
        return this.delegate;
    }

    @Override
    public int sign(Path jar, JarSigner.Options options) throws IOException {
        int ret = JarResigner.isAlreadySigned(jar) ? this.resign(jar, options) : this.delegate().sign(jar, options);
        return ret;
    }

    abstract int resign(Path var1, JarSigner.Options var2) throws IOException;

    @VisibleForTesting
    static boolean isAlreadySigned(Path jar) throws IOException {
        boolean alreadySigned = false;
        try (JarInputStream jis = new JarInputStream(Files.newInputStream(jar, new OpenOption[0]));){
            JarEntry nextJarEntry = jis.getNextJarEntry();
            while (nextJarEntry != null && !alreadySigned) {
                alreadySigned = JarResigner.isBlockOrSF(nextJarEntry.getName()) || JarResigner.hasManifestDigest(nextJarEntry.getAttributes());
                nextJarEntry = jis.getNextJarEntry();
            }
        }
        return alreadySigned;
    }

    static boolean hasManifestDigest(Attributes entryAttributes) {
        if (entryAttributes != null) {
            return entryAttributes.keySet().stream().anyMatch(k -> k.toString().endsWith(DIGEST_ATTRIBUTE_SUFFIX));
        }
        return false;
    }

    private static boolean isBlockOrSF(String entryName) {
        String uname = entryName.toUpperCase(Locale.ENGLISH);
        if (uname.startsWith("META-INF/") || uname.startsWith("/META-INF/")) {
            return uname.endsWith(".SF") || uname.endsWith(".DSA") || uname.endsWith(".RSA") || uname.endsWith(".EC");
        }
        return false;
    }

    public static JarSigner doNotResign(JarSigner jarSigner, Log log) {
        return new DoNotResign(jarSigner, log);
    }

    public static JarSigner throwException(JarSigner jarSigner, Log log) {
        return new ThrowException(jarSigner, log);
    }

    public static JarSigner resignWithSameDigestAlgorithm(JarSigner jarSigner, Log log) {
        return new ResignWithSameDigestAlg(jarSigner, log);
    }

    public static JarSigner resign(JarSigner jarSigner, Log log) {
        return new Resign(jarSigner, log);
    }

    public static JarSigner overwriteWithSameDigestAlgorithm(JarSigner jarSigner, Log log) {
        return new OverwriteSignatureWithSameDigestAlg(jarSigner, log);
    }

    public static JarSigner overwrite(JarSigner jarSigner, Log log) {
        return new OverwriteSignature(jarSigner, log);
    }

    @VisibleForTesting
    static Set<MessageDigestAlgorithm> getAllUsedDigestAlgorithm(Path jar) throws IOException {
        EnumSet<MessageDigestAlgorithm> usedDigestAlg = EnumSet.noneOf(MessageDigestAlgorithm.class);
        try (JarInputStream jis = new JarInputStream(Files.newInputStream(jar, new OpenOption[0]));){
            JarEntry jarEntry = jis.getNextJarEntry();
            while (jarEntry != null) {
                Attributes attributes = jarEntry.getAttributes();
                if (attributes != null) {
                    for (Object k : attributes.keySet()) {
                        if (!k.toString().endsWith(DIGEST_ATTRIBUTE_SUFFIX)) continue;
                        String digestAlgName = k.toString().substring(0, k.toString().length() - DIGEST_ATTRIBUTE_SUFFIX.length());
                        usedDigestAlg.add(MessageDigestAlgorithm.fromStandardName((String)digestAlgName));
                    }
                }
                jarEntry = jis.getNextJarEntry();
            }
        }
        return usedDigestAlg;
    }

    @VisibleForTesting
    static MessageDigestAlgorithm getDigestAlgorithmToReuse(Path jar) throws IOException {
        Set<MessageDigestAlgorithm> usedDigestAlg = JarResigner.getAllUsedDigestAlgorithm(jar);
        if (usedDigestAlg.isEmpty()) {
            throw new IllegalArgumentException("Jar '" + jar + "' is not signed while it was asked to be resigned.");
        }
        if (usedDigestAlg.size() > 1) {
            throw new IllegalArgumentException("Can't resign with the same digest algorithm. Jar '" + jar + "' contains entries that has been signed with more than one digest algorithm: '" + Joiner.on((String)", ").join(usedDigestAlg) + "'. Use another strategy to resign.");
        }
        return usedDigestAlg.iterator().next();
    }

    private static class OverwriteSignatureWithSameDigestAlg
    extends OverwriteSignature {
        private OverwriteSignatureWithSameDigestAlg(JarSigner delegate, Log log) {
            super(delegate, log);
        }

        @Override
        protected int resign(Path jar, JarSigner.Options options) throws IOException {
            this.log().debug((CharSequence)("Jar signing options before change by strategy '" + (Object)((Object)Strategy.OVERWRITE_WITH_SAME_DIGEST_ALGORITHM) + "': " + options));
            JarSigner.Options newOptions = JarSigner.Options.copy(options).digestAlgorithm(OverwriteSignatureWithSameDigestAlg.getDigestAlgorithmToReuse(jar)).build();
            return super.resign(jar, newOptions);
        }
    }

    private static class ResignWithSameDigestAlg
    extends Resign {
        public ResignWithSameDigestAlg(JarSigner delegate, Log log) {
            super(delegate, log);
        }

        @Override
        protected int resign(Path jar, JarSigner.Options options) throws IOException {
            this.log().debug((CharSequence)("Jar signing options before change by strategy '" + (Object)((Object)Strategy.RESIGN_WITH_SAME_DIGEST_ALGORITHM) + "': " + options));
            JarSigner.Options newOptions = JarSigner.Options.copy(options).digestAlgorithm(ResignWithSameDigestAlg.getDigestAlgorithmToReuse(jar)).build();
            return super.resign(jar, newOptions);
        }
    }

    private static class OverwriteSignature
    extends JarResigner {
        OverwriteSignature(JarSigner delegate, Log log) {
            super(delegate, log);
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        protected int resign(Path jar, JarSigner.Options options) throws IOException {
            Path unpackedJar = Files.createTempDirectory(Paths.getParent((Path)jar), "overwriteSignature-" + jar.getFileName() + "-", new FileAttribute[0]);
            try {
                Zips.unpackJar((Path)jar, (Path)unpackedJar);
                Path metaInf = unpackedJar.resolve("META-INF");
                boolean signatureFilesFound = this.removeSignatureFilesIfAny(metaInf);
                boolean manifestDigestsFound = this.removeManifestDigestsIfAny(metaInf.resolve("MANIFEST.MF"));
                if (signatureFilesFound || manifestDigestsFound) {
                    this.log().info((CharSequence)("Jar '" + jar.toString() + "' is already signed. The signature will be overwritten."));
                    Zips.packJar((Path)unpackedJar, (Path)jar, (boolean)false);
                } else {
                    this.log().info((CharSequence)("No signature was found in Jar '" + jar.toString() + "', it will be signed without touching it. Signature would have been overwritten otherwise."));
                }
                int n = this.delegate().sign(jar, options);
                return n;
            }
            finally {
                Paths.deleteQuietly((Path)unpackedJar);
            }
        }

        private boolean removeSignatureFilesIfAny(Path metaInf) throws IOException {
            if (Files.exists(metaInf, new LinkOption[0])) {
                try (DirectoryStream<Path> content = Files.newDirectoryStream(metaInf, "*.{SF,DSA,RSA,EC}");){
                    List signatureFiles = StreamSupport.stream(content.spliterator(), false).collect(Collectors.toList());
                    for (Path f : signatureFiles) {
                        this.log().debug((CharSequence)("Deleting signature file '" + f + "'"));
                        Paths.delete((Path)f);
                    }
                    boolean bl = !signatureFiles.isEmpty();
                    return bl;
                }
            }
            return false;
        }

        private boolean removeManifestDigestsIfAny(Path manifestPath) throws IOException {
            if (Files.exists(manifestPath, new LinkOption[0])) {
                Manifest manifest = OverwriteSignature.readManifest(manifestPath);
                List<String> keysOfRemovedDigests = this.removeDigestAttributes(manifest);
                if (!keysOfRemovedDigests.isEmpty()) {
                    this.pruneEmptyEntries(manifest, keysOfRemovedDigests);
                    OverwriteSignature.writeManifest(manifest, manifestPath);
                }
                return !keysOfRemovedDigests.isEmpty();
            }
            return false;
        }

        private List<String> removeDigestAttributes(Manifest manifest) {
            return manifest.getEntries().entrySet().stream().map(e -> {
                if (((Attributes)e.getValue()).keySet().removeIf(k -> k.toString().endsWith(JarResigner.DIGEST_ATTRIBUTE_SUFFIX))) {
                    this.log().debug((CharSequence)("Deleting digest attribute(s) of entry '" + (String)e.getKey() + "'"));
                    return (String)e.getKey();
                }
                return null;
            }).filter(Objects::nonNull).collect(Collectors.toList());
        }

        private void pruneEmptyEntries(Manifest manifest, List<String> keysOfRemovedDigests) {
            keysOfRemovedDigests.forEach(k -> {
                if (manifest.getAttributes((String)k).isEmpty()) {
                    this.log().debug((CharSequence)("Deleting manifest entry for '" + k + "' as it has no attribute anymore"));
                    manifest.getEntries().remove(k);
                }
            });
        }

        private static void writeManifest(Manifest manifest, Path path) throws IOException {
            try (BufferedOutputStream os = new BufferedOutputStream(Files.newOutputStream(path, StandardOpenOption.WRITE, StandardOpenOption.CREATE, StandardOpenOption.TRUNCATE_EXISTING));){
                manifest.write(os);
            }
        }

        private static Manifest readManifest(Path manifestPath) throws IOException {
            Manifest manifest = new Manifest();
            try (InputStream is = Files.newInputStream(manifestPath, StandardOpenOption.READ);){
                manifest.read(is);
            }
            return manifest;
        }
    }

    private static class Resign
    extends JarResigner {
        Resign(JarSigner delegate, Log log) {
            super(delegate, log);
        }

        @Override
        protected int resign(Path jar, JarSigner.Options options) throws IOException {
            this.log().info((CharSequence)("Jar '" + jar.toString() + "' is already signed and will be resigned."));
            return this.delegate().sign(jar, options);
        }
    }

    private static class ThrowException
    extends JarResigner {
        ThrowException(JarSigner delegate, Log log) {
            super(delegate, log);
        }

        @Override
        public int resign(Path jar, JarSigner.Options options) {
            throw new IllegalStateException("Jar '" + jar + "' is already signed");
        }
    }

    private static class DoNotResign
    extends JarResigner {
        DoNotResign(JarSigner delegate, Log log) {
            super(delegate, log);
        }

        @Override
        public int resign(Path jar, JarSigner.Options options) {
            this.log().info((CharSequence)("Jar '" + jar.toString() + "' is already signed and will *not* be resigned."));
            return 0;
        }
    }

    public static enum Strategy {
        DO_NOT_RESIGN,
        THROW_EXCEPTION,
        RESIGN,
        RESIGN_WITH_SAME_DIGEST_ALGORITHM,
        OVERWRITE,
        OVERWRITE_WITH_SAME_DIGEST_ALGORITHM;

    }
}

