package org.exoplatform.cs.portlet.ticketAdministration;

import juzu.HttpMethod;
import juzu.MimeType;
import juzu.Path;
import juzu.Resource;
import juzu.Response;
import juzu.View;
import juzu.plugin.jackson.Jackson;
import juzu.template.Template;
import org.exoplatform.commons.juzu.ajax.Ajax;
import org.exoplatform.commons.utils.ListAccess;
import org.exoplatform.commons.utils.PropertyManager;
import org.exoplatform.cs.dto.IssueSeverity;
import org.exoplatform.cs.dto.IssueType;
import org.exoplatform.cs.dto.TicketDTO;
import org.exoplatform.cs.entity.SpaceEntity;
import org.exoplatform.cs.portlet.common.CSControllerBase;
import org.exoplatform.cs.service.TicketService;
import org.exoplatform.services.log.ExoLogger;
import org.exoplatform.services.log.Log;
import org.exoplatform.services.organization.OrganizationService;
import org.exoplatform.services.organization.User;

import javax.inject.Inject;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;

/**
 * Created by exo on 8/3/16.
 */
public class TicketAdministrationController extends CSControllerBase {
  private static final String SUPPORT_TEAM_NAME_DEFAULT = "support-team";
  private static final String SUPPORT_GROUP_NAME_CONFIGURATION = "exo.addon.cs.support.group.name";
  private static Log  LOG = ExoLogger.getLogger(TicketAdministrationController.class);

  @Inject
  TicketService       ticketService;

  @Inject
  OrganizationService organizationService;

  @Inject
  @Path("index.gtmpl")
  Template            indexTmpl;

  @View
  public Response.Content index() {
    return indexTmpl.ok();
  }

  @Ajax
  @Resource
  @MimeType.JSON
  @Jackson
  public List<TicketDTO> getAllTickets() {
    try {
      List<TicketDTO> tickets = ticketService.getTickets();
      return tickets;
    } catch (Throwable e) {
      LOG.error("Exception when retrieving tickets" + e);
      return null;
    }
  }
  @Ajax
  @Resource
  @MimeType.JSON
  @Jackson
  public List<TicketDTO> getTicketsByAssignee(String assignee) {
    try {
      List<TicketDTO> tickets = ticketService.getAssignedTickets(assignee);
      return tickets;
    } catch (Throwable e) {
      LOG.error("Exception when retrieving tickets" + e);
      return null;
    }
  }

  @Ajax
  @Resource
  @MimeType.JSON
  @Jackson
  public Response getBundle(String locale) {
      return super.getBundle(new Locale(locale));
  }

  @Ajax
  @Resource(method = HttpMethod.POST)
  @MimeType.JSON
  @Jackson
  public TicketDTO saveTicket(@Jackson TicketDTO ticketDTO) throws Exception {
    try {
      String groupID = ticketDTO.getSpaceGroupId();
      SpaceEntity spaceEntity = ticketService.getSpaceByGroupId(groupID);
      return ticketService.updateTicketDTO(ticketDTO, spaceEntity);
    } catch (Exception e) {
      LOG.error("Error when updating ticket", e);
      throw e;
    }
  }

  /**
   * This service will return the names of all support team members
   * @param ticketDTO
   * @return
   */
  @Ajax
  @Resource(method = HttpMethod.POST)
  @MimeType.JSON
  @Jackson
  public List<User> getSupportUsernames(@Jackson TicketDTO ticketDTO) {
    try {
      String groupId = PropertyManager.getProperty(SUPPORT_GROUP_NAME_CONFIGURATION);
      if(groupId == null || groupId.isEmpty()){
        groupId = SUPPORT_TEAM_NAME_DEFAULT;
      }
      ListAccess<User> EngSupportList = organizationService.getUserHandler().findUsersByGroupId(groupId);
      User[] users = EngSupportList.load(0, EngSupportList.getSize() < 30 ? EngSupportList.getSize() : 30);
      List<User> engSupport = new ArrayList<>();
      for (User user : users) {
        engSupport.add(user);
      }
      return engSupport;
    } catch (Throwable e) {
      LOG.error("Can't retrieve the list of support engineers " + e);
      return null;
    }
  }

  /**
   * this method will return the list of severities
   * @return List of ticket severity
   */
  @Ajax
  @Resource(method = HttpMethod.POST)
  @MimeType.JSON
  @Jackson
  public List<String> getSeverities(@Jackson TicketDTO ticketDTO) {
      List<String> severities = new ArrayList<>();
      if(ticketDTO.getType().equals(IssueType.INCIDENT)){
        severities.add(IssueSeverity.SEVERITY_1.name());
        severities.add(IssueSeverity.SEVERITY_2.name());
        severities.add(IssueSeverity.SEVERITY_3.name());
      } else if(ticketDTO.getType().equals(IssueType.INFORMATION)) {
        severities.add(IssueSeverity.SEVERITY_4.name());
      }
      return severities;
  }

  /**
   * this method returns the static list of ticket types
   * @param ticketDTO
   * @return
   */
  @Ajax
  @Resource(method = HttpMethod.POST)
  @MimeType.JSON
  @Jackson
  public List<String> getTicketTypes(@Jackson TicketDTO ticketDTO) {
      List<String> ticketTypes = new ArrayList<>();
      ticketTypes.add(IssueType.INCIDENT.name());
      ticketTypes.add(IssueType.INFORMATION.name());
      return ticketTypes;
  }

  @Override
  public Log getLogger() {
    return LOG;
  }
}
