package org.exoplatform.addons.mattermost.services;

import org.apache.commons.lang3.StringUtils;
import org.exoplatform.addons.mattermost.model.Team;
import org.exoplatform.commons.utils.PropertyManager;
import org.exoplatform.container.ExoContainerContext;
import org.exoplatform.container.component.RequestLifeCycle;
import org.exoplatform.services.log.ExoLogger;
import org.exoplatform.services.log.Log;
import org.exoplatform.social.core.identity.model.Identity;
import org.exoplatform.social.core.manager.IdentityManager;
import org.exoplatform.social.core.space.SpaceListenerPlugin;
import org.exoplatform.social.core.space.model.Space;
import org.exoplatform.social.core.space.spi.SpaceLifeCycleEvent;
import org.exoplatform.social.metadata.MetadataService;
import org.exoplatform.social.metadata.model.*;

import java.util.*;

public class MattermostSpaceListener extends SpaceListenerPlugin {

  private static final Log   LOG                                   =
                                 ExoLogger.getLogger(MattermostSpaceListener.class.toString());

  public static final String SPACE                                 = "Space";

  private MetadataService    metadataService;

  private IdentityManager    identityManager;

  public static final String MATTERMOST_TEAM_ID                    = "MattermostTeamID";

  public static final String MATTERMOST_TEAM_NAME                  = "MattermostTeamName";

  public static final String MATTERMOST_METADATA_TYPE              = "matterMostSpaceIntegration";

  public static final String MATTERMOST_METADATA_NAME              = "MattermostTeamMetadata";

  public static final String EXO_ADDON_MATTERMOST_TEAM_NAME_PREFIX = "exo.addon.mattermost.team.name.prefix";

  private final String       teamNamePrefix;

  public MattermostSpaceListener(MetadataService metadataService, IdentityManager identityManager) {
    this.metadataService = metadataService;
    this.identityManager = identityManager;
    this.teamNamePrefix = PropertyManager.getProperty(EXO_ADDON_MATTERMOST_TEAM_NAME_PREFIX);
  }

  @Override
  public void spaceCreated(SpaceLifeCycleEvent event) {
    Space space = event.getSpace();
    String type = Space.OPEN.equalsIgnoreCase(space.getRegistration())
        && Space.PRIVATE.equalsIgnoreCase(space.getVisibility()) ? "O" : "I";
    RequestLifeCycle.begin(ExoContainerContext.getCurrentContainer());
    try {
      String cleanName = space.getPrettyName().toLowerCase().replaceAll("[^a-z0-9]+", "");
      String teamDisplayName = space.getDisplayName();
      if (StringUtils.isNotBlank(this.teamNamePrefix)) {
        cleanName = cleanName + this.teamNamePrefix.toLowerCase();
        teamDisplayName = teamDisplayName + " - " + teamNamePrefix;
      }
      Team mattermostTeam = MattermostUtils.createTeam(cleanName, teamDisplayName, type);
      if (mattermostTeam != null) {
        MetadataKey metadataKey = new MetadataKey(MATTERMOST_METADATA_TYPE, MATTERMOST_METADATA_NAME, 0);
        MetadataObject metadataObject = new MetadataObject(SPACE, space.getId(), null, 0);

        Map<String, String> properties = new HashMap<>();
        properties.put(MATTERMOST_TEAM_ID, mattermostTeam.getId());
        properties.put(MATTERMOST_TEAM_NAME, mattermostTeam.getName());
        metadataService.createMetadataItem(metadataObject, metadataKey, properties);

        LOG.info("Mattermost integration: successfully created a team for the space {}", space.getDisplayName());

        LinkedHashSet<String> allMembers = new LinkedHashSet<>();
        allMembers.addAll(Arrays.asList(space.getManagers()));
        allMembers.addAll(Arrays.asList(space.getMembers()));

        sendInvitationToMembers(new ArrayList<>(allMembers), mattermostTeam.getId());
      }
    } catch (Exception e) {
      LOG.error("Mattermost integration: Could not create a team for space {}", space.getDisplayName(), e);
    } finally {
      RequestLifeCycle.end();
    }
  }

  @Override
  public void spaceRemoved(SpaceLifeCycleEvent event) {
    Space space = event.getSpace();
    LOG.info("Mattermost integration: Deleting team bound to the space '{}'", space.getDisplayName());
    RequestLifeCycle.begin(ExoContainerContext.getCurrentContainer());
    try {
      MetadataKey metadataKey = new MetadataKey(MATTERMOST_METADATA_TYPE, MATTERMOST_METADATA_NAME, 0);
      MetadataObject metadataObject = new MetadataObject(SPACE, space.getId(), null, 0);
      List<MetadataItem> metadataItems = metadataService.getMetadataItemsByMetadataTypeAndObject(metadataKey.getType(),
                                                                                                 metadataObject);

      for (MetadataItem item : metadataItems) {
        String teamId = item.getProperties().get(MATTERMOST_TEAM_ID);
        if (StringUtils.isNotBlank(teamId)) {
          try {
            Team team = MattermostUtils.getTeamById(teamId);
            MattermostUtils.deleteTeam(team);
          } catch (Exception e) {
            LOG.error("Mattermost integration: Could not delete the team '{}' bound with space '{}'",
                      teamId,
                      space.getDisplayName(),
                      e);
          }
          LOG.info("Mattermost integration: successfully deleted the team with id {} bound to the space {}",
                   teamId,
                   space.getDisplayName());
        }
      }
    } finally {
      RequestLifeCycle.end();
    }
  }

  @Override
  public void spaceRenamed(SpaceLifeCycleEvent event) {
    Space space = event.getSpace();
    LOG.info("Mattermost integration: Updating team bound to the space '{}'", space.getDisplayName());
    RequestLifeCycle.begin(ExoContainerContext.getCurrentContainer());
    try {
      MetadataKey metadataKey = new MetadataKey(MATTERMOST_METADATA_TYPE, MATTERMOST_METADATA_NAME, 0);
      MetadataObject metadataObject = new MetadataObject(SPACE, space.getId(), null, 0);
      List<MetadataItem> metadataItems = metadataService.getMetadataItemsByMetadataTypeAndObject(metadataKey.getType(),
                                                                                                 metadataObject);
      if (metadataItems.size() != 1) {
        String message = metadataItems.isEmpty() ? "No team" : "too many teams (" + metadataItems.size() + ")";
        LOG.warn(message + " are bound to the space");
        return;
      }

      String teamId = metadataItems.get(0).getProperties().get(MATTERMOST_TEAM_ID);
      if (StringUtils.isNotBlank(teamId)) {
        try {
          Team team = MattermostUtils.getTeamById(teamId);
          String updatedDisplayName = space.getDisplayName();
          if (team != null) {
            if (StringUtils.isNotBlank(this.teamNamePrefix)) {
              updatedDisplayName = updatedDisplayName + " - " + this.teamNamePrefix;
            }
            team.setDisplayName(updatedDisplayName);
            MattermostUtils.updateTeam(team);
          } else {
            LOG.warn("Could not find a team with Id {} in Mattermost", teamId);
          }
        } catch (Exception e) {
          LOG.error("Mattermost integration: Could not update the display name of team '{}' for space '{}'",
                    teamId,
                    space.getDisplayName(),
                    e);
        }
        LOG.info("Mattermost integration: successfully updated the display name of the team with id {} bound to the space {}",
                 teamId,
                 space.getDisplayName());
      }
    } finally {
      RequestLifeCycle.end();
    }
  }

  @Override
  public void spaceAccessEdited(SpaceLifeCycleEvent event) {
    updateTeamPrivacy(event);
  }

  @Override
  public void spaceRegistrationEdited(SpaceLifeCycleEvent event) {
    updateTeamPrivacy(event);
  }

  private void updateTeamPrivacy(SpaceLifeCycleEvent event) {
    Space space = event.getSpace();
    LOG.info("Mattermost integration: Updating team bound to the space '{}'", space.getDisplayName());
    RequestLifeCycle.begin(ExoContainerContext.getCurrentContainer());
    String teamId = "";
    try {
      MetadataKey metadataKey = new MetadataKey(MATTERMOST_METADATA_TYPE, MATTERMOST_METADATA_NAME, 0);
      MetadataObject metadataObject = new MetadataObject(SPACE, space.getId(), null, 0);
      List<MetadataItem> metadataItems = metadataService.getMetadataItemsByMetadataTypeAndObject(metadataKey.getType(),
                                                                                                 metadataObject);
      if (metadataItems.size() != 1) {
        String message = metadataItems.isEmpty() ? "No team" : "too many teams (" + metadataItems.size() + ")";
        LOG.warn(message + " are bound to the space");
        return;
      }

      teamId = metadataItems.get(0).getProperties().get(MATTERMOST_TEAM_ID);
      if (StringUtils.isNotBlank(teamId)) {
        Team team = MattermostUtils.getTeamById(teamId);
        if (team != null) {
          String type = Space.OPEN.equals(space.getRegistration()) && Space.PRIVATE.equals(space.getVisibility()) ? "O" : "I";
          MattermostUtils.updateTeamPrivacy(team, type);
        }
      }
      LOG.info("Mattermost integration: successfully updated the team with id {} bound to the space {}",
              teamId,
              space.getDisplayName());
    } catch (Exception e) {
      LOG.error("Mattermost integration: Could not update the team '{}' for space '{}'", teamId, space.getDisplayName(), e);
    } finally {
      RequestLifeCycle.end();
    }
  }

  @Override
  public void joined(SpaceLifeCycleEvent event) {
    // Get team
    Space space = event.getSpace();
    String userId = event.getTarget();
    LOG.info("Mattermost integration: Inviting member to join the team bound to the space '{}'", space.getDisplayName());
    RequestLifeCycle.begin(ExoContainerContext.getCurrentContainer());
    try {
      MetadataKey metadataKey = new MetadataKey(MATTERMOST_METADATA_TYPE, MATTERMOST_METADATA_NAME, 0);
      MetadataObject metadataObject = new MetadataObject(SPACE, space.getId(), null, 0);
      List<MetadataItem> metadataItems = metadataService.getMetadataItemsByMetadataTypeAndObject(metadataKey.getType(),
              metadataObject);
      if (metadataItems.size() != 1) {
        String message = metadataItems.isEmpty() ? "No team" : "too many teams (" + metadataItems.size() + ")";
        LOG.warn(message + " are bound to the space");
        return;
      }

      String teamId = metadataItems.get(0).getProperties().get(MATTERMOST_TEAM_ID);
      if (StringUtils.isNotBlank(teamId)) {
        Team team = MattermostUtils.getTeamById(teamId);
        if (team != null) {
          sendInvitationToMembers(Collections.singletonList(userId), teamId);
        } else {
          LOG.warn("Could not find a team with ID {}", teamId);
        }
      }
      LOG.info("Mattermost integration: successfully Invited member {} to join the team bound to the space {}",
              userId,
              space.getDisplayName());
    } catch (Exception e) {
      LOG.error("Mattermost integration: Could not Invite member {} to join the team bound to the space {}", userId, space.getDisplayName(), e);
    } finally {
      RequestLifeCycle.end();
    }
  }

  private void sendInvitationToMembers(List<String> userIds, String teamId) {
    List<String> emailsList = new ArrayList<>();
    for(String userId : userIds) {
      Identity invitedUser = identityManager.getOrCreateUserIdentity(userId);
      if(invitedUser.getProfile() != null && StringUtils.isNotBlank(invitedUser.getProfile().getEmail())) {
        emailsList.add(invitedUser.getProfile().getEmail());
      }
    }
    try {
      MattermostUtils.inviteMemberToTeam(emailsList, teamId);
    } catch (Exception e) {
      LOG.error("Mattermost integration: Could not Invite member {} to join the team bound to the team {}", String.join(",", userIds), teamId, e);
    }
  }
}
