package org.exoplatform.addons.fido.services.impl;

import org.apache.commons.io.IOUtils;
import org.exoplatform.addons.fido.services.api.FidoConnector;
import org.exoplatform.commons.utils.CommonsUtils;
import org.exoplatform.container.xml.InitParams;
import org.exoplatform.services.log.ExoLogger;
import org.exoplatform.services.log.Log;
import org.exoplatform.services.organization.OrganizationService;
import org.exoplatform.services.organization.User;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.BufferedInputStream;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.util.Base64;

public class GluuFido2Connector extends FidoConnector {
  
  private static final String SERVER_URL = "serverUrl";
  
  private static final Log    LOG = ExoLogger.getLogger(GluuFido2Connector.class);
  
  private String serverUrl;
  
  private static final String GLUU_SERVICE = "gluu-service";
  
  public GluuFido2Connector(InitParams initParams) {
    if (initParams.getValueParam(SERVER_URL)!=null) {
      serverUrl = initParams.getValueParam(SERVER_URL).getValue();
    }
  }
  
  @Override
  public JSONObject startRegistration(String userId, String rpHostName) {
    LOG.info("Start FIDO Registration for user {}",userId);
    
    OrganizationService organizationService = CommonsUtils.getService(OrganizationService.class);
    User user;
    long startTime = System.currentTimeMillis();
    try {
      user = organizationService.getUserHandler().findUserByName(userId);
    } catch (Exception e) {
      e.printStackTrace();
      LOG.error("Unable to find user {}",userId,e);
      return null;
    }
    if (user==null) {
      LOG.warn("User {} not exists",userId);
      return null;
    }
    //TODO : get url by fetching
    // /.well-known/fido2-configuration
    String serviceUrl = serverUrl+"/fido2/restv1/fido2/attestation/options";
  
    JSONObject data = new JSONObject();
    try {
      data.put("username",userId);
      data.put("displayName",user.getDisplayName());
      data.put("attestation","direct");
      data.put("documentDomain",rpHostName);
      
    } catch (JSONException e) {
      e.printStackTrace();
      LOG.error("Error when building FIDO Step 1 data",e);
      return null;
  
    }
  
    if (LOG.isDebugEnabled()) {
      LOG.debug("Fido Registration Step 1 request : " + data.toString());
    }
    URL url;
    try {
      url = new URL(serviceUrl);
      HttpURLConnection connection= (HttpURLConnection) url.openConnection();
      connection.setRequestMethod("POST");
      connection.setRequestProperty("Content-Type", "application/json; charset=UTF-8");
      connection.setRequestProperty("Content-Length", String.valueOf(data.toString().getBytes(StandardCharsets.UTF_8)));
      connection.setDoOutput(true);
      connection.setDoInput(true);
      OutputStream outputStream = connection.getOutputStream();
      outputStream.write(data.toString().getBytes(StandardCharsets.UTF_8));
      connection.connect();
  
      int responseCode = connection.getResponseCode();
      if (responseCode==HttpURLConnection.HTTP_OK) {
  
        // read the response
        InputStream in = new BufferedInputStream(connection.getInputStream());
        String response = IOUtils.toString(in, "UTF-8");
  
        JSONObject jsonResponse = new JSONObject(response);
        //in excluded credential, t id is base64url encode
        //but to be able to transform it ot byte array in js, we need it in base64
        JSONArray excludedCredentials = jsonResponse.getJSONArray("excludeCredentials");
        for (int i = 0; i < excludedCredentials.length(); i++) {
          JSONObject excludedCredential = (JSONObject) excludedCredentials.get(i);
          excludedCredential.put("id", transformB64UrlToB64(excludedCredential.getString("id")));
        }
        
        //challenge and user.id are also base64url encode, and we need it as b64encoded
        jsonResponse.put("challenge", transformB64UrlToB64(jsonResponse.getString("challenge")));
  
        JSONObject userJson = jsonResponse.getJSONObject("user");
        userJson.put("id", transformB64UrlToB64(userJson.getString("id")));
  
        if (LOG.isDebugEnabled()) {
          LOG.debug("Fido Registration Step 1 response : " + jsonResponse);
        }
        LOG.info("remote_service={} operation={} parameters=\"user:{},request:{},response:{}\" status=ok " + "duration_ms={}",
                 GLUU_SERVICE,
                 "fido-registration-step-1",
                 userId,
                 data.toString(),
                 response,
                 System.currentTimeMillis() - startTime);
        return jsonResponse;
      } else {
        LOG.error("remote_service={} operation={} parameters=\"user:{}\" status=ko duration_ms={} error_msg=\"Error sending start "
                      + "registration request status : {}\"",
                  GLUU_SERVICE,
                  "fido-registration-step-1",
                  userId,
                  System.currentTimeMillis() - startTime,
                  responseCode);
        return null;
      }
    } catch (Exception e) {
      e.printStackTrace();
      LOG.error("remote_service={} operation={} parameters=\"user:{}\" status=ko " + "duration_ms={}",
                GLUU_SERVICE,
                "fido-registration-step-1",
                userId,
                System.currentTimeMillis() - startTime,e);
      return null;
    }
  
  }
  
  
  @Override
  public JSONObject finishRegistration(String userId, JSONObject data) {
    LOG.info("Finish FIDO Registration for user {}", userId);
    long startTime = System.currentTimeMillis();
  
    String serviceUrl = serverUrl+"/fido2/restv1/fido2/attestation/result";
    JSONObject dataToSend = new JSONObject();
    try {
      dataToSend.put("type",data.getString("type"));
      dataToSend.put("id",data.getString("id"));
  
  
      JSONObject response = new JSONObject();
      //gluu need clientDataJson base64Encoded
      String encodedClientData = Base64.getUrlEncoder().encodeToString(data.getJSONObject("response").getString("clientDataJSON").getBytes());
      response.put("clientDataJSON",encodedClientData);
      response.put("attestationObject",transformB64ToB64Url(data.getJSONObject("response").getString("attestationObject")));
  
      dataToSend.put("response",response);
    } catch (JSONException e) {
      e.printStackTrace();
      LOG.error("Error when building FIDO Step 2 data",e);
      return null;
  
    }
  
    if (LOG.isDebugEnabled()) {
      LOG.debug("Fido Registration Step 2 request : " + dataToSend.toString());
    }
    URL url;
    try {
      url = new URL(serviceUrl);
      HttpURLConnection connection= (HttpURLConnection) url.openConnection();
      connection.setRequestMethod("POST");
      connection.setRequestProperty("Content-Type", "application/json; charset=UTF-8");
      connection.setRequestProperty("Content-Length", String.valueOf(dataToSend.toString().getBytes(StandardCharsets.UTF_8)));
      connection.setDoOutput(true);
      connection.setDoInput(true);
    
      OutputStream outputStream = connection.getOutputStream();
      outputStream.write(dataToSend.toString().getBytes(StandardCharsets.UTF_8));
    
      connection.connect();
      int responseCode=connection.getResponseCode();
      if (responseCode==HttpURLConnection.HTTP_OK) {
  
        // read the response
        InputStream in = new BufferedInputStream(connection.getInputStream());
        String response = IOUtils.toString(in, "UTF-8");
        JSONObject jsonResponse = new JSONObject(response);
        LOG.info("FIDORegistrationStep2 Response : {}", jsonResponse);
        LOG.info("remote_service={} operation={} parameters=\"user:{},request:{},response:{}\" status=ok " + "duration_ms={}",
                 GLUU_SERVICE,
                 "fido-registration-step-2",
                 userId,
                 dataToSend.toString(),
                 response,
                 System.currentTimeMillis() - startTime);
        return jsonResponse;
      } else {
        LOG.error("remote_service={} operation={} parameters=\"user:{}\" status=ko duration_ms={} error_msg=\"Error sending "
                      + "finish registration request status : {}\"",
                  GLUU_SERVICE,
                  "fido-registration-step-2",
                  userId,
                  System.currentTimeMillis() - startTime,
                  responseCode);
        return null;
      }
    } catch (Exception e) {
      e.printStackTrace();
      LOG.error("remote_service={} operation={} parameters=\"user:{}\" status=ko " + "duration_ms={}",
                GLUU_SERVICE,
                "fido-registration-step-2",
                userId,
                System.currentTimeMillis() - startTime,e);
      return null;
    }
  }
  
  @Override
  public JSONObject startAuthentication(String userId, String rpHostName) {
    LOG.info("Start FIDO Authentication for user {}",userId);
    
    OrganizationService organizationService = CommonsUtils.getService(OrganizationService.class);
    User user;
    long startTime = System.currentTimeMillis();
    try {
      user = organizationService.getUserHandler().findUserByName(userId);
    } catch (Exception e) {
      e.printStackTrace();
      LOG.error("Unable to find user {}",userId,e);
      return null;
    }
    if (user==null) {
      LOG.warn("User {} not exists",userId);
    }
    //TODO : get url by fetching
    // /.well-known/fido2-configuration
    String serviceUrl = serverUrl+"/fido2/restv1/fido2/assertion/options";
    
    JSONObject data = new JSONObject();
    try {
      data.put("username",userId);
      data.put("documentDomain",rpHostName);
    } catch (JSONException e) {
      e.printStackTrace();
      LOG.error("Error when building FIDO Authentication Step 1 data",e);
      return null;
      
    }
    
    if (LOG.isDebugEnabled()) {
      LOG.debug("Fido Authentication Step 1 request : " + data.toString());
    }
    URL url;
    try {
      url = new URL(serviceUrl);
      HttpURLConnection connection= (HttpURLConnection) url.openConnection();
      connection.setRequestMethod("POST");
      connection.setRequestProperty("Content-Type", "application/json; charset=UTF-8");
      connection.setRequestProperty("Content-Length", String.valueOf(data.toString().getBytes(StandardCharsets.UTF_8)));
      connection.setDoOutput(true);
      connection.setDoInput(true);
      
      OutputStream outputStream = connection.getOutputStream();
      outputStream.write(data.toString().getBytes(StandardCharsets.UTF_8));
      
      connection.connect();
      int responseCode = connection.getResponseCode();
      if (responseCode==HttpURLConnection.HTTP_OK) {
        // read the response
        InputStream in = new BufferedInputStream(connection.getInputStream());
        String response = IOUtils.toString(in, "UTF-8");
        JSONObject jsonResponse = new JSONObject(response);
  
        //in allowed credential, t id is base64url encode
        //but to be able to transform it ot byte array in js, we need it in base64
        JSONArray allowedCredentials = jsonResponse.getJSONArray("allowCredentials");
        for (int i = 0; i < allowedCredentials.length(); i++) {
          JSONObject allowedCredential = (JSONObject) allowedCredentials.get(i);
          allowedCredential.put("id", transformB64UrlToB64(allowedCredential.getString("id")));
        }
  
        //challenge are base64url encode, and we need it as b64encoded
        jsonResponse.put("challenge", transformB64UrlToB64(jsonResponse.getString("challenge")));
        
        if (LOG.isDebugEnabled()) {
          LOG.debug("Fido Authentication Step 1 response : " + jsonResponse);
        }
        LOG.info("remote_service={} operation={} parameters=\"user:{},request:{},response:{}\" status=ok " + "duration_ms={}",
                 GLUU_SERVICE,
                 "fido-authentication-step-1",
                 userId,
                 data.toString(),
                 response,
                 System.currentTimeMillis() - startTime);
        return jsonResponse;
      } else {
        LOG.error("remote_service={} operation={} parameters=\"user:{}\" status=ko duration_ms={} error_msg=\"Error sending "
                      + "start authentication request status : {}\"",
                  GLUU_SERVICE,
                  "fido-authentication-step-1",
                  userId,
                  System.currentTimeMillis() - startTime,
                  responseCode);
        return null;
      }
    } catch (Exception e) {
      e.printStackTrace();
      LOG.error("remote_service={} operation={} parameters=\"user:{}\" status=ko " + "duration_ms={}",
                GLUU_SERVICE,
                "fido-authentication-step-1",
                userId,
                System.currentTimeMillis() - startTime,e);
      return null;
    }
    
  }
  
  @Override
  public JSONObject finishAuthentication(String userId, JSONObject data) {
    LOG.info("Finish FIDO Authentication for user {}", userId);
    long startTime = System.currentTimeMillis();
    
    String serviceUrl = serverUrl+"/fido2/restv1/fido2/assertion/result";
    JSONObject dataToSend = new JSONObject();
    try {
      dataToSend.put("type",data.getString("type"));
      dataToSend.put("id",data.getString("id"));
      dataToSend.put("rawId",data.getString("rawId"));
      
      JSONObject response = new JSONObject();

      //Gluu need clientDataJson base64UrlEncoded
      String encodedClientData = Base64.getUrlEncoder().encodeToString(data.getJSONObject("response").getString("clientDataJSON").getBytes());
      response.put("clientDataJSON",encodedClientData);
  
      response.put("authenticatorData",transformB64ToB64Url(data.getJSONObject("response").getString("authenticatorData")));
      
      response.put("signature",transformB64ToB64Url(data.getJSONObject("response").getString("signature")));
      
      dataToSend.put("response",response);
    } catch (JSONException e) {
      e.printStackTrace();
      LOG.error("Error when building FIDO Authentication Step 2 data",e);
      return null;
      
    }
    
    if (LOG.isDebugEnabled()) {
      LOG.debug("Fido Authentication Step 2 request : " + dataToSend.toString());
    }
    URL url;
    try {
      url = new URL(serviceUrl);
      HttpURLConnection connection= (HttpURLConnection) url.openConnection();
      connection.setRequestMethod("POST");
      connection.setRequestProperty("Content-Type", "application/json; charset=UTF-8");
      connection.setRequestProperty("Content-Length", String.valueOf(dataToSend.toString().getBytes(StandardCharsets.UTF_8)));
      connection.setDoOutput(true);
      connection.setDoInput(true);
      
      OutputStream outputStream = connection.getOutputStream();
      outputStream.write(dataToSend.toString().getBytes(StandardCharsets.UTF_8));
      
      connection.connect();
      int responseCode=connection.getResponseCode();
      if (responseCode==HttpURLConnection.HTTP_OK) {
        // read the response
        InputStream in = new BufferedInputStream(connection.getInputStream());
        String response = IOUtils.toString(in, "UTF-8");
        JSONObject jsonResponse = new JSONObject(response);
        LOG.info("FIDOAuthenticationStep2 Response : {}",jsonResponse);
        LOG.info("remote_service={} operation={} parameters=\"user:{},request:{},response:{}\" status=ok " + "duration_ms={}",
                 GLUU_SERVICE,
                 "fido-authentication-step-2",
                 userId,
                 dataToSend.toString(),
                 response,
                 System.currentTimeMillis() - startTime);
        return jsonResponse;
      } else {
        LOG.error("remote_service={} operation={} parameters=\"user:{}\" status=ko duration_ms={} error_msg=\"Error sending "
                      + "finish authentication request status : {}\"",
                  GLUU_SERVICE,
                  "fido-authentication-step-2",
                  userId,
                  System.currentTimeMillis() - startTime,
                  responseCode);
        return null;
      }
    
      
    } catch (Exception e) {
      e.printStackTrace();
      LOG.error("remote_service={} operation={} parameters=\"user:{}\" status=ko " + "duration_ms={}",
                GLUU_SERVICE,
                "fido-authentication-step-2",
                userId,
                System.currentTimeMillis() - startTime,e);
      return null;
    }
  
  }
  
  private String transformB64UrlToB64(String input) {
    byte[] bytesString = Base64.getUrlDecoder().decode(input);
    return Base64.getEncoder().encodeToString(bytesString);
  }
  
  private String transformB64ToB64Url(String input) {
    byte[] bytesString = Base64.getDecoder().decode(input.getBytes());
    return Base64.getUrlEncoder().encodeToString(bytesString);
  }
  
}
