package org.exoplatform.addons.oidc.services;

import io.jsonwebtoken.Claims;
import io.jsonwebtoken.InvalidClaimException;
import io.jsonwebtoken.Jws;
import io.jsonwebtoken.Jwt;
import io.jsonwebtoken.JwtException;
import io.jsonwebtoken.MissingClaimException;
import org.exoplatform.container.xml.InitParams;
import org.exoplatform.container.xml.ValueParam;

import io.jsonwebtoken.Jwts;
import org.exoplatform.services.log.ExoLogger;
import org.exoplatform.services.log.Log;

import java.time.Instant;
import java.util.Date;
import java.util.GregorianCalendar;

public class OIDCService {
  
  private String clientId;
  private String oidcServer;
  private String redirectUri;

  private static final Log LOG = ExoLogger.getLogger(OIDCService.class);

  private RemoteJwkSigningKeyResolver remoteJwkSigningKeyResolver;


  public OIDCService(InitParams initParams) {
  
    ValueParam clientId = initParams.getValueParam("clientId");
    if (clientId!=null) {
      this.clientId = clientId.getValue();
    }
    ValueParam oidcServer = initParams.getValueParam("oidcServer");
    if (oidcServer!=null) {
      this.oidcServer = oidcServer.getValue();
    }
    ValueParam redirectUri = initParams.getValueParam("redirectUri");
    if (redirectUri!=null) {
      this.redirectUri = redirectUri.getValue();
    }

    this.remoteJwkSigningKeyResolver = new RemoteJwkSigningKeyResolver(this.oidcServer);

  }

  
  public String getClientId() {
    return clientId;
  }
  
  public String getOidcServer() {
    return oidcServer;
  }
  
  public String getRedirectUri() {
    return redirectUri;
  }

  public Instant validateToken(String user, String token, String nonce, String issuer, String clientId) {
    try {
      //as we are in implicit flow, all this part is already verified in front part

      Jwt jwt = Jwts.parser()
                    .requireIssuer(issuer)
                    .requireAudience(clientId)
                    .require("nonce",nonce)
                    .setSigningKeyResolver(remoteJwkSigningKeyResolver)
                    .parse(token);
      Claims claims = (Claims) jwt.getBody();

      if (claims.getExpiration().toInstant().isBefore(Instant.now())) {
        LOG.error("Token is expired");
        return null;
      }
      return claims.getExpiration().toInstant();
    } catch (MissingClaimException e) {
      LOG.error("Missing Claim {}",e.getClaimName(),e);
    } catch (InvalidClaimException ice) {
      LOG.error("Invalide Claim {}", ice.getClaimName(),ice);
    } catch (JwtException e) {
      //don't trust the JWT!
      LOG.error("Jwt Signature Verification Failed",e);
    }

    return null;
  }

}
