package org.exoplatform.addons.oidc.services;

import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;
import org.exoplatform.common.http.HTTPStatus;
import org.exoplatform.commons.utils.CommonsUtils;
import org.exoplatform.container.PortalContainer;
import org.exoplatform.portal.config.UserPortalConfigService;
import org.exoplatform.services.log.ExoLogger;
import org.exoplatform.services.log.Log;
import org.exoplatform.services.rest.resource.ResourceContainer;

import io.swagger.annotations.Api;
import org.exoplatform.services.security.ConversationState;
import org.json.JSONException;
import org.json.JSONObject;

import javax.annotation.security.RolesAllowed;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpSession;
import javax.ws.rs.FormParam;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.Path;
import javax.ws.rs.Produces;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import java.time.Instant;

@Path("/oidc")
@Api(value = "/oidc", description = "Manages OIDC features")
public class OIDCRestService implements ResourceContainer {
  private static final Log LOG = ExoLogger.getLogger(OIDCRestService.class);

  @Path("/settings")
  @GET
  @Produces(MediaType.APPLICATION_JSON)
  @RolesAllowed("users")
  @ApiOperation(value = "Get OIDC Settings", httpMethod = "GET", response = Response.class, produces = MediaType.APPLICATION_JSON)
  @ApiResponses(value = {@ApiResponse(code = HTTPStatus.OK, message = "Request fulfilled"),
          @ApiResponse(code = HTTPStatus.UNAUTHORIZED, message = "Unauthorized operation"),
          @ApiResponse(code = HTTPStatus.INTERNAL_ERROR, message = "Internal server error"),})
  public Response getOidcSettings(@Context HttpServletRequest request) {
    OIDCService oidcService = CommonsUtils.getService(OIDCService.class);

    JSONObject result = new JSONObject();
    try {
      result.put("clientId", oidcService.getClientId());
      result.put("oidcServer", oidcService.getOidcServer());
      result.put("redirectUri", getUrl(request)+oidcService.getRedirectUri());
      return Response.ok().entity(result.toString()).build();
    } catch (JSONException e) {
      LOG.error("Unable to build response", e);
      return Response.serverError().build();

    }

  }


  @Path("/token")
  @POST
  @Produces(MediaType.APPLICATION_JSON)
  @RolesAllowed("users")
  @ApiOperation(value = "Register OIDC Token", httpMethod = "POST", response = Response.class, produces =
          MediaType.APPLICATION_JSON)
  @ApiResponses(value = {@ApiResponse(code = HTTPStatus.OK, message = "Request fulfilled"),
          @ApiResponse(code = HTTPStatus.UNAUTHORIZED, message = "Unauthorized operation"),
          @ApiResponse(code = 400, message = "Invalid query input"),
          @ApiResponse(code = HTTPStatus.INTERNAL_ERROR, message = "Internal server error"),})
  public Response registerToken(@ApiParam(value = "Token to register", required = true) @FormParam("token") String token,
                                @ApiParam(value = "Nonce to validate", required = true) @FormParam("nonce")  String nonce,
                                @ApiParam(value = "Issuer to validate", required = true) @FormParam("issuer")  String issuer,
                                @ApiParam(value = "ClientId to validate", required = true) @FormParam("clientId")  String clientId,
                                @Context HttpServletRequest request) {

    HttpSession session = request.getSession();

    OIDCService oidcService = CommonsUtils.getService(OIDCService.class);
    Instant expirationInstant = oidcService.validateToken(ConversationState.getCurrent().getIdentity().getUserId(), token,
                                                          nonce, issuer, clientId);
    //todo : move this in oidc service
    if (expirationInstant!=null) {
      session.setAttribute("mfaValidated", true);
      session.setAttribute("mfaExpiration", expirationInstant);
      return Response.ok().build();
    } else {
      return Response.serverError().build();
    }

  }


  private StringBuilder getUrl(HttpServletRequest request) {
    StringBuilder url = new StringBuilder();
    if (request != null) {
      url.append(request.getScheme()).append("://").append(request.getServerName());
      if (request.getServerPort() != 80 && request.getServerPort() != 443) {
        url.append(':').append(request.getServerPort());
      }
    }
    return url;
  }
}
