/*
 * Decompiled with CFR 0.152.
 */
package org.exoplatform.addon.ethereum.wallet.reward.service;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;
import org.apache.commons.codec.binary.StringUtils;
import org.exoplatform.addon.ethereum.wallet.model.Wallet;
import org.exoplatform.addon.ethereum.wallet.reward.api.RewardPlugin;
import org.exoplatform.addon.ethereum.wallet.reward.model.RewardBudgetType;
import org.exoplatform.addon.ethereum.wallet.reward.model.RewardMemberDetail;
import org.exoplatform.addon.ethereum.wallet.reward.model.RewardPeriod;
import org.exoplatform.addon.ethereum.wallet.reward.model.RewardPeriodType;
import org.exoplatform.addon.ethereum.wallet.reward.model.RewardPluginSettings;
import org.exoplatform.addon.ethereum.wallet.reward.model.RewardSettings;
import org.exoplatform.addon.ethereum.wallet.reward.model.RewardTeam;
import org.exoplatform.addon.ethereum.wallet.reward.model.RewardTeamMember;
import org.exoplatform.addon.ethereum.wallet.reward.service.RewardSettingsService;
import org.exoplatform.addon.ethereum.wallet.reward.service.RewardTeamService;
import org.exoplatform.addon.ethereum.wallet.reward.service.utils.RewardUtils;
import org.exoplatform.addon.ethereum.wallet.service.EthereumWalletAccountService;
import org.exoplatform.services.log.ExoLogger;
import org.exoplatform.services.log.Log;
import org.exoplatform.ws.frameworks.json.impl.JsonException;

public class RewardService {
    private static final Log LOG = ExoLogger.getLogger(RewardService.class);
    private RewardSettingsService rewardSettingsService;
    private RewardTeamService rewardTeamService;
    private EthereumWalletAccountService walletAccountService;

    public RewardService(EthereumWalletAccountService walletAccountService, RewardSettingsService rewardSettingsService, RewardTeamService rewardTeamService) {
        this.walletAccountService = walletAccountService;
        this.rewardSettingsService = rewardSettingsService;
        this.rewardTeamService = rewardTeamService;
    }

    public Set<RewardMemberDetail> computeReward(Set<Long> identityIds, long periodDateInSeconds) {
        RewardSettings rewardSettings;
        if (periodDateInSeconds == 0L) {
            throw new IllegalArgumentException("periodDate is mandatory");
        }
        if (identityIds == null || identityIds.isEmpty()) {
            return Collections.emptySet();
        }
        Collection<RewardPlugin> rewardPlugins = this.rewardSettingsService.getRewardPlugins();
        try {
            rewardSettings = this.rewardSettingsService.getSettings();
        }
        catch (JsonException e) {
            throw new IllegalStateException("Can't retrieve reward settings");
        }
        if (rewardSettings == null) {
            throw new IllegalStateException("Error computing rewards using empty settings");
        }
        if (rewardSettings.getPeriodType() == null) {
            throw new IllegalStateException("Error computing rewards using empty period type");
        }
        Set<RewardPluginSettings> pluginSettings = rewardSettings.getPluginSettings();
        if (pluginSettings == null || pluginSettings.isEmpty()) {
            throw new IllegalStateException("Error computing rewards using empty rewards types");
        }
        RewardPeriodType periodType = rewardSettings.getPeriodType();
        RewardPeriod periodOfTime = periodType.getPeriodOfTime(RewardUtils.timeFromSeconds(periodDateInSeconds));
        Set<Long> enabledIdentityIds = this.getEnabledWallets(identityIds);
        Set<Long> walletsWithEnabledTeam = this.getEnabledTeamMembers(enabledIdentityIds);
        HashSet<RewardMemberDetail> rewardMemberDetails = new HashSet<RewardMemberDetail>();
        for (RewardPlugin rewardPlugin : rewardPlugins) {
            RewardPluginSettings rewardPluginSettings;
            if (rewardPlugin == null || !rewardPlugin.isEnabled() || (rewardPluginSettings = this.getPluginSetting(pluginSettings, rewardPlugin.getPluginId())) == null) continue;
            Map<Long, Double> earnedPoints = rewardPlugin.gtEarnedPoints(identityIds, periodOfTime.getStartDateInSeconds(), periodOfTime.getEndDateInSeconds());
            Set<Long> validIdentityIdsToUse = rewardPluginSettings.isUsePools() ? walletsWithEnabledTeam : enabledIdentityIds;
            this.computeReward(rewardPluginSettings, earnedPoints, validIdentityIdsToUse, rewardMemberDetails);
        }
        return rewardMemberDetails;
    }

    private Set<Long> getEnabledTeamMembers(Set<Long> identityIds) {
        HashSet<Long> walletsWithEnabledTeam = new HashSet<Long>(identityIds);
        List<RewardTeam> teams = this.rewardTeamService.getTeams();
        for (RewardTeam rewardTeam : teams) {
            if (!rewardTeam.isDisabled() || rewardTeam.getMembers() == null || rewardTeam.getMembers().isEmpty()) continue;
            rewardTeam.getMembers().forEach(member -> walletsWithEnabledTeam.remove(member.getIdentityId()));
        }
        return walletsWithEnabledTeam;
    }

    private Set<Long> getEnabledWallets(Set<Long> identityIds) {
        Iterator<Long> identityIdsIterator = identityIds.iterator();
        while (identityIdsIterator.hasNext()) {
            Long identityId = identityIdsIterator.next();
            if (identityId == null || identityId == 0L) {
                identityIdsIterator.remove();
            }
            Wallet wallet = null;
            try {
                wallet = this.walletAccountService.getWalletByIdentityId(identityId.longValue());
            }
            catch (Exception e) {
                if (LOG.isDebugEnabled()) {
                    LOG.warn("Error while getting wallet of identity with id {}", new Object[]{identityId, e});
                }
                LOG.warn("Error while getting wallet of identity with id {}. Reason: {}", new Object[]{identityId, e.getMessage()});
            }
            if (wallet == null) {
                identityIdsIterator.remove();
                continue;
            }
            if (wallet.isEnabled()) continue;
            identityIdsIterator.remove();
        }
        return identityIds;
    }

    private RewardPluginSettings getPluginSetting(Set<RewardPluginSettings> pluginSettings, String pluginId) {
        for (RewardPluginSettings rewardPluginSettings : pluginSettings) {
            if (!StringUtils.equals((CharSequence)pluginId, (CharSequence)rewardPluginSettings.getPluginId())) continue;
            return rewardPluginSettings;
        }
        return null;
    }

    private void computeReward(RewardPluginSettings rewardPluginSettings, Map<Long, Double> earnedPoints, Set<Long> validIdentityIdsToUse, Set<RewardMemberDetail> rewardMemberDetails) {
        RewardBudgetType budgetType = rewardPluginSettings.getBudgetType();
        if (budgetType == null) {
            LOG.warn("Budget type of reward plugin {} is empty, thus no computing is possible", new Object[]{rewardPluginSettings.getPluginId()});
            return;
        }
        String pluginId = rewardPluginSettings.getPluginId();
        double configuredPluginAmount = rewardPluginSettings.getAmount();
        if (configuredPluginAmount < 0.0) {
            throw new IllegalStateException("Plugin " + pluginId + " has a configured negative reward amount (" + configuredPluginAmount + ")");
        }
        this.filterElligibleMembers(earnedPoints.entrySet(), validIdentityIdsToUse, rewardPluginSettings, rewardMemberDetails);
        double amountPerPoint = 0.0;
        double totalFixedBudget = 0.0;
        switch (budgetType) {
            case FIXED_PER_POINT: {
                amountPerPoint = configuredPluginAmount;
                this.addRewardsSwitchPointAmount(rewardMemberDetails, earnedPoints.entrySet(), pluginId, amountPerPoint);
                break;
            }
            case FIXED: {
                totalFixedBudget = configuredPluginAmount;
                this.addTeamMembersReward(rewardPluginSettings, earnedPoints, totalFixedBudget, rewardMemberDetails);
                break;
            }
            case FIXED_PER_MEMBER: {
                double budgetPerMember = configuredPluginAmount;
                int totalElligibleMembersCount = earnedPoints.size();
                totalFixedBudget = budgetPerMember * (double)totalElligibleMembersCount;
                this.addTeamMembersReward(rewardPluginSettings, earnedPoints, totalFixedBudget, rewardMemberDetails);
                break;
            }
            default: {
                throw new IllegalStateException("Budget type is not recognized in plugin settings: " + pluginId + ", budget type = " + (Object)((Object)budgetType));
            }
        }
    }

    private void addTeamMembersReward(RewardPluginSettings rewardPluginSettings, Map<Long, Double> earnedPoints, double totalFixedBudget, Set<RewardMemberDetail> rewardMemberDetails) {
        if (totalFixedBudget <= 0.0) {
            return;
        }
        if (rewardPluginSettings.isUsePools()) {
            List<RewardTeam> teams = this.rewardTeamService.getTeams();
            Set<Long> identityIds = this.filterEligibleMembersAndTeams(teams, earnedPoints);
            this.buildNoPoolUsers(earnedPoints, teams, identityIds);
            this.computeTeamsMembersBudget(rewardPluginSettings.getPluginId(), teams, totalFixedBudget, rewardMemberDetails, earnedPoints);
        } else {
            double totalPoints = earnedPoints.entrySet().stream().collect(Collectors.summingDouble(entry -> (Double)entry.getValue()));
            if (totalPoints <= 0.0 || totalFixedBudget <= 0.0) {
                return;
            }
            double amountPerPoint = totalFixedBudget / totalPoints;
            this.addRewardsSwitchPointAmount(rewardMemberDetails, earnedPoints.entrySet(), rewardPluginSettings.getPluginId(), amountPerPoint);
        }
    }

    private void addRewardsSwitchPointAmount(Set<RewardMemberDetail> rewardMemberDetails, Set<Map.Entry<Long, Double>> identitiesPointsEntries, String pluginId, double amountPerPoint) {
        for (Map.Entry<Long, Double> identitiyPointsEntry : identitiesPointsEntries) {
            Long identityId = identitiyPointsEntry.getKey();
            Double points = identitiyPointsEntry.getValue();
            double amount = points * amountPerPoint;
            RewardMemberDetail rewardMemberDetail = new RewardMemberDetail();
            rewardMemberDetail.setIdentityId(identityId);
            rewardMemberDetail.setPluginId(pluginId);
            rewardMemberDetail.setPoints(points);
            rewardMemberDetail.setAmount(amount);
            rewardMemberDetail.setPoolsUsed(false);
            rewardMemberDetails.add(rewardMemberDetail);
        }
    }

    private void filterElligibleMembers(Set<Map.Entry<Long, Double>> identitiesPointsEntries, Set<Long> validIdentityIdsToUse, RewardPluginSettings rewardPluginSettings, Set<RewardMemberDetail> rewardMemberDetails) {
        String pluginId = rewardPluginSettings.getPluginId();
        double threshold = rewardPluginSettings.getThreshold();
        Iterator<Map.Entry<Long, Double>> identitiesPointsIterator = identitiesPointsEntries.iterator();
        while (identitiesPointsIterator.hasNext()) {
            Map.Entry<Long, Double> entry = identitiesPointsIterator.next();
            Long identityId = entry.getKey();
            Double points = entry.getValue();
            points = points == null ? 0.0 : points;
            if (points < 0.0) {
                throw new IllegalStateException("Plugin with id " + pluginId + " has assigned a negative points (" + points + ") to user with id " + identityId);
            }
            if (!(points < threshold) && points != 0.0 && validIdentityIdsToUse.contains(identityId)) continue;
            identitiesPointsIterator.remove();
            if (!(points > 0.0)) continue;
            RewardMemberDetail rewardMemberDetail = new RewardMemberDetail();
            rewardMemberDetail.setIdentityId(identityId);
            rewardMemberDetail.setPluginId(pluginId);
            rewardMemberDetail.setPoints(points);
            rewardMemberDetail.setAmount(0.0);
            rewardMemberDetail.setPoolsUsed(rewardPluginSettings.isUsePools());
            rewardMemberDetails.add(rewardMemberDetail);
        }
    }

    private void computeTeamsMembersBudget(String pluginId, List<RewardTeam> teams, double totalTeamsBudget, Set<RewardMemberDetail> rewardMemberDetails, Map<Long, Double> earnedPoints) {
        double totalFixedTeamsBudget = 0.0;
        double computedRecipientsCount = 0.0;
        ArrayList<RewardTeam> computedBudgetTeams = new ArrayList<RewardTeam>();
        HashMap<Long, Double> totalPointsPerTeam = new HashMap<Long, Double>();
        for (RewardTeam rewardTeam2 : teams) {
            double totalTeamBudget;
            RewardBudgetType teamBudgetType = rewardTeam2.getRewardType();
            double totalTeamPoints = rewardTeam2.getMembers().stream().collect(Collectors.summingDouble(member -> (Double)earnedPoints.get(member.getIdentityId())));
            if (teamBudgetType == RewardBudgetType.COMPUTED) {
                computedRecipientsCount += (double)rewardTeam2.getMembers().size();
                computedBudgetTeams.add(rewardTeam2);
                totalPointsPerTeam.put(rewardTeam2.getId(), totalTeamPoints);
                continue;
            }
            if (teamBudgetType == RewardBudgetType.FIXED_PER_MEMBER) {
                totalTeamBudget = rewardTeam2.getBudget() * (double)rewardTeam2.getMembers().size();
                this.addTeamRewardRepartition(rewardTeam2, totalTeamBudget, totalTeamPoints, pluginId, earnedPoints, rewardMemberDetails);
                totalFixedTeamsBudget += totalTeamBudget;
                continue;
            }
            if (teamBudgetType != RewardBudgetType.FIXED) continue;
            totalTeamBudget = rewardTeam2.getBudget();
            this.addTeamRewardRepartition(rewardTeam2, totalTeamBudget, totalTeamPoints, pluginId, earnedPoints, rewardMemberDetails);
            totalFixedTeamsBudget += rewardTeam2.getBudget().doubleValue();
        }
        if (totalFixedTeamsBudget >= totalTeamsBudget) {
            throw new IllegalStateException("Total fixed teams budget is higher than fixed budget for all users");
        }
        if (computedRecipientsCount > 0.0 && !computedBudgetTeams.isEmpty()) {
            double remaingBudgetForComputedTeams = totalTeamsBudget - totalFixedTeamsBudget;
            double budgetPerTeamMember = remaingBudgetForComputedTeams / computedRecipientsCount;
            computedBudgetTeams.forEach(rewardTeam -> {
                double totalTeamBudget = budgetPerTeamMember * (double)rewardTeam.getMembers().size();
                Double totalTeamPoints = (Double)totalPointsPerTeam.get(rewardTeam.getId());
                this.addTeamRewardRepartition((RewardTeam)rewardTeam, totalTeamBudget, totalTeamPoints, pluginId, earnedPoints, rewardMemberDetails);
            });
        }
    }

    private void buildNoPoolUsers(Map<Long, Double> earnedPoints, List<RewardTeam> teams, Set<Long> identityIds) {
        ArrayList<Long> noPoolsIdentityIds = new ArrayList<Long>(earnedPoints.keySet());
        noPoolsIdentityIds.removeAll(identityIds);
        if (!noPoolsIdentityIds.isEmpty()) {
            RewardTeam noPoolRewardTeam = new RewardTeam();
            noPoolRewardTeam.setDisabled(false);
            List<RewardTeamMember> noPoolRewardTeamList = noPoolsIdentityIds.stream().map(identityId -> {
                RewardTeamMember rewardTeamMember = new RewardTeamMember();
                rewardTeamMember.setIdentityId((Long)identityId);
                return rewardTeamMember;
            }).collect(Collectors.toList());
            noPoolRewardTeam.setMembers(noPoolRewardTeamList);
            noPoolRewardTeam.setId(0L);
            noPoolRewardTeam.setRewardType(RewardBudgetType.COMPUTED);
            teams.add(noPoolRewardTeam);
        }
    }

    private Set<Long> filterEligibleMembersAndTeams(List<RewardTeam> teams, Map<Long, Double> earnedPoints) {
        HashSet<Long> identityIds = new HashSet<Long>();
        Iterator<RewardTeam> teamsIterator = teams.iterator();
        while (teamsIterator.hasNext()) {
            RewardTeam rewardTeam = teamsIterator.next();
            List<RewardTeamMember> members = rewardTeam.getMembers();
            if (members != null && !members.isEmpty()) {
                Iterator<RewardTeamMember> membersIterator = members.iterator();
                while (membersIterator.hasNext()) {
                    RewardTeamMember member = membersIterator.next();
                    Long identityId = member.getIdentityId();
                    if (identityIds.contains(identityId)) {
                        throw new IllegalStateException("Team " + rewardTeam.getName() + " has a duplicated member in another Team");
                    }
                    identityIds.add(identityId);
                    if (earnedPoints.containsKey(identityId)) continue;
                    membersIterator.remove();
                }
            }
            if (members != null && !members.isEmpty()) continue;
            teamsIterator.remove();
        }
        return identityIds;
    }

    private void addTeamRewardRepartition(RewardTeam rewardTeam, double totalTeamBudget, double totalTeamPoints, String pluginId, Map<Long, Double> earnedPoints, Set<RewardMemberDetail> rewardMemberDetails) {
        if (rewardTeam.getMembers().isEmpty() || totalTeamBudget <= 0.0 || totalTeamPoints <= 0.0) {
            return;
        }
        double amountPerPoint = totalTeamBudget / totalTeamPoints;
        rewardTeam.getMembers().forEach(member -> {
            Long identityId = member.getIdentityId();
            Double points = (Double)earnedPoints.get(identityId);
            RewardMemberDetail rewardMemberDetail = new RewardMemberDetail();
            rewardMemberDetail.setIdentityId(identityId);
            rewardMemberDetail.setPluginId(pluginId);
            rewardMemberDetail.setPoints(points);
            rewardMemberDetail.setAmount(points * amountPerPoint);
            rewardMemberDetail.setPoolsUsed(true);
            rewardMemberDetails.add(rewardMemberDetail);
        });
    }
}

